/*==================================================================================================
	CAHALAudioObject.cpp

	$Log: CAHALAudioObject.cpp,v $
	Revision 1.1  2005/02/14 05:38:13  jcm10
	first checked in
	
	Revision 0.0  2/12/05 11:26:51 PM  moorf
	Created
		
	$NoKeywords: $
==================================================================================================*/

//==================================================================================================
//	Includes
//==================================================================================================

//	Self Include
#include "CAHALAudioObject.h"

//	PublicUtility Includes
#include "CAAutoDisposer.h"
#include "CADebugMacros.h"
#include "CAException.h"
#include "CAPropertyAddress.h"

//==================================================================================================
//	CAHALAudioObject
//==================================================================================================

CAHALAudioObject::CAHALAudioObject(AudioObjectID inObjectID)
:
	mObjectID(inObjectID)
{
}

CAHALAudioObject::~CAHALAudioObject()
{
}

AudioObjectID	CAHALAudioObject::GetObjectID() const
{
	return mObjectID;
}

AudioClassID	CAHALAudioObject::GetClassID() const
{
	//	set up the return value
	AudioClassID theAnswer = 0;
	
	//	set up the property address
	CAPropertyAddress theAddress(kAudioObjectPropertyClass);
	
	//	make sure the property exists
	if(HasProperty(theAddress))
	{
		UInt32 theSize = sizeof(AudioClassID);
		GetPropertyData(theAddress, 0, NULL, theSize, &theAnswer);
	}
	
	return theAnswer;
}

AudioObjectID	CAHALAudioObject::GetOwnerObjectID() const
{
	//	set up the return value
	AudioObjectID theAnswer = 0;
	
	//	set up the property address
	CAPropertyAddress theAddress(kAudioObjectPropertyOwner);
	
	//	make sure the property exists
	if(HasProperty(theAddress))
	{
		//	get the property data
		UInt32 theSize = sizeof(AudioObjectID);
		GetPropertyData(theAddress, 0, NULL, theSize, &theAnswer);
	}
	
	return theAnswer;
}

CFStringRef	CAHALAudioObject::CopyOwningPlugInBundleID() const
{
	//	set up the return value
	CFStringRef theAnswer = NULL;
	
	//	set up the property address
	CAPropertyAddress theAddress(kAudioObjectPropertyCreator);
	
	//	make sure the property exists
	if(HasProperty(theAddress))
	{
		//	get the property data
		UInt32 theSize = sizeof(CFStringRef);
		GetPropertyData(theAddress, 0, NULL, theSize, &theAnswer);
	}
	
	return theAnswer;
}

CFStringRef	CAHALAudioObject::CopyName() const
{
	//	set up the return value
	CFStringRef theAnswer = NULL;
	
	//	set up the property address
	CAPropertyAddress theAddress(kAudioObjectPropertyName);
	
	//	make sure the property exists
	if(HasProperty(theAddress))
	{
		//	get the property data
		UInt32 theSize = sizeof(CFStringRef);
		GetPropertyData(theAddress, 0, NULL, theSize, &theAnswer);
	}
	
	return theAnswer;
}

CFStringRef	CAHALAudioObject::CopyManufacturer() const
{
	//	set up the return value
	CFStringRef theAnswer = NULL;
	
	//	set up the property address
	CAPropertyAddress theAddress(kAudioObjectPropertyManufacturer);
	
	//	make sure the property exists
	if(HasProperty(theAddress))
	{
		//	get the property data
		UInt32 theSize = sizeof(CFStringRef);
		GetPropertyData(theAddress, 0, NULL, theSize, &theAnswer);
	}
	
	return theAnswer;
}

CFStringRef	CAHALAudioObject::CopyNameForElement(AudioObjectPropertyScope inScope, AudioObjectPropertyElement inElement) const
{
	//	set up the return value
	CFStringRef theAnswer = NULL;
	
	//	set up the property address
	CAPropertyAddress theAddress(kAudioObjectPropertyElementName, inScope, inElement);
	
	//	make sure the property exists
	if(HasProperty(theAddress))
	{
		//	get the property data
		UInt32 theSize = sizeof(CFStringRef);
		GetPropertyData(theAddress, 0, NULL, theSize, &theAnswer);
	}
	
	return theAnswer;
}

CFStringRef	CAHALAudioObject::CopyCategoryNameForElement(AudioObjectPropertyScope inScope, AudioObjectPropertyElement inElement) const
{
	//	set up the return value
	CFStringRef theAnswer = NULL;
	
	//	set up the property address
	CAPropertyAddress theAddress(kAudioObjectPropertyElementCategoryName, inScope, inElement);
	
	//	make sure the property exists
	if(HasProperty(theAddress))
	{
		//	get the property data
		UInt32 theSize = sizeof(CFStringRef);
		GetPropertyData(theAddress, 0, NULL, theSize, &theAnswer);
	}
	
	return theAnswer;
}

CFStringRef	CAHALAudioObject::CopyNumberNameForElement(AudioObjectPropertyScope inScope, AudioObjectPropertyElement inElement) const
{
	//	set up the return value
	CFStringRef theAnswer = NULL;
	
	//	set up the property address
	CAPropertyAddress theAddress(kAudioObjectPropertyElementNumberName, inScope, inElement);
	
	//	make sure the property exists
	if(HasProperty(theAddress))
	{
		//	get the property data
		UInt32 theSize = sizeof(CFStringRef);
		GetPropertyData(theAddress, 0, NULL, theSize, &theAnswer);
	}
	
	return theAnswer;
}

UInt32	CAHALAudioObject::GetNumberOwnedObjects(AudioClassID inClass) const
{
	//	set up the return value
	UInt32 theAnswer = 0;
	
	//	set up the property address
	CAPropertyAddress theAddress(kAudioObjectPropertyOwnedObjects);
	
	//	figure out the qualifier
	UInt32 theQualifierSize = 0;
	void* theQualifierData = NULL;
	if(inClass != 0)
	{
		theQualifierSize = sizeof(AudioObjectID);
		theQualifierData = &inClass;
	}
	
	//	get the property data size
	theAnswer = GetPropertyDataSize(theAddress, theQualifierSize, theQualifierData);
	
	//	calculate the number of object IDs
	theAnswer /= sizeof(AudioObjectID);
	
	return theAnswer;
}

void	CAHALAudioObject::GetAllOwnedObjects(AudioClassID inClass, UInt32& ioNumberObjects, AudioObjectID* ioObjectIDs) const
{
	//	set up the property address
	CAPropertyAddress theAddress(kAudioObjectPropertyOwnedObjects);
	
	//	figure out the qualifier
	UInt32 theQualifierSize = 0;
	void* theQualifierData = NULL;
	if(inClass != 0)
	{
		theQualifierSize = sizeof(AudioObjectID);
		theQualifierData = &inClass;
	}
	
	//	get the property data
	UInt32 theDataSize = ioNumberObjects * sizeof(AudioClassID);
	GetPropertyData(theAddress, theQualifierSize, theQualifierData, theDataSize, ioObjectIDs);
	
	//	set the number of object IDs being returned
	ioNumberObjects = theDataSize / sizeof(AudioObjectID);
}

AudioObjectID	CAHALAudioObject::GetOwnedObjectByIndex(AudioClassID inClass, UInt32 inIndex)
{
	//	set up the property address
	CAPropertyAddress theAddress(kAudioObjectPropertyOwnedObjects);
	
	//	figure out the qualifier
	UInt32 theQualifierSize = 0;
	void* theQualifierData = NULL;
	if(inClass != 0)
	{
		theQualifierSize = sizeof(AudioObjectID);
		theQualifierData = &inClass;
	}
	
	//	figure out how much space to allocate
	UInt32 theDataSize = GetPropertyDataSize(theAddress, theQualifierSize, theQualifierData);
	UInt32 theNumberObjectIDs = theDataSize / sizeof(AudioObjectID);
	
	//	set up the return value
	AudioObjectID theAnswer = 0;
	
	//	maker sure the index is in range
	if(inIndex < theNumberObjectIDs)
	{
		//	allocate it
		CAAutoArrayDelete<AudioObjectID> theObjectList(theDataSize / sizeof(AudioObjectID));
		
		//	get the property data
		GetPropertyData(theAddress, theQualifierSize, theQualifierData, theDataSize, theObjectList);
		
		//	get the return value
		theAnswer = theObjectList[inIndex];
	}
	
	return theAnswer;
}

bool	CAHALAudioObject::HasProperty(AudioObjectPropertyAddress& inAddress) const
{
	return AudioObjectHasProperty(mObjectID, &inAddress);
}

bool	CAHALAudioObject::IsPropertySettable(AudioObjectPropertyAddress& inAddress) const
{
	Boolean isSettable = false;
	OSStatus theError = AudioObjectIsPropertySettable(mObjectID, &inAddress, &isSettable);
	ThrowIfError(theError, CAException(theError), "CAHALAudioObject::IsPropertySettable: got an error getting info about a property");
	return isSettable != 0;
}

UInt32	CAHALAudioObject::GetPropertyDataSize(AudioObjectPropertyAddress& inAddress, UInt32 inQualifierDataSize, const void* inQualifierData) const
{
	UInt32 theDataSize = 0;
	OSStatus theError = AudioObjectGetPropertyDataSize(mObjectID, &inAddress, inQualifierDataSize, inQualifierData, &theDataSize);
	ThrowIfError(theError, CAException(theError), "CAHALAudioObject::GetPropertyDataSize: got an error getting the property data size");
	return theDataSize;
}

void	CAHALAudioObject::GetPropertyData(AudioObjectPropertyAddress& inAddress, UInt32 inQualifierDataSize, const void* inQualifierData, UInt32& ioDataSize, void* outData) const
{
	OSStatus theError = AudioObjectGetPropertyData(mObjectID, &inAddress, inQualifierDataSize, inQualifierData, &ioDataSize, outData);
	ThrowIfError(theError, CAException(theError), "CAHALAudioObject::GetPropertyData: got an error getting the property data");
}

void	CAHALAudioObject::SetPropertyData(AudioObjectPropertyAddress& inAddress, UInt32 inQualifierDataSize, const void* inQualifierData, UInt32 inDataSize, const void* inData)
{
	OSStatus theError = AudioObjectSetPropertyData(mObjectID, &inAddress, inQualifierDataSize, inQualifierData, inDataSize, inData);
	ThrowIfError(theError, CAException(theError), "CAHALAudioObject::SetPropertyData: got an error setting the property data");
}

void	CAHALAudioObject::AddPropertyListener(AudioObjectPropertyAddress& inAddress, AudioObjectPropertyListenerProc inListenerProc, void* inClientData)
{
	OSStatus theError = AudioObjectAddPropertyListener(mObjectID, &inAddress, inListenerProc, inClientData);
	ThrowIfError(theError, CAException(theError), "CAHALAudioObject::AddPropertyListener: got an error adding a property listener");
}

void	CAHALAudioObject::RemovePropertyListener(AudioObjectPropertyAddress& inAddress, AudioObjectPropertyListenerProc inListenerProc, void* inClientData)
{
	OSStatus theError = AudioObjectRemovePropertyListener(mObjectID, &inAddress, inListenerProc, inClientData);
	ThrowIfError(theError, CAException(theError), "CAHALAudioObject::RemovePropertyListener: got an error removing a property listener");
}
