/*=============================================================================
	CABufferList.h
	
	$Log: CABufferList.h,v $
	Revision 1.30  2005/01/05 23:56:54  dwyatt
	satisfy gcc 4 (SDK)
	
	Revision 1.29  2004/08/24 00:23:22  bills
	fix warnings
	
	Revision 1.28  2004/08/23 19:20:20  jcm10
	clean up after the Windows changes
	
	Revision 1.27  2004/08/23 19:07:11  dwyatt
	fix operator delete
	
	Revision 1.26  2004/08/23 06:23:25  jcm10
	make it build on Windows
	
	Revision 1.25  2004/03/15 22:56:34  dwyatt
	cleanup, lose changes made for benefit of CAAudioUnit, remove unimplemented "use external buffer" stuff
	
	Revision 1.24  2003/12/16 20:14:44  dwyatt
	Windows compile
	
	Revision 1.23  2003/12/04 22:55:20  luke
	add PrepareBuffer() function & make prepareXXX() args frames, not bytes
	
	Revision 1.22  2003/12/03 23:14:18  bills
	change prepare NULL buffer list to return a pointer not a reference
	
	Revision 1.21  2003/12/02 20:51:34  luke
	add PrepareNullBuffer (from AUBuffer) for CAAudioUnit's benefit
	
	Revision 1.20  2003/11/22 00:04:13  dwyatt
	fix flat header include
	
	Revision 1.19  2003/11/20 22:56:53  dwyatt
	__COREAUDIO_USE_FLAT_INCLUDES__
	
	Revision 1.18  2003/08/02 00:26:40  dwyatt
	revamp print
	
	Revision 1.17  2003/07/17 18:50:24  dwyatt
	allow 0 channels per buffer in constructor, to prevent massive debug spew when AudioFormat makes converters
	
	Revision 1.16  2003/06/18 22:56:25  asynth
	advance buffer pointer dec mDataByteSize
	
	Revision 1.15  2003/04/23 22:33:13  dwyatt
	more fallout of class merge
	
	Revision 1.14  2003/04/09 23:20:38  asynth
	remove printfs
	
	Revision 1.13  2003/04/08 21:49:57  asynth
	FillBufferFromInputProc was copying memory that had just been freed it in a certain case.
	
	Revision 1.12  2003/04/03 01:36:15  asynth
	AudioConverterReset needs to release buffers
	
	Revision 1.11  2003/04/03 00:12:41  asynth
	add SameDataAs
	
	Revision 1.10  2002/06/13 08:01:33  dwyatt
	fix bug in operator new fix
	
	Revision 1.9  2002/06/11 02:23:01  dwyatt
	make latest gcc3 happy - no offsetof in C++ class
	
	Revision 1.8  2002/06/06 01:34:24  dwyatt
	moved from Public/VLBufferList.h
	
	Revision 1.7  2002/05/20 05:15:56  dwyatt
	DeallocateBuffers now correctly clears mDataByteSize,
	used by AllocateBuffers to decide whether to reallocate
	
	Revision 1.6  2002/05/20 02:17:26  dwyatt
	add SetToZeroes method
	
	Revision 1.5  2002/05/13 13:04:34  dwyatt
	ACStreamDescription => CAStreamBasicDescription, now in separate file
	
	Revision 1.4  2002/04/30 01:20:48  dwyatt
	add utility methods to CAStreamBasicDescription
	
	Revision 1.3  2002/04/25 04:43:46  dwyatt
	fix bug in check in constructor
	
	Revision 1.2  2002/04/23 00:57:41  jcm10
	defined an external reference to the CAShow thingy so it could build
	
	Revision 1.1  2002/04/17 18:02:19  dwyatt
	initial checkin
	
	created 11 Apr 2002, 00:08, Doug Wyatt
	Copyright (c) 2002 Apple Computer, Inc.  All Rights Reserved

	$NoKeywords: $
=============================================================================*/

#ifndef __CABufferList_h__
#define __CABufferList_h__

#include <stddef.h>
#include "CAStreamBasicDescription.h"
#if !defined(__COREAUDIO_USE_FLAT_INCLUDES__)
	#include <CoreServices/CoreServices.h>
#else
	#include <AssertMacros.h>
#endif

extern "C" void CAShowAudioBufferList(const AudioBufferList *abl, int framesToPrint, int wordSize);
				// wordSize: 0 = float32, else integer word size, negative if little-endian

/* ____________________________________________________________________________
//	CABufferList - variable length buffer list

	This class is designed for use in non-simplistic cases. For AudioUnits, AUBufferList
	is preferred.
	
	CABufferList can be used in one of two ways:
		- as mutable pointers into non-owned memory
		- as an immutable array of buffers (owns its own memory).

 	All buffers are assumed to have the same format (number of channels, word size), so that
		we can assume their mDataByteSizes are all the same.
____________________________________________________________________________ */
class CABufferList {
public:
	void *	operator new(size_t /*size*/, int nBuffers) {
				return ::operator new(sizeof(CABufferList) + (nBuffers-1) * sizeof(AudioBuffer));
			}
	static CABufferList *	New(const char *name, const CAStreamBasicDescription &format)
	{
		UInt32 numBuffers = format.NumberChannelStreams(), channelsPerBuffer = format.NumberInterleavedChannels();
		return new(numBuffers) CABufferList(name, numBuffers, channelsPerBuffer);
	}

protected:
	CABufferList(const char *name, UInt32 numBuffers, UInt32 channelsPerBuffer) :
		mName(name),
		mBufferMemory(NULL)
	{
		check(numBuffers > 0 /*&& channelsPerBuffer > 0*/);
		mNumberBuffers = numBuffers;
		AudioBuffer *buf = mBuffers;
		for (UInt32 i = mNumberBuffers; i--; ++buf) {
			buf->mNumberChannels = channelsPerBuffer;
			buf->mDataByteSize = 0;
			buf->mData = NULL;
		}
	}

public:
	~CABufferList()
	{
		if (mBufferMemory)
			delete[] mBufferMemory;
	}
	
	const char *				Name() { return mName; }
	
	const AudioBufferList &		GetBufferList() const { return *(AudioBufferList *)&mNumberBuffers; }
	
	AudioBufferList &			GetModifiableBufferList()
	{
		VerifyNotTrashingOwnedBuffer();
		return _GetBufferList();
	}
	
	UInt32		GetNumBytes() const
	{
		return mBuffers[0].mDataByteSize;
	}
	
	void		SetBytes(UInt32 nBytes, void *data)
	{
		VerifyNotTrashingOwnedBuffer();
		check(mNumberBuffers == 1);
		mBuffers[0].mDataByteSize = nBytes;
		mBuffers[0].mData = data;
	}
	
	void		CopyAllFrom(CABufferList *srcbl, CABufferList *ptrbl)
					// copies bytes from srcbl
					// make ptrbl reflect the length copied
					// note that srcbl may be same as ptrbl!
	{
		// Note that this buffer *can* own memory and its pointers/lengths are not
		// altered; only its buffer contents, which are copied from srcbl.
		// The pointers/lengths in ptrbl are updated to reflect the addresses/lengths
		// of the copied data, and srcbl's contents are consumed.
		ptrbl->VerifyNotTrashingOwnedBuffer();
		UInt32 nBytes = srcbl->GetNumBytes();
		AudioBuffer *mybuf = mBuffers, *srcbuf = srcbl->mBuffers,
					*ptrbuf = ptrbl->mBuffers;
		for (UInt32 i = mNumberBuffers; i--; ++mybuf, ++srcbuf, ++ptrbuf) {
			memmove(mybuf->mData, srcbuf->mData, srcbuf->mDataByteSize);
			ptrbuf->mData = mybuf->mData;
			ptrbuf->mDataByteSize = srcbuf->mDataByteSize;
		}
		if (srcbl != ptrbl)
			srcbl->BytesConsumed(nBytes);
	}
	
	void		AppendFrom(CABufferList *blp, UInt32 nBytes)
	{
		VerifyNotTrashingOwnedBuffer();
		AudioBuffer *mybuf = mBuffers, *srcbuf = blp->mBuffers;
		for (UInt32 i = mNumberBuffers; i--; ++mybuf, ++srcbuf) {
			check(nBytes <= srcbuf->mDataByteSize);
			memcpy((Byte *)mybuf->mData + mybuf->mDataByteSize, srcbuf->mData, nBytes);
			mybuf->mDataByteSize += nBytes;
		}
		blp->BytesConsumed(nBytes);
	}
	
	void		PadWithZeroes(UInt32 desiredBufferSize)
					// for cases where an algorithm (e.g. SRC) requires some
					// padding to create silence following end-of-file
	{
		VerifyNotTrashingOwnedBuffer();
		if (GetNumBytes() > desiredBufferSize) return;
		AudioBuffer *buf = mBuffers;
		for (UInt32 i = mNumberBuffers; i--; ++buf) {
			memset((Byte *)buf->mData + buf->mDataByteSize, 0, desiredBufferSize - buf->mDataByteSize);
			buf->mDataByteSize = desiredBufferSize;
		}
	}
	
	void		SetToZeroes(UInt32 nBytes)
	{
		VerifyNotTrashingOwnedBuffer();
		AudioBuffer *buf = mBuffers;
		for (UInt32 i = mNumberBuffers; i--; ++buf) {
			memset((Byte *)buf->mData, 0, nBytes);
			buf->mDataByteSize = nBytes;
		}
	}
	
	void		Reset()
	{
		DeallocateBuffers();
	}
	
	Boolean SameDataAs(const CABufferList* anotherBufferList)
	{
		// check to see if two buffer lists point to the same memory.
		if (mNumberBuffers != anotherBufferList->mNumberBuffers) return false;
		
		for (UInt32 i = 0; i < mNumberBuffers; ++i) {
			if (mBuffers[i].mData != anotherBufferList->mBuffers[i].mData) return false;
		}
		return true;
	}
	
	void		BytesConsumed(UInt32 nBytes)
					// advance buffer pointers, decrease buffer sizes
	{
		VerifyNotTrashingOwnedBuffer();
		AudioBuffer *buf = mBuffers;
		for (UInt32 i = mNumberBuffers; i--; ++buf) {
			check(nBytes <= buf->mDataByteSize);
			buf->mData = (Byte *)buf->mData + nBytes;
			buf->mDataByteSize -= nBytes;
		}
	}
	
	void		SetFrom(const AudioBufferList *abl)
	{
		VerifyNotTrashingOwnedBuffer();
		memcpy(&_GetBufferList(), abl, (char *)&abl->mBuffers[abl->mNumberBuffers] - (char *)abl);
	}
	
	void		SetFrom(const CABufferList *blp)
	{
		SetFrom(&blp->GetBufferList());
	}
	
	void		SetFrom(const AudioBufferList *abl, UInt32 nBytes)
	{
		VerifyNotTrashingOwnedBuffer();
		AudioBuffer *mybuf = mBuffers;
		const AudioBuffer *srcbuf = abl->mBuffers;
		for (UInt32 i = mNumberBuffers; i--; ++mybuf, ++srcbuf) {
			mybuf->mNumberChannels = srcbuf->mNumberChannels;
			mybuf->mDataByteSize = nBytes;
			mybuf->mData = srcbuf->mData;
		}
	}
	
	void		SetFrom(const CABufferList *blp, UInt32 nBytes)
	{
		SetFrom(&blp->GetBufferList(), nBytes);
	}
	
	AudioBufferList *	ToAudioBufferList(AudioBufferList *abl) const
	{
		memcpy(abl, &GetBufferList(), (char *)&abl->mBuffers[mNumberBuffers] - (char *)abl);
		return abl;
	}
	
	void		AllocateBuffers(UInt32 nBytes);
	void		AllocateBuffersAndCopyFrom(UInt32 nBytes, CABufferList *inCopyFromList, CABufferList *inSetPtrList);
	
	void		DeallocateBuffers();
	
	void		UseExternalBuffer(Byte *ptr, UInt32 nBytes);
    
	void		AdvanceBufferPointers(UInt32 nBytes)
					// this is for bufferlists that function simply as
					// an array of pointers into another bufferlist, being advanced,
					// as in RenderOutput implementations
	{
		VerifyNotTrashingOwnedBuffer();
		AudioBuffer *buf = mBuffers;
		for (UInt32 i = mNumberBuffers; i--; ++buf) {
			buf->mData = (Byte *)buf->mData + nBytes;
			buf->mDataByteSize -= nBytes;
		}
	}
	
	void		SetNumBytes(UInt32 nBytes)
	{
		VerifyNotTrashingOwnedBuffer();
		AudioBuffer *buf = mBuffers;
		for (UInt32 i = mNumberBuffers; i--; ++buf)
			buf->mDataByteSize = nBytes;
	}

	void		Print(const char *label=NULL, int nframes=0, int wordSize=0) const
	{
		if (label == NULL)
			label = mName;
		printf("%s - ", label);
		CAShowAudioBufferList(&GetBufferList(), nframes, wordSize);
		if (mBufferMemory)
			printf("  owned memory @ 0x%p:\n", mBufferMemory);
	}

protected:
	AudioBufferList &	_GetBufferList() { return *(AudioBufferList *)&mNumberBuffers; }	// use with care
							// if we make this public, then we lose ability to call VerifyNotTrashingOwnedBuffer
	void				VerifyNotTrashingOwnedBuffer()
	{
		// This needs to be called from places where we are modifying the buffer list.
		// It's an error to modify the buffer pointers or lengths if we own the buffer memory.
		check(mBufferMemory == NULL);
	}

	const char *						mName;	// for debugging
	Byte *								mBufferMemory;
	// the rest must exactly mirror the structure of AudioBufferList
	UInt32								mNumberBuffers;
	AudioBuffer							mBuffers[1];
};

#endif // __CABufferList_h__
