/* *INDENT-OFF* */ /* ATTR_FORMAT confuses indent, avoid running it for now */
/* Basic, host-specific, and target-specific definitions for GDB.
   Copyright 1986, 1988, 1989, 1990, 1991, 1992, 1993, 1994, 1995, 1996,
   1997, 1998, 1999, 2000, 2001, 2002, 2003, 2004, 2005
   Free Software Foundation, Inc.

   This file is part of GDB.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#ifndef DEFS_H
#define DEFS_H

#include "config.h"		/* Generated by configure.  */

#include <sys/types.h>
#include <stdio.h>
#include <errno.h>		/* System call error return status.  */
#include <limits.h>

#ifdef HAVE_STDDEF_H
#include <stddef.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

/* APPLE LOCAL: We use radical new-wave C language features like uint32_t */
#include <stdint.h>

/* First include ansidecl.h so we can use the various macro definitions
   here and in all subsequent file inclusions.  */

#include "ansidecl.h"

#include "gdb_locale.h"

/* For ``enum target_signal''.  */
#include "gdb/signals.h"

/* Just in case they're not defined in stdio.h.  */

#ifndef SEEK_SET
#define SEEK_SET 0
#endif
#ifndef SEEK_CUR
#define SEEK_CUR 1
#endif

#include <stdarg.h>		/* For va_list.  */

#include "libiberty.h"

/* Rather than duplicate all the logic in BFD for figuring out what
   types to use (which can be pretty complicated), symply define them
   in terms of the corresponding type from BFD.  */

#include "bfd.h"

/* A byte from the program being debugged.  */
typedef bfd_byte gdb_byte;

/* An address in the program being debugged.  Host byte order.  */
typedef bfd_vma CORE_ADDR;

/* This is to make sure that LONGEST is at least as big as CORE_ADDR.  */

#ifndef LONGEST

#ifdef BFD64

#define LONGEST BFD_HOST_64_BIT
#define ULONGEST BFD_HOST_U_64_BIT

#else /* No BFD64 */

#ifdef CC_HAS_LONG_LONG
#define LONGEST long long
#define ULONGEST unsigned long long
#else
#ifdef BFD_HOST_64_BIT
/* BFD_HOST_64_BIT is defined for some hosts that don't have long long
   (e.g. i386-windows) so try it.  */
#define LONGEST BFD_HOST_64_BIT
#define ULONGEST BFD_HOST_U_64_BIT
#else
#define LONGEST long
#define ULONGEST unsigned long
#endif
#endif

#endif /* No BFD64 */

#endif /* ! LONGEST */

#ifndef min
#define min(a, b) ((a) < (b) ? (a) : (b))
#endif
#ifndef max
#define max(a, b) ((a) > (b) ? (a) : (b))
#endif

/* Macros to do string compares.

   NOTE: cagney/2000-03-14:

   While old code can continue to refer to these macros, new code is
   probably better off using strcmp() directly vis: ``strcmp() == 0''
   and ``strcmp() != 0''.

   This is because modern compilers can directly inline strcmp()
   making the original justification for these macros - avoid function
   call overhead by pre-testing the first characters
   (``*X==*Y?...:0'') - redundant.

   ``Even if [...] testing the first character does have a modest
   performance improvement, I'd rather that whenever a performance
   issue is found that we spend the effort on algorithmic
   optimizations than micro-optimizing.'' J.T. */

/* NOTE: cagney/2003-11-23: All instances of STREQ[N] covered by
   testing GDB on a stabs system have been replaced by equivalent
   str[n]cmp calls.  To avoid the possability of introducing bugs when
   making untested changes, the remaining references were deprecated
   rather than replaced.  */

/* DISCLAIMER: cagney/2003-11-23: Simplified definition of these
   macros so that they just map directly onto strcmp equivalent.  I'm
   not responsible for any breakage due to code that relied on the old
   underlying implementation.  */

#define DEPRECATED_STREQ(a,b) (strcmp ((a), (b)) == 0)
#define DEPRECATED_STREQN(a,b,c) (strncmp ((a), (b), (c)) == 0)

/* Check if a character is one of the commonly used C++ marker characters.  */
extern int is_cplus_marker (int);

/* enable xdb commands if set */
extern int xdb_commands;

/* enable dbx commands if set */
extern int dbx_commands;

/* System root path, used to find libraries etc.  */
extern char *gdb_sysroot;

extern int quit_flag;
extern int immediate_quit;
extern int sevenbit_strings;

extern void quit (void);

/* FIXME: cagney/2000-03-13: It has been suggested that the peformance
   benefits of having a ``QUIT'' macro rather than a function are
   marginal.  If the overhead of a QUIT function call is proving
   significant then its calling frequency should probably be reduced
   [kingdon].  A profile analyzing the current situtation is
   needed. */

#ifdef QUIT
/* do twice to force compiler warning */
#define QUIT_FIXME "FIXME"
#define QUIT_FIXME "ignoring redefinition of QUIT"
#else
#define QUIT { \
  if (quit_flag) quit (); \
  if (deprecated_interactive_hook) deprecated_interactive_hook (); \
}
#endif

/* Languages represented in the symbol table and elsewhere.
   This should probably be in language.h, but since enum's can't
   be forward declared to satisfy opaque references before their
   actual definition, needs to be here. */

enum language
  {
    language_unknown,		/* Language not known */
    language_auto,		/* Placeholder for automatic setting */
    language_c,			/* C */
    language_cplus,		/* C++ */
    language_objc,		/* Objective-C */
    /* APPLE LOCAL objcplus */
    language_objcplus,		/* Objective-C++ */
    language_java,		/* Java */
    language_fortran,		/* Fortran */
    language_m2,		/* Modula-2 */
    language_asm,		/* Assembly language */
    language_scm,    		/* Scheme / Guile */
    language_pascal,		/* Pascal */
    language_ada,		/* Ada */
    language_minimal,		/* All other languages, minimal support only */
    nr_languages
  };

enum precision_type
  {
    single_precision,
    double_precision,
    unspecified_precision
  };

/* A generic, not quite boolean, enumeration.  */
enum auto_boolean
{
  AUTO_BOOLEAN_TRUE,
  AUTO_BOOLEAN_FALSE,
  AUTO_BOOLEAN_AUTO
};

/* Potential ways that a function can return a value of a given type.  */
enum return_value_convention
{
  /* Where the return value has been squeezed into one or more
     registers.  */
  RETURN_VALUE_REGISTER_CONVENTION,
  /* Commonly known as the "struct return convention".  The caller
     passes an additional hidden first parameter to the caller.  That
     parameter contains the address at which the value being returned
     should be stored.  While typically, and historically, used for
     large structs, this is convention is applied to values of many
     different types.  */
  RETURN_VALUE_STRUCT_CONVENTION,
  /* Like the "struct return convention" above, but where the ABI
     guarantees that the called function stores the address at which
     the value being returned is stored in a well-defined location,
     such as a register or memory slot in the stack frame.  Don't use
     this if the ABI doesn't explicitly guarantees this.  */
  RETURN_VALUE_ABI_RETURNS_ADDRESS,
  /* Like the "struct return convention" above, but where the ABI
     guarantees that the address at which the value being returned is
     stored will be available in a well-defined location, such as a
     register or memory slot in the stack frame.  Don't use this if
     the ABI doesn't explicitly guarantees this.  */
  RETURN_VALUE_ABI_PRESERVES_ADDRESS,
};

/* the cleanup list records things that have to be undone
   if an error happens (descriptors to be closed, memory to be freed, etc.)
   Each link in the chain records a function to call and an
   argument to give it.

   Use make_cleanup to add an element to the cleanup chain.
   Use do_cleanups to do all cleanup actions back to a given
   point in the chain.  Use discard_cleanups to remove cleanups
   from the chain back to a given point, not doing them.  */

struct cleanup
  {
    struct cleanup *next;
    void (*function) (void *);
    void *arg;
  };


/* The ability to declare that a function never returns is useful, but
   not really required to compile GDB successfully, so the NORETURN and
   ATTR_NORETURN macros normally expand into nothing.  */

/* If compiling with older versions of GCC, a function may be declared
   "volatile" to indicate that it does not return.  */

#ifndef NORETURN
#if defined(__GNUC__) \
     && (__GNUC__ == 1 || (__GNUC__ == 2 && __GNUC_MINOR__ < 7))
#define NORETURN volatile
#else
#define NORETURN		/* nothing */
#endif
#endif

/* GCC 2.5 and later versions define a function attribute "noreturn",
   which is the preferred way to declare that a function never returns.
   However GCC 2.7 appears to be the first version in which this fully
   works everywhere we use it. */

#ifndef ATTR_NORETURN
#if defined(__GNUC__) && (__GNUC__ > 2 || (__GNUC__ == 2 && __GNUC_MINOR__ >= 7))
#define ATTR_NORETURN __attribute__ ((noreturn))
#else
#define ATTR_NORETURN		/* nothing */
#endif
#endif

#ifndef ATTR_FORMAT
#if defined(__GNUC__) && (__GNUC__ > 2 || (__GNUC__ == 2 && __GNUC_MINOR__ >= 4))
#define ATTR_FORMAT(type, x, y) __attribute__ ((format(type, x, y)))
#else
#define ATTR_FORMAT(type, x, y)	/* nothing */
#endif
#endif

/* Be conservative and use enum bitfields only with GCC.
   This is copied from gcc 3.3.1, system.h.  */

#if defined(__GNUC__) && (__GNUC__ >= 2)
#define ENUM_BITFIELD(TYPE) enum TYPE
#else
#define ENUM_BITFIELD(TYPE) unsigned int
#endif

/* Needed for various prototypes */

struct symtab;
struct breakpoint;
struct frame_info;

/* APPLE LOCAL more forward structs */
struct symtab_and_line;
struct ui_out;
struct symbol;

/* From utils.c */

/* APPLE LOCAL begin size */
extern void set_screen_size (void);
extern void set_width (void);
/* APPLE LOCAL end size */

extern void initialize_utils (void);

extern void notice_quit (void);

extern int strcmp_iw (const char *, const char *);

extern int strcmp_iw_ordered (const char *, const char *);

extern int streq (const char *, const char *);

extern int subset_compare (char *, char *);

extern char *safe_strerror (int);

/* APPLE LOCAL begin mmalloc */
extern void *init_malloc (void *);
extern void init_mmalloc_default_pool (void *);
/* APPLE LOCAL end mmalloc */

extern void request_quit (int);

#define	ALL_CLEANUPS	((struct cleanup *)0)

extern void do_cleanups (struct cleanup *);
extern void do_final_cleanups (struct cleanup *);
extern void do_run_cleanups (struct cleanup *);
extern void do_exec_cleanups (struct cleanup *);
extern void do_exec_error_cleanups (struct cleanup *);

extern void discard_cleanups (struct cleanup *);
extern void discard_final_cleanups (struct cleanup *);
extern void discard_exec_error_cleanups (struct cleanup *);
extern void discard_my_cleanups (struct cleanup **, struct cleanup *);

/* NOTE: cagney/2000-03-04: This typedef is strictly for the
   make_cleanup function declarations below. Do not use this typedef
   as a cast when passing functions into the make_cleanup() code.
   Instead either use a bounce function or add a wrapper function.
   Calling a f(char*) function with f(void*) is non-portable. */
typedef void (make_cleanup_ftype) (void *);

extern struct cleanup *make_cleanup (make_cleanup_ftype *, void *);

extern struct cleanup *make_cleanup_freeargv (char **);

struct ui_file;
extern struct cleanup *make_cleanup_ui_file_delete (struct ui_file *);

/* APPLE LOCAL begin cleanup */
struct cleanup *make_cleanup_ui_out_delete (struct ui_out *);
struct cleanup *make_cleanup_restore_uiout (struct ui_out *);
/* APPLE LOCAL end cleanup */

struct section_addr_info;
extern struct cleanup *(make_cleanup_free_section_addr_info 
                        (struct section_addr_info *));

extern struct cleanup *make_cleanup_close (int fd);

extern struct cleanup *make_cleanup_bfd_close (bfd *abfd);

extern struct cleanup *make_final_cleanup (make_cleanup_ftype *, void *);

extern struct cleanup *make_my_cleanup (struct cleanup **,
					make_cleanup_ftype *, void *);

extern struct cleanup *make_run_cleanup (make_cleanup_ftype *, void *);

extern struct cleanup *make_exec_cleanup (make_cleanup_ftype *, void *);
extern struct cleanup *make_exec_error_cleanup (make_cleanup_ftype *, void *);

extern struct cleanup *save_cleanups (void);
extern struct cleanup *save_final_cleanups (void);
extern struct cleanup *save_my_cleanups (struct cleanup **);

extern void restore_cleanups (struct cleanup *);
extern void restore_final_cleanups (struct cleanup *);
extern void restore_my_cleanups (struct cleanup **, struct cleanup *);

extern void free_current_contents (void *);

extern void null_cleanup (void *);

extern int myread (int, char *, int);

extern int query (const char *, ...) ATTR_FORMAT (printf, 1, 2);
extern int nquery (const char *, ...) ATTR_FORMAT (printf, 1, 2);
extern int yquery (const char *, ...) ATTR_FORMAT (printf, 1, 2);

extern void init_page_info (void);

extern char *gdb_realpath (const char *);
extern char *xfullpath (const char *);

extern unsigned long gnu_debuglink_crc32 (unsigned long crc,
                                          unsigned char *buf, size_t len);

/* From demangle.c */

extern void set_demangling_style (char *);

/* APPLE LOCAL fix-and-continue */
/* From fix-and-continue.h */

extern CORE_ADDR decode_fix_and_continue_trampoline (CORE_ADDR);
extern void update_picbase_register (struct symbol *);
extern void fix_command_1 (const char *, const char *, const char *, const char *);
int fix_and_continue_supported (void);
int file_exists_p (const char *);

/* From tm.h */

struct type;
typedef int (use_struct_convention_fn) (int gcc_p, struct type * value_type);
extern use_struct_convention_fn generic_use_struct_convention;


/* Annotation stuff.  */

extern int annotation_level;	/* in stack.c */

extern void begin_line (void);

extern void wrap_here (char *);

extern void reinitialize_more_filter (void);

/* Normal results */
extern struct ui_file *gdb_stdout;
/* Input stream */
extern struct ui_file *gdb_stdin;
/* Serious error notifications */
extern struct ui_file *gdb_stderr;
/* Log/debug/trace messages that should bypass normal stdout/stderr
   filtering.  For moment, always call this stream using
   *_unfiltered. In the very near future that restriction shall be
   removed - either call shall be unfiltered. (cagney 1999-06-13). */
extern struct ui_file *gdb_stdlog;
/* Target output that should bypass normal stdout/stderr filtering.
   For moment, always call this stream using *_unfiltered. In the
   very near future that restriction shall be removed - either call
   shall be unfiltered. (cagney 1999-07-02). */
extern struct ui_file *gdb_stdtarg;
extern struct ui_file *gdb_stdtargerr;
extern struct ui_file *gdb_stdtargin;
/* APPLE LOCAL gdb_null */
extern struct ui_file *gdb_null;

#include "ui-file.h"

/* More generic printf like operations.  Filtered versions may return
   non-locally on error.  */

extern void fputs_filtered (const char *, struct ui_file *);

extern void fputs_unfiltered (const char *, struct ui_file *);

extern int fputc_filtered (int c, struct ui_file *);

extern int fputc_unfiltered (int c, struct ui_file *);

extern int putchar_filtered (int c);

extern int putchar_unfiltered (int c);

extern void puts_filtered (const char *);

extern void puts_unfiltered (const char *);

extern void puts_filtered_tabular (char *string, int width, int right);

extern void puts_debug (char *prefix, char *string, char *suffix);

extern void vprintf_filtered (const char *, va_list) ATTR_FORMAT (printf, 1, 0);

extern void vfprintf_filtered (struct ui_file *, const char *, va_list) ATTR_FORMAT (printf, 2, 0);

extern void fprintf_filtered (struct ui_file *, const char *, ...) ATTR_FORMAT (printf, 2, 3);

extern void fprintfi_filtered (int, struct ui_file *, const char *, ...) ATTR_FORMAT (printf, 3, 4);

extern void printf_filtered (const char *, ...) ATTR_FORMAT (printf, 1, 2);

extern void printfi_filtered (int, const char *, ...) ATTR_FORMAT (printf, 2, 3);

extern void vprintf_unfiltered (const char *, va_list) ATTR_FORMAT (printf, 1, 0);

extern void vfprintf_unfiltered (struct ui_file *, const char *, va_list) ATTR_FORMAT (printf, 2, 0);

extern void fprintf_unfiltered (struct ui_file *, const char *, ...) ATTR_FORMAT (printf, 2, 3);

extern void printf_unfiltered (const char *, ...) ATTR_FORMAT (printf, 1, 2);

extern void print_spaces (int, struct ui_file *);

extern void print_spaces_filtered (int, struct ui_file *);

extern char *n_spaces (int);

extern void fputstr_filtered (const char *str, int quotr, struct ui_file * stream);

extern void fputstr_unfiltered (const char *str, int quotr, struct ui_file * stream);

extern void fputstrn_unfiltered (const char *str, int n, int quotr, struct ui_file * stream);

/* Display the host ADDR on STREAM formatted as ``0x%x''. */
extern void gdb_print_host_address (const void *addr, struct ui_file *stream);

/* Convert a CORE_ADDR into a HEX string.  paddr() is like %08lx.
   paddr_nz() is like %lx.  paddr_u() is like %lu. paddr_width() is
   for ``%*''. */
extern int strlen_paddr (void);
extern char *paddr (CORE_ADDR addr);
extern char *paddr_nz (CORE_ADDR addr);
extern char *paddr_u (CORE_ADDR addr);
extern char *paddr_d (LONGEST addr);

/* Like 0x%lx, replaces deprecated_print_address_numeric.  */
extern const char *paddress (CORE_ADDR addr);

extern char *phex (ULONGEST l, int sizeof_l);
extern char *phex_nz (ULONGEST l, int sizeof_l);
extern char *int_string (LONGEST, int, int, int, int);

/* Like paddr() only print/scan raw CORE_ADDR.  The output from
   core_addr_to_string() can be passed direct to
   string_to_core_addr().  */
extern const char *core_addr_to_string (const CORE_ADDR addr);
extern const char *core_addr_to_string_nz (const CORE_ADDR addr);
extern CORE_ADDR string_to_core_addr (const char *my_string);

/* Return a string that contains a number formatted as a hex
   string.  */
extern char *hex_string (LONGEST);
extern char *hex_string_custom (LONGEST, int);

extern void fprintf_symbol_filtered (struct ui_file *, char *,
				     enum language, int);

extern NORETURN void perror_with_name (const char *) ATTR_NORETURN;

extern void print_sys_errmsg (const char *, int);

/* From regex.c or libc.  BSD 4.4 declares this with the argument type as
   "const char *" in unistd.h, so we can't declare the argument
   as "char *".  */

extern char *re_comp (const char *);

/* From symfile.c */

extern void symbol_file_command (char *, int);

/* Remote targets may wish to use this as their load function.  */
extern void generic_load (char *name, int from_tty);

/* Summarise a download */
struct timeval;
extern void print_transfer_performance (struct ui_file *stream,
					unsigned long data_count,
					unsigned long write_count,
					const struct timeval *start_time,
					const struct timeval *end_time);

/* From top.c */

typedef void initialize_file_ftype (void);

extern char *skip_quoted (char *);

extern char *gdb_readline (char *);

extern char *gdb_readline_wrapper (char *);

extern char *command_line_input (char *, int, char *);

extern void print_prompt (void);

extern int input_from_terminal_p (void);

extern int info_verbose;

/* APPLE LOCAL gdb_quitting */
extern int gdb_quitting;

/* From printcmd.c */

extern void set_next_address (CORE_ADDR);

extern void print_address_symbolic (CORE_ADDR, struct ui_file *, int,
				    char *);

extern int build_address_symbolic (CORE_ADDR addr,
				   int do_demangle, 
				   char **name, 
				   int *offset, 
				   char **filename, 
				   int *line, 	
				   int *unmapped);

extern void deprecated_print_address_numeric (CORE_ADDR, int, struct ui_file *);

extern void print_address (CORE_ADDR, struct ui_file *);

/* From source.c */

#define OPF_TRY_CWD_FIRST     0x01
#define OPF_SEARCH_IN_PATH    0x02

extern int openp (const char *, int, const char *, int, int, char **);

extern int source_full_path_of (char *, char **);

extern void mod_path (char *, char **);

extern void add_path (char *, char **, int);

extern void directory_command (char *, int);

extern char *source_path;

extern void init_source_path (void);

extern void init_last_source_visited (void);

/* From exec.c */

extern void exec_set_section_offsets (bfd_signed_vma text_off,
				      bfd_signed_vma data_off,
				      bfd_signed_vma bss_off);

/* Take over the 'find_mapped_memory' vector from exec.c. */
extern void exec_set_find_memory_regions (int (*) (int (*) (CORE_ADDR, 
							    unsigned long, 
							    int, int, int, 
							    void *),
						   void *));

/* Possible lvalue types.  Like enum language, this should be in
   value.h, but needs to be here for the same reason. */

enum lval_type
  {
    /* Not an lval. */
    not_lval,
    /* In memory.  */
    lval_memory,
    /* In a register.  Registers are relative to a frame.  */
    lval_register,
    /* APPLE LOCAL begin literal register setting */
    /* In a register, specified directly by name.  */
    lval_register_literal,
    /* APPLE LOCAL end literal register setting */
    /* In a gdb internal variable.  */
    lval_internalvar,
    /* Part of a gdb internal variable (structure field).  */
    lval_internalvar_component
  };

/* Control types for commands */

enum misc_command_type
  {
    ok_command,
    end_command,
    else_command,
    nop_command
  };

enum command_control_type
  {
    simple_control,
    break_control,
    continue_control,
    while_control,
    if_control,
    invalid_control
  };

/* Structure for saved commands lines
   (for breakpoints, defined commands, etc).  */

struct command_line
  {
    struct command_line *next;
    char *line;
    enum command_control_type control_type;
    int body_count;
    struct command_line **body_list;
  };

/* APPLE LOCAL: Break out the read_command_lines from Readline,
   so you can read command lines directly from the MI.  */

extern struct command_line *read_command_lines (char *, int);
extern struct command_line * read_command_lines_1 (char * (*read_next_line_func) ());

extern void free_command_lines (struct command_line **);

/* To continue the execution commands when running gdb asynchronously. 
   A continuation structure contains a pointer to a function to be called 
   to finish the command, once the target has stopped. Such mechanism is
   used bt the finish and until commands, and in the remote protocol
   when opening an extended-remote connection. */

struct continuation_arg
  {
    struct continuation_arg *next;
    union continuation_data {
      void *pointer;
      int   integer;
      long  longint;
    } data;
  };

struct continuation
  {
    void (*continuation_hook) (struct continuation_arg *);
    struct continuation_arg *arg_list;
    struct continuation *next;
  };

/* In infrun.c. */
extern struct continuation *cmd_continuation;
/* Used only by the step_1 function. */
extern struct continuation *intermediate_continuation;

/* From utils.c */
extern void add_continuation (void (*)(struct continuation_arg *),
			      struct continuation_arg *);
extern void do_all_continuations (void);
extern void discard_all_continuations (void);

extern void add_intermediate_continuation (void (*)(struct continuation_arg *),
			      struct continuation_arg *);
extern void do_all_intermediate_continuations (void);
extern void discard_all_intermediate_continuations (void);

/* String containing the current directory (what getwd would return).  */

extern char *current_directory;

/* Default radixes for input and output.  Only some values supported.  */
extern unsigned input_radix;
extern unsigned output_radix;

/* Possibilities for prettyprint parameters to routines which print
   things.  Like enum language, this should be in value.h, but needs
   to be here for the same reason.  FIXME:  If we can eliminate this
   as an arg to LA_VAL_PRINT, then we can probably move it back to
   value.h. */

enum val_prettyprint
  {
    Val_no_prettyprint = 0,
    Val_prettyprint,
    /* Use the default setting which the user has specified.  */
    Val_pretty_default
  };

/* The ptid struct is a collection of the various "ids" necessary
   for identifying the inferior.  This consists of the process id
   (pid), thread id (tid), and other fields necessary for uniquely
   identifying the inferior process/thread being debugged.  When
   manipulating ptids, the constructors, accessors, and predicate
   declared in inferior.h should be used.  These are as follows:

      ptid_build	- Make a new ptid from a pid, lwp, and tid.
      pid_to_ptid	- Make a new ptid from just a pid.
      ptid_get_pid	- Fetch the pid component of a ptid.
      ptid_get_lwp	- Fetch the lwp component of a ptid.
      ptid_get_tid	- Fetch the tid component of a ptid.
      ptid_equal	- Test to see if two ptids are equal.

   Please do NOT access the struct ptid members directly (except, of
   course, in the implementation of the above ptid manipulation
   functions).  */

struct ptid
  {
    /* Process id */
    int pid;

    /* Lightweight process id */
    long lwp;

    /* Thread id */
    long tid;
  };

typedef struct ptid ptid_t;

/* APPLE LOCAL begin scheduler locking */
/* infrun.c controls the scheduler locking, but other parts of the
   code may need to swap the mode for their own purposes.  These
   functions allow you to do so. */

enum scheduler_locking_mode {
  scheduler_locking_off = 0,
  scheduler_locking_on = 1,
  scheduler_locking_step = 2
};

enum scheduler_locking_mode 
  set_scheduler_locking_mode (enum scheduler_locking_mode new_mode);
void scheduler_run_this_ptid (struct ptid this_ptid);
int scheduler_lock_on_p ();
struct ptid get_scheduler_lock_ptid ();
struct cleanup * 
  make_cleanup_set_restore_scheduler_locking_mode (enum scheduler_locking_mode new_mode);
/* APPLE LOCAL end scheduler locking */


/* Optional host machine definition.  Pure autoconf targets will not
   need a "xm.h" file.  This will be a symlink to one of the xm-*.h
   files, built by the `configure' script.  */

#ifdef GDB_XM_FILE
#include "xm.h"
#endif

/* Optional native machine support.  Non-native (and possibly pure
   multi-arch) targets do not need a "nm.h" file.  This will be a
   symlink to one of the nm-*.h files, built by the `configure'
   script.  */

#ifdef GDB_NM_FILE
#include "nm.h"
#endif

/* Optional target machine definition.  Pure multi-arch configurations
   do not need a "tm.h" file.  This will be a symlink to one of the
   tm-*.h files, built by the `configure' script.  */

#ifdef GDB_TM_FILE
#include "tm.h"
#endif

/* Assume that fopen accepts the letter "b" in the mode string.
   It is demanded by ISO C9X, and should be supported on all
   platforms that claim to have a standard-conforming C library.  On
   true POSIX systems it will be ignored and have no effect.  There
   may still be systems without a standard-conforming C library where
   an ISO C9X compiler (GCC) is available.  Known examples are SunOS
   4.x and 4.3BSD.  This assumption means these systems are no longer
   supported.  */
#ifndef FOPEN_RB
# include "fopen-bin.h"
#endif

/* Defaults for system-wide constants (if not defined by xm.h, we fake it).
   FIXME: Assumes 2's complement arithmetic */

#if !defined (UINT_MAX)
#define	UINT_MAX ((unsigned int)(~0))	/* 0xFFFFFFFF for 32-bits */
#endif

#if !defined (INT_MAX)
#define	INT_MAX ((int)(UINT_MAX >> 1))	/* 0x7FFFFFFF for 32-bits */
#endif

#if !defined (INT_MIN)
#define INT_MIN ((int)((int) ~0 ^ INT_MAX))	/* 0x80000000 for 32-bits */
#endif

#if !defined (ULONG_MAX)
#define	ULONG_MAX ((unsigned long)(~0L))	/* 0xFFFFFFFF for 32-bits */
#endif

#if !defined (LONG_MAX)
#define	LONG_MAX ((long)(ULONG_MAX >> 1))	/* 0x7FFFFFFF for 32-bits */
#endif

#if !defined (ULONGEST_MAX)
#define	ULONGEST_MAX (~(ULONGEST)0)        /* 0xFFFFFFFFFFFFFFFF for 64-bits */
#endif

#if !defined (LONGEST_MAX)                 /* 0x7FFFFFFFFFFFFFFF for 64-bits */
#define	LONGEST_MAX ((LONGEST)(ULONGEST_MAX >> 1))
#endif

/* Convert a LONGEST to an int.  This is used in contexts (e.g. number of
   arguments to a function, number in a value history, register number, etc.)
   where the value must not be larger than can fit in an int.  */

extern int longest_to_int (LONGEST);

/* Assorted functions we can declare, now that const and volatile are 
   defined.  */

extern char *savestring (const char *, size_t);

/* APPLE LOCAL begin mmalloc */
extern char *msavestring (void *, const char *, size_t);

char *strsave (const char *ptr);

extern char *mstrsave (void *, const char *);

/* Robust versions of same.  Throw an internal error when no memory,
   guard against stray NULL arguments. */
extern void *xmmalloc (void *md, size_t size);
extern void *xmrealloc (void *md, void *ptr, size_t size);
extern void *xmcalloc (void *md, size_t number, size_t size);
extern void xmfree (void *md, void *ptr);
/* APPLE LOCAL end mmalloc */

/* xmalloc(), xrealloc() and xcalloc() have already been declared in
   "libiberty.h". */
extern void xfree (void *);

/* Like xmalloc, but zero the memory.  */
extern void *xzalloc (size_t);

/* Utility macros to allocate typed memory.  Avoids errors like:
   struct foo *foo = xmalloc (sizeof struct bar); and memset (foo,
   sizeof (struct foo), 0).  */
#define XZALLOC(TYPE) ((TYPE*) xzalloc (sizeof (TYPE)))
#define XMALLOC(TYPE) ((TYPE*) xmalloc (sizeof (TYPE)))
#define XCALLOC(NMEMB, TYPE) ((TYPE*) xcalloc ((NMEMB), sizeof (TYPE)))

/* Like asprintf/vasprintf but get an internal_error if the call
   fails. */
extern void xasprintf (char **ret, const char *format, ...) ATTR_FORMAT (printf, 2, 3);
extern void xvasprintf (char **ret, const char *format, va_list ap)
     ATTR_FORMAT (printf, 2, 0);

/* Like asprintf and vasprintf, but return the string, throw an error
   if no memory.  */
extern char *xstrprintf (const char *format, ...) ATTR_FORMAT (printf, 1, 2);
extern char *xstrvprintf (const char *format, va_list ap)
     ATTR_FORMAT (printf, 1, 0);

/* Like snprintf, but throw an error if the output buffer is too small.  */
extern int xsnprintf (char *str, size_t size, const char *format, ...)
     ATTR_FORMAT (printf, 3, 4);

extern int parse_escape (char **);

/* Message to be printed before the error message, when an error occurs.  */

extern char *error_pre_print;

/* Message to be printed before the error message, when an error occurs.  */

extern char *quit_pre_print;

/* Message to be printed before the warning message, when a warning occurs.  */

extern char *warning_pre_print;

extern NORETURN void verror (const char *fmt, va_list ap)
     ATTR_NORETURN ATTR_FORMAT (printf, 1, 0);

extern NORETURN void error (const char *fmt, ...) ATTR_NORETURN ATTR_FORMAT (printf, 1, 2);

extern NORETURN void error_stream (struct ui_file *) ATTR_NORETURN;

extern NORETURN void vfatal (const char *fmt, va_list ap)
     ATTR_NORETURN ATTR_FORMAT (printf, 1, 0);

extern NORETURN void fatal (const char *fmt, ...) ATTR_NORETURN ATTR_FORMAT (printf, 1, 2);

extern NORETURN void internal_verror (const char *file, int line,
				      const char *, va_list ap)
     ATTR_NORETURN ATTR_FORMAT (printf, 3, 0);

extern NORETURN void internal_error (const char *file, int line,
				     const char *, ...) ATTR_NORETURN ATTR_FORMAT (printf, 3, 4);

extern void internal_vwarning (const char *file, int line,
			       const char *, va_list ap)
     ATTR_FORMAT (printf, 3, 0);

extern void internal_warning (const char *file, int line,
			      const char *, ...) ATTR_FORMAT (printf, 3, 4);

extern NORETURN void nomem (long) ATTR_NORETURN;

extern void warning (const char *, ...) ATTR_FORMAT (printf, 1, 2);

extern void vwarning (const char *, va_list args) ATTR_FORMAT (printf, 1, 0);

/* List of known OS ABIs.  If you change this, make sure to update the
   table in osabi.c.  */
enum gdb_osabi
{
  GDB_OSABI_UNINITIALIZED = -1, /* For struct gdbarch_info.  */

  GDB_OSABI_UNKNOWN = 0,	/* keep this zero */

  GDB_OSABI_SVR4,
  GDB_OSABI_HURD,
  GDB_OSABI_SOLARIS,
  GDB_OSABI_OSF1,
  GDB_OSABI_LINUX,
  GDB_OSABI_FREEBSD_AOUT,
  GDB_OSABI_FREEBSD_ELF,
  GDB_OSABI_NETBSD_AOUT,
  GDB_OSABI_NETBSD_ELF,
  GDB_OSABI_OPENBSD_ELF,
  GDB_OSABI_WINCE,
  GDB_OSABI_GO32,
  GDB_OSABI_NETWARE,
  GDB_OSABI_IRIX,
  GDB_OSABI_LYNXOS,
  GDB_OSABI_INTERIX,
  GDB_OSABI_HPUX_ELF,
  GDB_OSABI_HPUX_SOM,
  /* APPLE LOCAL begin Darwin */
  GDB_OSABI_DARWIN,
  GDB_OSABI_DARWIN64,
  /* APPLE LOCAL end Darwin */

  GDB_OSABI_QNXNTO,

  GDB_OSABI_CYGWIN,

  GDB_OSABI_INVALID		/* keep this last */
};

/* Global functions from other, non-gdb GNU thingies.
   Libiberty thingies are no longer declared here.  We include libiberty.h
   above, instead.  */

/* From other system libraries */

#ifdef HAVE_STDDEF_H
#include <stddef.h>
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifndef min
#define min(a, b) ((a) < (b) ? (a) : (b))
#endif
#ifndef max
#define max(a, b) ((a) > (b) ? (a) : (b))
#endif


#ifndef atof
extern double atof (const char *);	/* X3.159-1989  4.10.1.1 */
#endif

/* Various possibilities for alloca.  */
#ifndef alloca
#ifdef __GNUC__
#define alloca __builtin_alloca
#else /* Not GNU C */
#ifdef HAVE_ALLOCA_H
#include <alloca.h>
#else
#ifdef _AIX
#pragma alloca
#else

/* We need to be careful not to declare this in a way which conflicts with
   bison.  Bison never declares it as char *, but under various circumstances
   (like __hpux) we need to use void *.  */
extern void *alloca ();
#endif /* Not _AIX */
#endif /* Not HAVE_ALLOCA_H */
#endif /* Not GNU C */
#endif /* alloca not defined */

/* Dynamic target-system-dependent parameters for GDB. */
#include "gdbarch.h"

/* Maximum size of a register.  Something small, but large enough for
   all known ISAs.  If it turns out to be too small, make it bigger.  */

enum { MAX_REGISTER_SIZE = 16 };

/* Static target-system-dependent parameters for GDB. */

/* Number of bits in a char or unsigned char for the target machine.
   Just like CHAR_BIT in <limits.h> but describes the target machine.  */
#if !defined (TARGET_CHAR_BIT)
#define TARGET_CHAR_BIT 8
#endif

/* If we picked up a copy of CHAR_BIT from a configuration file
   (which may get it by including <limits.h>) then use it to set
   the number of bits in a host char.  If not, use the same size
   as the target. */

#if defined (CHAR_BIT)
#define HOST_CHAR_BIT CHAR_BIT
#else
#define HOST_CHAR_BIT TARGET_CHAR_BIT
#endif

/* The bit byte-order has to do just with numbering of bits in
   debugging symbols and such.  Conceptually, it's quite separate
   from byte/word byte order.  */

#if !defined (BITS_BIG_ENDIAN)
#define BITS_BIG_ENDIAN (TARGET_BYTE_ORDER == BFD_ENDIAN_BIG)
#endif

/* In findvar.c.  */

extern LONGEST extract_signed_integer (const gdb_byte *, int);

/* APPLE LOCAL explicit byte order */
extern LONGEST extract_signed_integer_with_byte_order (const gdb_byte *, int, int);

extern ULONGEST extract_unsigned_integer (const gdb_byte *, int);

/* APPLE LOCAL explicit byte order */
extern ULONGEST extract_unsigned_integer_with_byte_order (const gdb_byte *, int, int);

extern int extract_long_unsigned_integer (const gdb_byte *, int, LONGEST *);

/* APPLE LOCAL explicit byte order */
extern int extract_long_unsigned_integer_with_byte_order (const gdb_byte *, int, LONGEST *, int);

extern CORE_ADDR extract_typed_address (const gdb_byte *buf,
					struct type *type);

extern void store_signed_integer (gdb_byte *, int, LONGEST);

/* APPLE LOCAL explicit byte order */
extern void store_signed_integer_with_byte_order (gdb_byte *, int, LONGEST, int);

extern void store_unsigned_integer (gdb_byte *, int, ULONGEST);

/* APPLE LOCAL explicit byte order */
extern void store_unsigned_integer_with_byte_order (gdb_byte *, int, ULONGEST, int);

extern void store_typed_address (gdb_byte *buf, struct type *type,
				 CORE_ADDR addr);


/* From valops.c */

extern int watchdog;

/* Hooks for alternate command interfaces.  */

/* The name of the interpreter if specified on the command line. */
extern char *interpreter_p;

/* If a given interpreter matches INTERPRETER_P then it should update
   deprecated_command_loop_hook and deprecated_init_ui_hook with the
   per-interpreter implementation.  */
/* FIXME: deprecated_command_loop_hook and deprecated_init_ui_hook
   should be moved here. */

struct target_waitstatus;
struct cmd_list_element;

/* APPLE LOCAL gdb_client_data */
/* Typedef for anonymous data type for event data */
typedef void * gdb_client_data;

/* APPLE LOCAL Debugger_state */
typedef enum {
  STATE_NOT_ACTIVE,
  STATE_ACTIVE,
  STATE_INFERIOR_LOADED,
  STATE_INFERIOR_EXITED,
  STATE_INFERIOR_LOGICALLY_RUNNING,
  STATE_INFERIOR_STOPPED
} Debugger_state;

/* APPLE LOCAL begin hooks */
extern void (*init_ui_hook) (char *argv0);
extern void (*command_loop_hook) (void);
/* APPLE LOCAL end hooks */
extern void (*deprecated_pre_add_symbol_hook) (const char *);
extern void (*deprecated_post_add_symbol_hook) (void);
extern void (*selected_frame_level_changed_hook) (int);
extern int (*deprecated_ui_loop_hook) (int signo);
extern void (*deprecated_init_ui_hook) (char *argv0);
extern void (*deprecated_command_loop_hook) (void);
extern void (*deprecated_show_load_progress) (const char *section,
					      unsigned long section_sent, 
					      unsigned long section_size, 
					      unsigned long total_sent, 
					      unsigned long total_size);
extern void (*deprecated_print_frame_info_listing_hook) (struct symtab * s,
							 int line, int stopline,
							 int noerror);
/* APPLE LOCAL begin hooks */
extern void (*print_frame_more_info_hook) (struct ui_out *uiout,
					    struct symtab_and_line *sal,
					    struct frame_info *fi);
/* APPLE LOCAL end hooks */

/* APPLE LOCAL begin hooks */
/* called in place of printing a source line */
extern void (*print_source_lines_hook)
     (struct symtab *s, int line, int stopline);

/* called when the state of the debugger (i.e. gdb) changes */
extern void (*state_change_hook) (Debugger_state new_state);

/* called when the frame changes (e.g. as the result of "up") */
extern void (*frame_changed_hook) (int new_frame_number);

/* called when the stack changes (i.e. a new frame is added) */
extern void (*stack_changed_hook) (void);

/* called when command line input is needed */
extern char * (*command_line_input_hook) (char *, int, char *);

/* these <command>_hooks are called after the command has processed its arguments 
   and just before it calls out to do the work of the command */
/* called when a stepping command (step, next, stepi, nexti) is issued */
extern void (*stepping_command_hook) (void);
/* APPLE LOCAL end hooks */

extern int (*deprecated_query_hook) (const char *, va_list)
     ATTRIBUTE_FPTR_PRINTF(1,0);
extern void (*deprecated_warning_hook) (const char *, va_list)
     ATTRIBUTE_FPTR_PRINTF(1,0);

extern void (*deprecated_flush_hook) (struct ui_file * stream);
extern void (*deprecated_create_breakpoint_hook) (struct breakpoint * b);
extern void (*deprecated_delete_breakpoint_hook) (struct breakpoint * bpt);
extern void (*deprecated_modify_breakpoint_hook) (struct breakpoint * bpt);
extern void (*deprecated_interactive_hook) (void);
extern void (*deprecated_registers_changed_hook) (void);
extern void (*deprecated_readline_begin_hook) (char *, ...)
     ATTRIBUTE_FPTR_PRINTF_1;
extern char *(*deprecated_readline_hook) (char *);
extern void (*deprecated_readline_end_hook) (void);
extern void (*deprecated_register_changed_hook) (int regno);
extern void (*deprecated_memory_changed_hook) (CORE_ADDR addr, int len);
extern void (*deprecated_context_hook) (int);
extern ptid_t (*deprecated_target_wait_hook) (ptid_t ptid,
					      /* APPLE LOCAL target wait hook */
					      struct target_waitstatus * status,
					      gdb_client_data client_data);

extern void (*deprecated_attach_hook) (void);
extern void (*deprecated_detach_hook) (void);
extern void (*deprecated_call_command_hook) (struct cmd_list_element * c,
					     char *cmd, int from_tty);

extern void (*deprecated_set_hook) (struct cmd_list_element * c);

extern void (*deprecated_error_hook) (void);

/* APPLE LOCAL begin hooks */
/* called when the continue command is issued */
extern void (*continue_command_hook) (void);
/* APPLE LOCAL end hooks */

extern void (*deprecated_error_begin_hook) (void);

/* APPLE LOCAL begin hooks */
/* called when the run command is issued; return 1 means do the run; 0 means do not */
extern int (*run_command_hook) (void);
/* APPLE LOCAL end hooks */

extern int (*deprecated_ui_load_progress_hook) (const char *section,
						unsigned long num);


/* Inhibit window interface if non-zero. */

extern int use_windows;

/* Symbolic definitions of filename-related things.  */
/* FIXME, this doesn't work very well if host and executable
   filesystems conventions are different.  */

#ifdef __MSDOS__
# define CANT_FORK
# define GLOBAL_CURDIR
# define DIRNAME_SEPARATOR ';'
#endif

#ifndef DIRNAME_SEPARATOR
#define DIRNAME_SEPARATOR ':'
#endif

#ifndef SLASH_STRING
#define SLASH_STRING "/"
#endif

/* Provide default definitions of PIDGET, TIDGET, and MERGEPID.
   The name ``TIDGET'' is a historical accident.  Many uses of TIDGET
   in the code actually refer to a lightweight process id, i.e,
   something that can be considered a process id in its own right for
   certain purposes.  */

#ifndef PIDGET
#define PIDGET(PTID) (ptid_get_pid (PTID))
#define TIDGET(PTID) (ptid_get_lwp (PTID))
#define MERGEPID(PID, TID) ptid_build (PID, TID, 0)
#endif

/* Define well known filenos if the system does not define them.  */
#ifndef STDIN_FILENO
#define STDIN_FILENO   0
#endif
#ifndef STDOUT_FILENO
#define STDOUT_FILENO  1
#endif
#ifndef STDERR_FILENO
#define STDERR_FILENO  2
#endif

/* If this definition isn't overridden by the header files, assume
   that isatty and fileno exist on this system.  */
#ifndef ISATTY
#define ISATTY(FP)	(isatty (fileno (FP)))
#endif

/* Ensure that V is aligned to an N byte boundary (B's assumed to be a
   power of 2).  Round up/down when necessary.  Examples of correct
   use include:

   addr = align_up (addr, 8); -- VALUE needs 8 byte alignment
   write_memory (addr, value, len);
   addr += len;

   and:

   sp = align_down (sp - len, 16); -- Keep SP 16 byte aligned
   write_memory (sp, value, len);

   Note that uses such as:

   write_memory (addr, value, len);
   addr += align_up (len, 8);

   and:

   sp -= align_up (len, 8);
   write_memory (sp, value, len);

   are typically not correct as they don't ensure that the address (SP
   or ADDR) is correctly aligned (relying on previous alignment to
   keep things right).  This is also why the methods are called
   "align_..." instead of "round_..." as the latter reads better with
   this incorrect coding style.  */

extern ULONGEST align_up (ULONGEST v, int n);
extern ULONGEST align_down (ULONGEST v, int n);

/* APPLE LOCAL begin CHECK macro */
#define __CHECK_FUNCTION __PRETTY_FUNCTION__

#define CHECK(expression) \
  ((void) ((expression) ? 0 : gdb_check (#expression, __FILE__, __LINE__, __CHECK_FUNCTION)))

#define CHECK_FATAL(expression) \
  ((void) ((expression) ? 0 : gdb_check_fatal (#expression, __FILE__, __LINE__, __CHECK_FUNCTION)))

void gdb_check (const char *str, const char *file, unsigned int line, const char *func);
void gdb_check_fatal (const char *str, const char *file, unsigned int line, const char *func);
/* APPLE LOCAL end CHECK macro */

#endif /* #ifndef DEFS_H */
