/*
 * Copyright (c) 2000-2001 Boris Popov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *    This product includes software developed by Boris Popov.
 * 4. Neither the name of the author nor the names of any co-contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: smb_subr.c,v 1.27.108.2 2005/07/20 05:27:00 lindak Exp $
 */
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/malloc.h>
#include <sys/proc.h>
#include <sys/lock.h>
#include <sys/sysctl.h>
#include <sys/socket.h>
#include <sys/signalvar.h>
#include <sys/mbuf.h>
#include <sys/vnode.h>

#include <sys/smb_apple.h>
#include <sys/utfconv.h>

#include <sys/smb_iconv.h>

#include <netsmb/smb.h>
#include <netsmb/smb_conn.h>
#include <netsmb/smb_rq.h>
#include <netsmb/smb_subr.h>

MALLOC_DEFINE(M_SMBDATA, "SMBDATA", "Misc netsmb data");
MALLOC_DEFINE(M_SMBSTR, "SMBSTR", "netsmb string data");
MALLOC_DEFINE(M_SMBTEMP, "SMBTEMP", "Temp netsmb data");

smb_unichar smb_unieol = 0;

void
smb_scred_init(struct smb_cred *scred, vfs_context_t vfsctx)
{
	scred->scr_vfsctx = vfsctx;
}

PRIVSYM int
smb_sigintr(vfs_context_t vfsctx)
{
	if (vfsctx && vfs_context_issignal(vfsctx, SMB_SIGMASK))
		return (EINTR);
	return (0);
}

char *
smb_strdup(const char *s)
{
	char *p;
	int len;

	len = s ? strlen(s) + 1 : 1;
	p = malloc(len, M_SMBSTR, M_WAITOK);
	if (s)
		bcopy(s, p, len);
	else
		*p = 0;
	return p;
}

/*
 * duplicate string from a user space.
 */
char *
smb_strdupin(char *s, int maxlen)
{
	char *p, bt;
	int len = 0;

	for (p = s; ;p++) {
		if (copyin(CAST_USER_ADDR_T(p), &bt, 1))
			return NULL;
		len++;
		if (maxlen && len > maxlen)
			return NULL;
		if (bt == 0)
			break;
	}
	p = malloc(len, M_SMBSTR, M_WAITOK);
	copyin(CAST_USER_ADDR_T(s), p, len);
	return p;
}

/*
 * duplicate memory block from a user space.
 */
void *
smb_memdupin(void *umem, int len)
{
	char *p;

	if (len > 32 * 1024)
		return NULL;
	p = malloc(len, M_SMBSTR, M_WAITOK);
	if (copyin(CAST_USER_ADDR_T(umem), p, len) == 0)
		return p;
	free(p, M_SMBSTR);
	return NULL;
}

/*
 * duplicate memory block in the kernel space.
 */
void *
smb_memdup(const void *umem, int len)
{
	char *p;

	if (len > 32 * 1024)
		return NULL;
	p = malloc(len, M_SMBSTR, M_WAITOK);
	if (p == NULL)
		return NULL;
	bcopy(umem, p, len);
	return p;
}

void
smb_strfree(char *s)
{
	free(s, M_SMBSTR);
}

void
smb_memfree(void *s)
{
	free(s, M_SMBSTR);
}

void *
smb_zmalloc(unsigned long size, int type, int flags)
{

#ifdef M_ZERO
	return malloc(size, type, flags | M_ZERO);
#else
	void *p = malloc(size, type, flags);
	if (p)
		bzero(p, size);
	return p;
#endif
}

size_t
smb_strtouni(u_int16_t *dst, const char *src, size_t inlen, int flags)
{
	size_t outlen;

        if (!inlen)
                 inlen = strlen(src);
	if (BYTE_ORDER != LITTLE_ENDIAN)
		flags |= UTF_REVERSE_ENDIAN;
	if (utf8_decodestr((u_int8_t *)src, inlen, dst, &outlen, inlen * 2, 0, flags) != 0)
                outlen = 0;
        if (!(flags & UTF_NO_NULL_TERM)) {
                dst[outlen/2] = 0;
                outlen += 2;
        }       
        return (outlen);
}

#ifdef SMB_SOCKETDATA_DEBUG
void
m_dumpm(struct mbuf *m) {
	char *p;
	int len;
	printf("d=");
	while(m) {
		p=mtod(m,char *);
		len=m->m_len;
		printf("(%d)",len);
		while(len--){
			printf("%02x ",((int)*(p++)) & 0xff);
		}
		m=m->m_next;
	};
	printf("\n");
}
#endif

/* all these need review XXX */
#ifndef EPROTO
#define EPROTO ECONNABORTED
#endif
#ifndef ELIBACC
#define ELIBACC ENOENT
#endif
#ifndef ENODATA
#define ENODATA EINVAL
#endif
#ifndef ENOTUNIQ
#define ENOTUNIQ EADDRINUSE
#endif
#ifndef ECOMM
#define ECOMM EIO
#endif
#ifndef ENOMEDIUM
#define ENOMEDIUM EIO
#endif
#ifndef ETIME
#define ETIME ETIMEDOUT
#endif

static struct {
	unsigned nterr;
	unsigned errno;
} nt2errno[] = {
	{NT_STATUS_ACCESS_DENIED,		EACCES},
	{NT_STATUS_ACCESS_VIOLATION,		EACCES},
	{NT_STATUS_ACCOUNT_DISABLED,		EACCES},
	{NT_STATUS_ACCOUNT_RESTRICTION,		EACCES},
	{NT_STATUS_ADDRESS_ALREADY_EXISTS,	EADDRINUSE},
        {NT_STATUS_BAD_NETWORK_NAME,		ENOENT},
	{NT_STATUS_BUFFER_TOO_SMALL,		EMOREDATA},
	{NT_STATUS_CONFLICTING_ADDRESSES,	EADDRINUSE},
	{NT_STATUS_CONNECTION_ABORTED,		ECONNABORTED},
	{NT_STATUS_CONNECTION_DISCONNECTED,	ECONNABORTED},
	{NT_STATUS_CONNECTION_REFUSED,		ECONNREFUSED},
	{NT_STATUS_CONNECTION_RESET,		ENETRESET},
	{NT_STATUS_DEVICE_DOES_NOT_EXIST,	ENODEV},
	{NT_STATUS_DEVICE_PROTOCOL_ERROR,	EPROTO},
	{NT_STATUS_DIRECTORY_NOT_EMPTY,		ENOTEMPTY},
	{NT_STATUS_DISK_FULL,			ENOSPC},
	{NT_STATUS_DLL_NOT_FOUND,		ELIBACC},
	{NT_STATUS_END_OF_FILE,			ENODATA},
	{NT_STATUS_FILE_IS_A_DIRECTORY,		EISDIR},
	{NT_STATUS_FLOAT_INEXACT_RESULT,	ERANGE},
	{NT_STATUS_FLOAT_OVERFLOW,		ERANGE},
	{NT_STATUS_FLOAT_UNDERFLOW,		ERANGE},
	{NT_STATUS_HOST_UNREACHABLE,		EHOSTUNREACH},
	{NT_STATUS_ILL_FORMED_PASSWORD,		EACCES},
	{NT_STATUS_INTEGER_OVERFLOW,		ERANGE},
	{NT_STATUS_INVALID_HANDLE,		EBADF},
	{NT_STATUS_INVALID_LOGON_HOURS,		EACCES},
	{NT_STATUS_INVALID_PARAMETER,		EINVAL},
	{NT_STATUS_INVALID_PIPE_STATE,		EPIPE},
	{NT_STATUS_INVALID_WORKSTATION,		EACCES},
	{NT_STATUS_IN_PAGE_ERROR,		EFAULT},
	{NT_STATUS_IO_TIMEOUT,			ETIMEDOUT},
	{NT_STATUS_IP_ADDRESS_CONFLICT1,	ENOTUNIQ},
	{NT_STATUS_IP_ADDRESS_CONFLICT2,	ENOTUNIQ},
	{NT_STATUS_LICENSE_QUOTA_EXCEEDED,	EDQUOT},
	{NT_STATUS_LOGON_FAILURE,		EACCES},
	{NT_STATUS_MEDIA_WRITE_PROTECTED,	EROFS},
	{NT_STATUS_MEMORY_NOT_ALLOCATED,	EFAULT},
	{NT_STATUS_NAME_TOO_LONG,		ENAMETOOLONG},
	{NT_STATUS_NETWORK_ACCESS_DENIED,	EACCES},
	{NT_STATUS_NETWORK_BUSY,		EBUSY},
	{NT_STATUS_NETWORK_UNREACHABLE,		ENETUNREACH},
	{NT_STATUS_NET_WRITE_FAULT,		ECOMM},
	{NT_STATUS_NONEXISTENT_SECTOR,		ESPIPE},
	{NT_STATUS_NOT_A_DIRECTORY,		ENOTDIR},
	{NT_STATUS_NOT_IMPLEMENTED,		ENOSYS},
	{NT_STATUS_NOT_MAPPED_VIEW,		EINVAL},
	{NT_STATUS_NOT_SUPPORTED,		ENOSYS},
	{NT_STATUS_NO_MEDIA,			ENOMEDIUM},
	{NT_STATUS_NO_MEDIA_IN_DEVICE,		ENOMEDIUM},
	{NT_STATUS_NO_MEMORY,			ENOMEM},
	{NT_STATUS_NO_SUCH_DEVICE,		ENODEV},
	{NT_STATUS_NO_SUCH_FILE,		ENOENT},
	{NT_STATUS_OBJECT_NAME_COLLISION,	EEXIST},
	{NT_STATUS_OBJECT_NAME_NOT_FOUND,	ENOENT},
	{NT_STATUS_OBJECT_PATH_INVALID,		ENOTDIR},
	{NT_STATUS_PAGEFILE_QUOTA,		EDQUOT},
	{NT_STATUS_PASSWORD_EXPIRED,		EACCES},
	{NT_STATUS_PASSWORD_RESTRICTION,	EACCES},
	{NT_STATUS_PATH_NOT_COVERED,		ENOENT},
	{NT_STATUS_PIPE_BROKEN,			EPIPE},
	{NT_STATUS_PIPE_BUSY,			EPIPE},
	{NT_STATUS_PIPE_CONNECTED,		EISCONN},
	{NT_STATUS_PIPE_DISCONNECTED,		EPIPE},
	{NT_STATUS_PIPE_NOT_AVAILABLE,		ENOSYS},
	{NT_STATUS_PORT_CONNECTION_REFUSED,	ECONNREFUSED},
	{NT_STATUS_PORT_MESSAGE_TOO_LONG,	EMSGSIZE},
	{NT_STATUS_PORT_UNREACHABLE,		EHOSTUNREACH},
	{NT_STATUS_PROTOCOL_UNREACHABLE,	ENOPROTOOPT},
	{NT_STATUS_QUOTA_EXCEEDED,		EDQUOT},
	{NT_STATUS_REGISTRY_QUOTA_LIMIT,	EDQUOT},
	{NT_STATUS_REMOTE_DISCONNECT,		ESHUTDOWN},
	{NT_STATUS_REMOTE_NOT_LISTENING,	ECONNREFUSED},
	{NT_STATUS_REQUEST_NOT_ACCEPTED,	EACCES},
	{NT_STATUS_RETRY,			EAGAIN},
	{NT_STATUS_SHARING_VIOLATION,		EBUSY},
	{NT_STATUS_TIMER_NOT_CANCELED,		ETIME},
	{NT_STATUS_TOO_MANY_LINKS,		EMLINK},
	{NT_STATUS_TOO_MANY_OPENED_FILES,	EMFILE},
	{NT_STATUS_UNABLE_TO_FREE_VM,		EADDRINUSE},
	{NT_STATUS_UNSUCCESSFUL,		EINVAL},
	{NT_STATUS_WRONG_PASSWORD,		EACCES},
	{0,	0}
};

static struct {
	unsigned dclass;
	unsigned derr;
	unsigned nterr;
} nt2doserr[] = {
	{ERRDOS,	ERRgeneral,	NT_STATUS_UNSUCCESSFUL},
	{ERRDOS,	ERRbadfunc,	NT_STATUS_NOT_IMPLEMENTED},
	{ERRDOS,	ERRinvalidparam,	NT_STATUS_INVALID_INFO_CLASS},
	{ERRDOS,	ERRbadlength,	NT_STATUS_INFO_LENGTH_MISMATCH},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ACCESS_VIOLATION},
	{ERRHRD,	ERRgeneral,	NT_STATUS_IN_PAGE_ERROR},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PAGEFILE_QUOTA},
	{ERRDOS,	ERRbadfid,	NT_STATUS_INVALID_HANDLE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_BAD_INITIAL_STACK},
	{ERRDOS,	193,	NT_STATUS_BAD_INITIAL_PC},
	{ERRDOS,	ERRinvalidparam,	NT_STATUS_INVALID_CID},
	{ERRHRD,	ERRgeneral,	NT_STATUS_TIMER_NOT_CANCELED},
	{ERRDOS,	ERRinvalidparam,	NT_STATUS_INVALID_PARAMETER},
	{ERRDOS,	ERRbadfile,	NT_STATUS_NO_SUCH_DEVICE},
	{ERRDOS,	ERRbadfile,	NT_STATUS_NO_SUCH_FILE},
	{ERRDOS,	ERRbadfunc,	NT_STATUS_INVALID_DEVICE_REQUEST},
	{ERRDOS,	ERRhandleeof,	NT_STATUS_END_OF_FILE},
	{ERRDOS,	ERRwrongdisk,	NT_STATUS_WRONG_VOLUME},
	{ERRDOS,	ERRnotready,	NT_STATUS_NO_MEDIA_IN_DEVICE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_UNRECOGNIZED_MEDIA},
	{ERRDOS,	ERRsectornotfound,	NT_STATUS_NONEXISTENT_SECTOR},
	{ERRDOS,	ERRnomem,	NT_STATUS_NO_MEMORY},
	{ERRDOS,	487,	NT_STATUS_CONFLICTING_ADDRESSES},
	{ERRDOS,	487,	NT_STATUS_NOT_MAPPED_VIEW},
	{ERRDOS,	ERRinvalidparam,	NT_STATUS_UNABLE_TO_FREE_VM},
	{ERRDOS,	ERRinvalidparam,	NT_STATUS_UNABLE_TO_DELETE_SECTION},
	{ERRDOS,	2142,	NT_STATUS_INVALID_SYSTEM_SERVICE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ILLEGAL_INSTRUCTION},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_INVALID_LOCK_SEQUENCE},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_INVALID_VIEW_SIZE},
	{ERRDOS,	193,	NT_STATUS_INVALID_FILE_FOR_SECTION},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_ALREADY_COMMITTED},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_ACCESS_DENIED},
	{ERRDOS,	111,	NT_STATUS_BUFFER_TOO_SMALL},
	{ERRDOS,	ERRbadfid,	NT_STATUS_OBJECT_TYPE_MISMATCH},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NONCONTINUABLE_EXCEPTION},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_DISPOSITION},
	{ERRHRD,	ERRgeneral,	NT_STATUS_UNWIND},
	{ERRHRD,	ERRgeneral,	NT_STATUS_BAD_STACK},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_UNWIND_TARGET},
	{ERRDOS,	158,	NT_STATUS_NOT_LOCKED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PARITY_ERROR},
	{ERRDOS,	487,	NT_STATUS_UNABLE_TO_DECOMMIT_VM},
	{ERRDOS,	487,	NT_STATUS_NOT_COMMITTED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_PORT_ATTRIBUTES},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PORT_MESSAGE_TOO_LONG},
	{ERRDOS,	ERRinvalidparam,	NT_STATUS_INVALID_PARAMETER_MIX},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_QUOTA_LOWER},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DISK_CORRUPT_ERROR},
	{ERRDOS,	ERRinvalidname,	NT_STATUS_OBJECT_NAME_INVALID},
	{ERRDOS,	ERRbadfile,	NT_STATUS_OBJECT_NAME_NOT_FOUND},
	{ERRDOS,	183,	NT_STATUS_OBJECT_NAME_COLLISION},
	{ERRHRD,	ERRgeneral,	NT_STATUS_HANDLE_NOT_WAITABLE},
	{ERRDOS,	ERRbadfid,	NT_STATUS_PORT_DISCONNECTED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DEVICE_ALREADY_ATTACHED},
	{ERRDOS,	161,	NT_STATUS_OBJECT_PATH_INVALID},
	{ERRDOS,	ERRbadpath,	NT_STATUS_OBJECT_PATH_NOT_FOUND},
	{ERRDOS,	161,	NT_STATUS_OBJECT_PATH_SYNTAX_BAD},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DATA_OVERRUN},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DATA_LATE_ERROR},
	{ERRDOS,	ERRcrc,	NT_STATUS_DATA_ERROR},
	{ERRDOS,	ERRcrc,	NT_STATUS_CRC_ERROR},
	{ERRDOS,	ERRnomem,	NT_STATUS_SECTION_TOO_BIG},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_PORT_CONNECTION_REFUSED},
	{ERRDOS,	ERRbadfid,	NT_STATUS_INVALID_PORT_HANDLE},
	{ERRDOS,	ERRbadshare,	NT_STATUS_SHARING_VIOLATION},
	{ERRHRD,	ERRgeneral,	NT_STATUS_QUOTA_EXCEEDED},
	{ERRDOS,	ERRinvalidparam,	NT_STATUS_INVALID_PAGE_PROTECTION},
	{ERRDOS,	288,	NT_STATUS_MUTANT_NOT_OWNED},
	{ERRDOS,	298,	NT_STATUS_SEMAPHORE_LIMIT_EXCEEDED},
	{ERRDOS,	ERRinvalidparam,	NT_STATUS_PORT_ALREADY_SET},
	{ERRDOS,	ERRinvalidparam,	NT_STATUS_SECTION_NOT_IMAGE},
	{ERRDOS,	156,	NT_STATUS_SUSPEND_COUNT_EXCEEDED},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_THREAD_IS_TERMINATING},
	{ERRDOS,	ERRinvalidparam,	NT_STATUS_BAD_WORKING_SET_LIMIT},
	{ERRDOS,	ERRinvalidparam,	NT_STATUS_INCOMPATIBLE_FILE_MAP},
	{ERRDOS,	ERRinvalidparam,	NT_STATUS_SECTION_PROTECTION},
	{ERRDOS,	282,	NT_STATUS_EAS_NOT_SUPPORTED},
	{ERRDOS,	255,	NT_STATUS_EA_TOO_LARGE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NONEXISTENT_EA_ENTRY},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_EAS_ON_FILE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_EA_CORRUPT_ERROR},
	{ERRDOS,	ERRlock,	NT_STATUS_FILE_LOCK_CONFLICT},
	{ERRDOS,	ERRlock,	NT_STATUS_LOCK_NOT_GRANTED},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_DELETE_PENDING},
	{ERRDOS,	ERRunsup,	NT_STATUS_CTL_FILE_NOT_SUPPORTED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_UNKNOWN_REVISION},
	{ERRHRD,	ERRgeneral,	NT_STATUS_REVISION_MISMATCH},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_OWNER},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_PRIMARY_GROUP},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_IMPERSONATION_TOKEN},
	{ERRHRD,	ERRgeneral,	NT_STATUS_CANT_DISABLE_MANDATORY},
	{ERRDOS,	2215,	NT_STATUS_NO_LOGON_SERVERS},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_SUCH_LOGON_SESSION},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_SUCH_PRIVILEGE},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_PRIVILEGE_NOT_HELD},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_ACCOUNT_NAME},
	{ERRHRD,	ERRgeneral,	NT_STATUS_USER_EXISTS},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_NO_SUCH_USER},
	{ERRHRD,	ERRgeneral,	NT_STATUS_GROUP_EXISTS},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_SUCH_GROUP},
	{ERRHRD,	ERRgeneral,	NT_STATUS_MEMBER_IN_GROUP},
	{ERRHRD,	ERRgeneral,	NT_STATUS_MEMBER_NOT_IN_GROUP},
	{ERRHRD,	ERRgeneral,	NT_STATUS_LAST_ADMIN},
	{ERRSRV,	ERRbadpw,	NT_STATUS_WRONG_PASSWORD},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ILL_FORMED_PASSWORD},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PASSWORD_RESTRICTION},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_LOGON_FAILURE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ACCOUNT_RESTRICTION},
	{ERRSRV,	2241,	NT_STATUS_INVALID_LOGON_HOURS},
	{ERRSRV,	2240,	NT_STATUS_INVALID_WORKSTATION},
	{ERRSRV,	2242,	NT_STATUS_PASSWORD_EXPIRED},
	{ERRSRV,	2239,	NT_STATUS_ACCOUNT_DISABLED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NONE_MAPPED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_TOO_MANY_LUIDS_REQUESTED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_LUIDS_EXHAUSTED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_SUB_AUTHORITY},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_ACL},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_SID},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_SECURITY_DESCR},
	{ERRDOS,	127,	NT_STATUS_PROCEDURE_NOT_FOUND},
	{ERRDOS,	193,	NT_STATUS_INVALID_IMAGE_FORMAT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_TOKEN},
	{ERRHRD,	ERRgeneral,	NT_STATUS_BAD_INHERITANCE_ACL},
	{ERRDOS,	158,	NT_STATUS_RANGE_NOT_LOCKED},
	{ERRDOS,	112,	NT_STATUS_DISK_FULL},
	{ERRHRD,	ERRgeneral,	NT_STATUS_SERVER_DISABLED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_SERVER_NOT_DISABLED},
	{ERRDOS,	ERRtoomanynames,	NT_STATUS_TOO_MANY_GUIDS_REQUESTED},
	{ERRDOS,	259,	NT_STATUS_GUIDS_EXHAUSTED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_ID_AUTHORITY},
	{ERRDOS,	259,	NT_STATUS_AGENTS_EXHAUSTED},
	{ERRDOS,	154,	NT_STATUS_INVALID_VOLUME_LABEL},
	{ERRDOS,	ERRoutofmem,	NT_STATUS_SECTION_NOT_EXTENDED},
	{ERRDOS,	487,	NT_STATUS_NOT_MAPPED_DATA},
	{ERRHRD,	ERRgeneral,	NT_STATUS_RESOURCE_DATA_NOT_FOUND},
	{ERRHRD,	ERRgeneral,	NT_STATUS_RESOURCE_TYPE_NOT_FOUND},
	{ERRHRD,	ERRgeneral,	NT_STATUS_RESOURCE_NAME_NOT_FOUND},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ARRAY_BOUNDS_EXCEEDED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FLOAT_DENORMAL_OPERAND},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FLOAT_DIVIDE_BY_ZERO},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FLOAT_INEXACT_RESULT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FLOAT_INVALID_OPERATION},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FLOAT_OVERFLOW},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FLOAT_STACK_CHECK},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FLOAT_UNDERFLOW},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INTEGER_DIVIDE_BY_ZERO},
	{ERRDOS,	534,	NT_STATUS_INTEGER_OVERFLOW},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PRIVILEGED_INSTRUCTION},
	{ERRDOS,	ERRnomem,	NT_STATUS_TOO_MANY_PAGING_FILES},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FILE_INVALID},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ALLOTTED_SPACE_EXCEEDED},
	{ERRDOS,	ERRnomem,	NT_STATUS_INSUFFICIENT_RESOURCES},
	{ERRDOS,	ERRbadpath,	NT_STATUS_DFS_EXIT_PATH_FOUND},
	{ERRDOS,	ERRcrc,	NT_STATUS_DEVICE_DATA_ERROR},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DEVICE_NOT_CONNECTED},
	{ERRDOS,	ERRnotready,	NT_STATUS_DEVICE_POWER_FAILURE},
	{ERRDOS,	487,	NT_STATUS_FREE_VM_NOT_AT_BASE},
	{ERRDOS,	487,	NT_STATUS_MEMORY_NOT_ALLOCATED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_WORKING_SET_QUOTA},
	{ERRDOS,	ERRwriteprotect,	NT_STATUS_MEDIA_WRITE_PROTECTED},
	{ERRDOS,	ERRnotready,	NT_STATUS_DEVICE_NOT_READY},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_GROUP_ATTRIBUTES},
	{ERRHRD,	ERRgeneral,	NT_STATUS_BAD_IMPERSONATION_LEVEL},
	{ERRHRD,	ERRgeneral,	NT_STATUS_CANT_OPEN_ANONYMOUS},
	{ERRHRD,	ERRgeneral,	NT_STATUS_BAD_VALIDATION_CLASS},
	{ERRHRD,	ERRgeneral,	NT_STATUS_BAD_TOKEN_TYPE},
	{ERRDOS,	ERRinvalidparam,	NT_STATUS_BAD_MASTER_BOOT_RECORD},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INSTRUCTION_MISALIGNMENT},
	{ERRDOS,	ERRpipebusy,	NT_STATUS_INSTANCE_NOT_AVAILABLE},
	{ERRDOS,	ERRpipebusy,	NT_STATUS_PIPE_NOT_AVAILABLE},
	{ERRDOS,	ERRbadpipe,	NT_STATUS_INVALID_PIPE_STATE},
	{ERRDOS,	ERRpipebusy,	NT_STATUS_PIPE_BUSY},
	{ERRDOS,	ERRbadfunc,	NT_STATUS_ILLEGAL_FUNCTION},
	{ERRDOS,	ERRnotconnected,	NT_STATUS_PIPE_DISCONNECTED},
	{ERRDOS,	ERRpipeclosing,	NT_STATUS_PIPE_CLOSING},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PIPE_CONNECTED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PIPE_LISTENING},
	{ERRDOS,	ERRbadpipe,	NT_STATUS_INVALID_READ_MODE},
	{ERRDOS,	121,	NT_STATUS_IO_TIMEOUT},
	{ERRDOS,	ERRhandleeof,	NT_STATUS_FILE_FORCED_CLOSED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PROFILING_NOT_STARTED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PROFILING_NOT_STOPPED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_COULD_NOT_INTERPRET},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_FILE_IS_A_DIRECTORY},
	{ERRDOS,	ERRunsup,	NT_STATUS_NOT_SUPPORTED},
	{ERRDOS,	51,	NT_STATUS_REMOTE_NOT_LISTENING},
	{ERRDOS,	52,	NT_STATUS_DUPLICATE_NAME},
	{ERRDOS,	53,	NT_STATUS_BAD_NETWORK_PATH},
	{ERRDOS,	54,	NT_STATUS_NETWORK_BUSY},
	{ERRDOS,	55,	NT_STATUS_DEVICE_DOES_NOT_EXIST},
	{ERRDOS,	56,	NT_STATUS_TOO_MANY_COMMANDS},
	{ERRDOS,	57,	NT_STATUS_ADAPTER_HARDWARE_ERROR},
	{ERRDOS,	58,	NT_STATUS_INVALID_NETWORK_RESPONSE},
	{ERRDOS,	59,	NT_STATUS_UNEXPECTED_NETWORK_ERROR},
	{ERRDOS,	60,	NT_STATUS_BAD_REMOTE_ADAPTER},
	{ERRDOS,	61,	NT_STATUS_PRINT_QUEUE_FULL},
	{ERRDOS,	62,	NT_STATUS_NO_SPOOL_SPACE},
	{ERRDOS,	63,	NT_STATUS_PRINT_CANCELLED},
	{ERRDOS,	64,	NT_STATUS_NETWORK_NAME_DELETED},
	{ERRDOS,	65,	NT_STATUS_NETWORK_ACCESS_DENIED},
	{ERRDOS,	66,	NT_STATUS_BAD_DEVICE_TYPE},
	{ERRDOS,	ERRnosuchshare,	NT_STATUS_BAD_NETWORK_NAME},
	{ERRDOS,	68,	NT_STATUS_TOO_MANY_NAMES},
	{ERRDOS,	69,	NT_STATUS_TOO_MANY_SESSIONS},
	{ERRDOS,	70,	NT_STATUS_SHARING_PAUSED},
	{ERRDOS,	71,	NT_STATUS_REQUEST_NOT_ACCEPTED},
	{ERRDOS,	72,	NT_STATUS_REDIRECTOR_PAUSED},
	{ERRDOS,	88,	NT_STATUS_NET_WRITE_FAULT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PROFILING_AT_LIMIT},
	{ERRDOS,	ERRdiffdevice,	NT_STATUS_NOT_SAME_DEVICE},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_FILE_RENAMED},
	{ERRDOS,	240,	NT_STATUS_VIRTUAL_CIRCUIT_CLOSED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_SECURITY_ON_OBJECT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_CANT_WAIT},
	{ERRDOS,	ERRpipeclosing,	NT_STATUS_PIPE_EMPTY},
	{ERRHRD,	ERRgeneral,	NT_STATUS_CANT_ACCESS_DOMAIN_INFO},
	{ERRHRD,	ERRgeneral,	NT_STATUS_CANT_TERMINATE_SELF},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_SERVER_STATE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_DOMAIN_STATE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_DOMAIN_ROLE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_SUCH_DOMAIN},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DOMAIN_EXISTS},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DOMAIN_LIMIT_EXCEEDED},
	{ERRDOS,	300,	NT_STATUS_OPLOCK_NOT_GRANTED},
	{ERRDOS,	301,	NT_STATUS_INVALID_OPLOCK_PROTOCOL},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INTERNAL_DB_CORRUPTION},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INTERNAL_ERROR},
	{ERRHRD,	ERRgeneral,	NT_STATUS_GENERIC_NOT_MAPPED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_BAD_DESCRIPTOR_FORMAT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_USER_BUFFER},
	{ERRHRD,	ERRgeneral,	NT_STATUS_UNEXPECTED_IO_ERROR},
	{ERRHRD,	ERRgeneral,	NT_STATUS_UNEXPECTED_MM_CREATE_ERR},
	{ERRHRD,	ERRgeneral,	NT_STATUS_UNEXPECTED_MM_MAP_ERROR},
	{ERRHRD,	ERRgeneral,	NT_STATUS_UNEXPECTED_MM_EXTEND_ERR},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NOT_LOGON_PROCESS},
	{ERRHRD,	ERRgeneral,	NT_STATUS_LOGON_SESSION_EXISTS},
	{ERRDOS,	ERRinvalidparam,	NT_STATUS_INVALID_PARAMETER_1},
	{ERRDOS,	ERRinvalidparam,	NT_STATUS_INVALID_PARAMETER_2},
	{ERRDOS,	ERRinvalidparam,	NT_STATUS_INVALID_PARAMETER_3},
	{ERRDOS,	ERRinvalidparam,	NT_STATUS_INVALID_PARAMETER_4},
	{ERRDOS,	ERRinvalidparam,	NT_STATUS_INVALID_PARAMETER_5},
	{ERRDOS,	ERRinvalidparam,	NT_STATUS_INVALID_PARAMETER_6},
	{ERRDOS,	ERRinvalidparam,	NT_STATUS_INVALID_PARAMETER_7},
	{ERRDOS,	ERRinvalidparam,	NT_STATUS_INVALID_PARAMETER_8},
	{ERRDOS,	ERRinvalidparam,	NT_STATUS_INVALID_PARAMETER_9},
	{ERRDOS,	ERRinvalidparam,	NT_STATUS_INVALID_PARAMETER_10},
	{ERRDOS,	ERRinvalidparam,	NT_STATUS_INVALID_PARAMETER_11},
	{ERRDOS,	ERRinvalidparam,	NT_STATUS_INVALID_PARAMETER_12},
	{ERRDOS,	ERRbadpath,	NT_STATUS_REDIRECTOR_NOT_STARTED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_REDIRECTOR_STARTED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_STACK_OVERFLOW},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_SUCH_PACKAGE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_BAD_FUNCTION_TABLE},
	{ERRDOS,	203,	NT_STATUS_VARIABLE_NOT_FOUND},
	{ERRDOS,	145,	NT_STATUS_DIRECTORY_NOT_EMPTY},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FILE_CORRUPT_ERROR},
	{ERRDOS,	267,	NT_STATUS_NOT_A_DIRECTORY},
	{ERRHRD,	ERRgeneral,	NT_STATUS_BAD_LOGON_SESSION_STATE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_LOGON_SESSION_COLLISION},
	{ERRDOS,	206,	NT_STATUS_NAME_TOO_LONG},
	{ERRDOS,	2401,	NT_STATUS_FILES_OPEN},
	{ERRDOS,	2404,	NT_STATUS_CONNECTION_IN_USE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_MESSAGE_NOT_FOUND},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_PROCESS_IS_TERMINATING},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_LOGON_TYPE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_GUID_TRANSLATION},
	{ERRHRD,	ERRgeneral,	NT_STATUS_CANNOT_IMPERSONATE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_IMAGE_ALREADY_LOADED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ABIOS_NOT_PRESENT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ABIOS_LID_NOT_EXIST},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ABIOS_LID_ALREADY_OWNED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ABIOS_NOT_LID_OWNER},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ABIOS_INVALID_COMMAND},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ABIOS_INVALID_LID},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ABIOS_SELECTOR_NOT_AVAILABLE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ABIOS_INVALID_SELECTOR},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_LDT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_LDT_SIZE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_LDT_OFFSET},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_LDT_DESCRIPTOR},
	{ERRDOS,	193,	NT_STATUS_INVALID_IMAGE_NE_FORMAT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_RXACT_INVALID_STATE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_RXACT_COMMIT_FAILURE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_MAPPED_FILE_SIZE_ZERO},
	{ERRDOS,	ERRnofids,	NT_STATUS_TOO_MANY_OPENED_FILES},
	{ERRHRD,	ERRgeneral,	NT_STATUS_CANCELLED},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_CANNOT_DELETE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_COMPUTER_NAME},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_FILE_DELETED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_SPECIAL_ACCOUNT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_SPECIAL_GROUP},
	{ERRHRD,	ERRgeneral,	NT_STATUS_SPECIAL_USER},
	{ERRHRD,	ERRgeneral,	NT_STATUS_MEMBERS_PRIMARY_GROUP},
	{ERRDOS,	ERRbadfid,	NT_STATUS_FILE_CLOSED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_TOO_MANY_THREADS},
	{ERRHRD,	ERRgeneral,	NT_STATUS_THREAD_NOT_IN_PROCESS},
	{ERRHRD,	ERRgeneral,	NT_STATUS_TOKEN_ALREADY_IN_USE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PAGEFILE_QUOTA_EXCEEDED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_COMMITMENT_LIMIT},
	{ERRDOS,	193,	NT_STATUS_INVALID_IMAGE_LE_FORMAT},
	{ERRDOS,	193,	NT_STATUS_INVALID_IMAGE_NOT_MZ},
	{ERRDOS,	193,	NT_STATUS_INVALID_IMAGE_PROTECT},
	{ERRDOS,	193,	NT_STATUS_INVALID_IMAGE_WIN_16},
	{ERRHRD,	ERRgeneral,	NT_STATUS_LOGON_SERVER_CONFLICT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_TIME_DIFFERENCE_AT_DC},
	{ERRHRD,	ERRgeneral,	NT_STATUS_SYNCHRONIZATION_REQUIRED},
	{ERRDOS,	126,	NT_STATUS_DLL_NOT_FOUND},
	{ERRHRD,	ERRgeneral,	NT_STATUS_OPEN_FAILED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_IO_PRIVILEGE_FAILED},
	{ERRDOS,	182,	NT_STATUS_ORDINAL_NOT_FOUND},
	{ERRDOS,	127,	NT_STATUS_ENTRYPOINT_NOT_FOUND},
	{ERRHRD,	ERRgeneral,	NT_STATUS_CONTROL_C_EXIT},
	{ERRDOS,	64,	NT_STATUS_LOCAL_DISCONNECT},
	{ERRDOS,	64,	NT_STATUS_REMOTE_DISCONNECT},
	{ERRDOS,	51,	NT_STATUS_REMOTE_RESOURCES},
	{ERRDOS,	59,	NT_STATUS_LINK_FAILED},
	{ERRDOS,	59,	NT_STATUS_LINK_TIMEOUT},
	{ERRDOS,	59,	NT_STATUS_INVALID_CONNECTION},
	{ERRDOS,	59,	NT_STATUS_INVALID_ADDRESS},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DLL_INIT_FAILED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_MISSING_SYSTEMFILE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_UNHANDLED_EXCEPTION},
	{ERRHRD,	ERRgeneral,	NT_STATUS_APP_INIT_FAILURE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PAGEFILE_CREATE_FAILED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_PAGEFILE},
	{ERRDOS,	124,	NT_STATUS_INVALID_LEVEL},
	{ERRDOS,	86,	NT_STATUS_WRONG_PASSWORD_CORE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ILLEGAL_FLOAT_CONTEXT},
	{ERRDOS,	109,	NT_STATUS_PIPE_BROKEN},
	{ERRHRD,	ERRgeneral,	NT_STATUS_REGISTRY_CORRUPT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_REGISTRY_IO_FAILED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_EVENT_PAIR},
	{ERRHRD,	ERRgeneral,	NT_STATUS_UNRECOGNIZED_VOLUME},
	{ERRHRD,	ERRgeneral,	NT_STATUS_SERIAL_NO_DEVICE_INITED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_SUCH_ALIAS},
	{ERRHRD,	ERRgeneral,	NT_STATUS_MEMBER_NOT_IN_ALIAS},
	{ERRHRD,	ERRgeneral,	NT_STATUS_MEMBER_IN_ALIAS},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ALIAS_EXISTS},
	{ERRHRD,	ERRgeneral,	NT_STATUS_LOGON_NOT_GRANTED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_TOO_MANY_SECRETS},
	{ERRHRD,	ERRgeneral,	NT_STATUS_SECRET_TOO_LONG},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INTERNAL_DB_ERROR},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FULLSCREEN_MODE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_TOO_MANY_CONTEXT_IDS},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_LOGON_TYPE_NOT_GRANTED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NOT_REGISTRY_FILE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NT_CROSS_ENCRYPTION_REQUIRED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DOMAIN_CTRLR_CONFIG_ERROR},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FT_MISSING_MEMBER},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ILL_FORMED_SERVICE_ENTRY},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ILLEGAL_CHARACTER},
	{ERRHRD,	ERRgeneral,	NT_STATUS_UNMAPPABLE_CHARACTER},
	{ERRHRD,	ERRgeneral,	NT_STATUS_UNDEFINED_CHARACTER},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FLOPPY_VOLUME},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FLOPPY_ID_MARK_NOT_FOUND},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FLOPPY_WRONG_CYLINDER},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FLOPPY_UNKNOWN_ERROR},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FLOPPY_BAD_REGISTERS},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DISK_RECALIBRATE_FAILED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DISK_OPERATION_FAILED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DISK_RESET_FAILED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_SHARED_IRQ_BUSY},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FT_ORPHANING},
	{ERRHRD,	ERRgeneral,	NT_STATUS_BIOS_FAILED_TO_CONNECT_INTERRUPT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_16F},
	{ERRHRD,	ERRgeneral,	NT_STATUS_170},
	{ERRHRD,	ERRgeneral,	NT_STATUS_171},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PARTITION_FAILURE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_BLOCK_LENGTH},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DEVICE_NOT_PARTITIONED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_UNABLE_TO_LOCK_MEDIA},
	{ERRHRD,	ERRgeneral,	NT_STATUS_UNABLE_TO_UNLOAD_MEDIA},
	{ERRHRD,	ERRgeneral,	NT_STATUS_EOM_OVERFLOW},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_MEDIA},
	{ERRHRD,	ERRgeneral,	NT_STATUS_179},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_SUCH_MEMBER},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_MEMBER},
	{ERRHRD,	ERRgeneral,	NT_STATUS_KEY_DELETED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_LOG_SPACE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_TOO_MANY_SIDS},
	{ERRHRD,	ERRgeneral,	NT_STATUS_LM_CROSS_ENCRYPTION_REQUIRED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_KEY_HAS_CHILDREN},
	{ERRHRD,	ERRgeneral,	NT_STATUS_CHILD_MUST_BE_VOLATILE},
	{ERRDOS,	ERRinvalidparam,	NT_STATUS_DEVICE_CONFIGURATION_ERROR},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DRIVER_INTERNAL_ERROR},
	{ERRDOS,	ERRbadcmd,	NT_STATUS_INVALID_DEVICE_STATE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_IO_DEVICE_ERROR},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DEVICE_PROTOCOL_ERROR},
	{ERRHRD,	ERRgeneral,	NT_STATUS_BACKUP_CONTROLLER},
	{ERRHRD,	ERRgeneral,	NT_STATUS_LOG_FILE_FULL},
	{ERRDOS,	ERRwriteprotect,	NT_STATUS_TOO_LATE},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_NO_TRUST_LSA_SECRET},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_NO_TRUST_SAM_ACCOUNT},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_TRUSTED_DOMAIN_FAILURE},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_TRUSTED_RELATIONSHIP_FAILURE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_EVENTLOG_FILE_CORRUPT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_EVENTLOG_CANT_START},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_TRUST_FAILURE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_MUTANT_LIMIT_EXCEEDED},
	{ERRDOS,	ERRinvgroup,	NT_STATUS_NETLOGON_NOT_STARTED},
	{ERRSRV,	2239,	NT_STATUS_ACCOUNT_EXPIRED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_POSSIBLE_DEADLOCK},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NETWORK_CREDENTIAL_CONFLICT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_REMOTE_SESSION_LIMIT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_EVENTLOG_FILE_CHANGED},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_NOLOGON_INTERDOMAIN_TRUST_ACCOUNT},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_NOLOGON_WORKSTATION_TRUST_ACCOUNT},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_NOLOGON_SERVER_TRUST_ACCOUNT},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_DOMAIN_TRUST_INCONSISTENT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FS_DRIVER_REQUIRED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_USER_SESSION_KEY},
	{ERRDOS,	59,	NT_STATUS_USER_SESSION_DELETED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_RESOURCE_LANG_NOT_FOUND},
	{ERRDOS,	ERRnomem,	NT_STATUS_INSUFF_SERVER_RESOURCES},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_BUFFER_SIZE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_ADDRESS_COMPONENT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_ADDRESS_WILDCARD},
	{ERRDOS,	68,	NT_STATUS_TOO_MANY_ADDRESSES},
	{ERRDOS,	52,	NT_STATUS_ADDRESS_ALREADY_EXISTS},
	{ERRDOS,	64,	NT_STATUS_ADDRESS_CLOSED},
	{ERRDOS,	64,	NT_STATUS_CONNECTION_DISCONNECTED},
	{ERRDOS,	64,	NT_STATUS_CONNECTION_RESET},
	{ERRDOS,	68,	NT_STATUS_TOO_MANY_NODES},
	{ERRDOS,	59,	NT_STATUS_TRANSACTION_ABORTED},
	{ERRDOS,	59,	NT_STATUS_TRANSACTION_TIMED_OUT},
	{ERRDOS,	59,	NT_STATUS_TRANSACTION_NO_RELEASE},
	{ERRDOS,	59,	NT_STATUS_TRANSACTION_NO_MATCH},
	{ERRDOS,	59,	NT_STATUS_TRANSACTION_RESPONDED},
	{ERRDOS,	59,	NT_STATUS_TRANSACTION_INVALID_ID},
	{ERRDOS,	59,	NT_STATUS_TRANSACTION_INVALID_TYPE},
	{ERRDOS,	ERRunsup,	NT_STATUS_NOT_SERVER_SESSION},
	{ERRDOS,	ERRunsup,	NT_STATUS_NOT_CLIENT_SESSION},
	{ERRHRD,	ERRgeneral,	NT_STATUS_CANNOT_LOAD_REGISTRY_FILE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DEBUG_ATTACH_FAILED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_SYSTEM_PROCESS_TERMINATED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DATA_NOT_ACCEPTED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_BROWSER_SERVERS_FOUND},
	{ERRHRD,	ERRgeneral,	NT_STATUS_VDM_HARD_ERROR},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DRIVER_CANCEL_TIMEOUT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_REPLY_MESSAGE_MISMATCH},
	{ERRHRD,	ERRgeneral,	NT_STATUS_MAPPED_ALIGNMENT},
	{ERRDOS,	193,	NT_STATUS_IMAGE_CHECKSUM_MISMATCH},
	{ERRHRD,	ERRgeneral,	NT_STATUS_LOST_WRITEBEHIND_DATA},
	{ERRHRD,	ERRgeneral,	NT_STATUS_CLIENT_SERVER_PARAMETERS_INVALID},
	{ERRSRV,	2242,	NT_STATUS_PASSWORD_MUST_CHANGE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NOT_FOUND},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NOT_TINY_STREAM},
	{ERRHRD,	ERRgeneral,	NT_STATUS_RECOVERY_FAILURE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_STACK_OVERFLOW_READ},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FAIL_CHECK},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DUPLICATE_OBJECTID},
	{ERRHRD,	ERRgeneral,	NT_STATUS_OBJECTID_EXISTS},
	{ERRHRD,	ERRgeneral,	NT_STATUS_CONVERT_TO_LARGE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_RETRY},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FOUND_OUT_OF_SCOPE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ALLOCATE_BUCKET},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PROPSET_NOT_FOUND},
	{ERRHRD,	ERRgeneral,	NT_STATUS_MARSHALL_OVERFLOW},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_VARIANT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_DOMAIN_CONTROLLER_NOT_FOUND},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_ACCOUNT_LOCKED_OUT},
	{ERRDOS,	ERRbadfid,	NT_STATUS_HANDLE_NOT_CLOSABLE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_CONNECTION_REFUSED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_GRACEFUL_DISCONNECT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ADDRESS_ALREADY_ASSOCIATED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_ADDRESS_NOT_ASSOCIATED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_CONNECTION_INVALID},
	{ERRHRD,	ERRgeneral,	NT_STATUS_CONNECTION_ACTIVE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NETWORK_UNREACHABLE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_HOST_UNREACHABLE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PROTOCOL_UNREACHABLE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PORT_UNREACHABLE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_REQUEST_ABORTED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_CONNECTION_ABORTED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_BAD_COMPRESSION_BUFFER},
	{ERRHRD,	ERRgeneral,	NT_STATUS_USER_MAPPED_FILE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_AUDIT_FAILED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_TIMER_RESOLUTION_NOT_SET},
	{ERRHRD,	ERRgeneral,	NT_STATUS_CONNECTION_COUNT_LIMIT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_LOGIN_TIME_RESTRICTION},
	{ERRHRD,	ERRgeneral,	NT_STATUS_LOGIN_WKSTA_RESTRICTION},
	{ERRDOS,	193,	NT_STATUS_IMAGE_MP_UP_MISMATCH},
	{ERRHRD,	ERRgeneral,	0x000024a},
	{ERRHRD,	ERRgeneral,	0x000024b},
	{ERRHRD,	ERRgeneral,	0x000024c},
	{ERRHRD,	ERRgeneral,	0x000024d},
	{ERRHRD,	ERRgeneral,	0x000024e},
	{ERRHRD,	ERRgeneral,	0x000024f},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INSUFFICIENT_LOGON_INFO},
	{ERRHRD,	ERRgeneral,	NT_STATUS_BAD_DLL_ENTRYPOINT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_BAD_SERVICE_ENTRYPOINT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_LPC_REPLY_LOST},
	{ERRHRD,	ERRgeneral,	NT_STATUS_IP_ADDRESS_CONFLICT1},
	{ERRHRD,	ERRgeneral,	NT_STATUS_IP_ADDRESS_CONFLICT2},
	{ERRHRD,	ERRgeneral,	NT_STATUS_REGISTRY_QUOTA_LIMIT},
	{ERRSRV,	ERRbadtype,	NT_STATUS_PATH_NOT_COVERED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_NO_CALLBACK_ACTIVE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_LICENSE_QUOTA_EXCEEDED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PWD_TOO_SHORT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PWD_TOO_RECENT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PWD_HISTORY_CONFLICT},
	{ERRHRD,	ERRgeneral,	0x000025d},
	{ERRHRD,	ERRgeneral,	NT_STATUS_PLUGPLAY_NO_DEVICE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_UNSUPPORTED_COMPRESSION},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_HW_PROFILE},
	{ERRHRD,	ERRgeneral,	NT_STATUS_INVALID_PLUGPLAY_DEVICE_PATH},
	{ERRDOS,	182,		NT_STATUS_DRIVER_ORDINAL_NOT_FOUND},
	{ERRDOS,	127,		NT_STATUS_DRIVER_ENTRYPOINT_NOT_FOUND},
	{ERRDOS,	288,		NT_STATUS_RESOURCE_NOT_OWNED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_TOO_MANY_LINKS},
	{ERRHRD,	ERRgeneral,	NT_STATUS_QUOTA_LIST_INCONSISTENT},
	{ERRHRD,	ERRgeneral,	NT_STATUS_FILE_IS_OFFLINE},
	{ERRDOS,	ERRnotready,		NT_STATUS_VOLUME_DISMOUNTED},
	{ERRDOS,	161,		NT_STATUS_DIRECTORY_IS_A_REPARSE_POINT},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_ENCRYPTION_FAILED},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_DECRYPTION_FAILED},
	{ERRHRD,	ERRgeneral,	NT_STATUS_RANGE_NOT_FOUND},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_NO_RECOVERY_POLICY},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_NO_EFS},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_WRONG_EFS},
	{ERRDOS,	ERRnoaccess,	NT_STATUS_NO_USER_KEYS},
	{ERRDOS,	ERRbadfunc,	NT_STATUS_VOLUME_NOT_UPGRADED},
};

u_int32_t
smb_maperr32(u_int32_t eno)
{
	int	i;
	unsigned	orig = eno;

	/*
	 * Hi two bits are "severity".  Ignore "success" (0) and
	 * "informational" (1) values.
	 */
	if (!(eno & 0x80000000))
		return 0;
	/* mask off "severity" and the "component"  bit */
	eno &= ~(0xe0000000);

	/* first try direct map to unix */
	for (i = 0; nt2errno[i].errno; i++)
		if (nt2errno[i].nterr == eno)
			return (nt2errno[i].errno);
	SMBERROR("no direct map for 32 bit server error (0x%x)\n", orig);

	/* ok, then try mapping to dos to unix */
	for (i = 0; nt2doserr[i].derr; i++)
		if (nt2doserr[i].nterr == eno)
			return (smb_maperror(nt2doserr[i].dclass,
					     nt2doserr[i].derr));
	return (smb_maperror(ERRHRD, ERRgeneral));
}


int
smb_maperror(int eclass, int eno)
{
	if (eclass == 0 && eno == 0)
		return 0;
	switch (eclass) {
	    case ERRDOS:
		switch (eno) {
		    case ERRbadfunc:
		    case ERRbadenv:
		    case ERRbadformat:
		    case ERRremcd:
		    case ERRrmuns:
			return EINVAL;
		    case ERRbadfile:
		    case ERRbadpath:
		    case ERRnoipc:
		    case ERRnosuchshare:
			return ENOENT;
		    case ERRnofids:
			return EMFILE;
		    case ERRnoaccess:
			/*
			 * XXX CSM Reported on samba-technical 12/7/2002
			 *
			 * There is a case for which server(s) return
			 * ERRnoaccess but should return ERRdiskfull: When
			 * the offset for a write is exactly the server
			 * file size limit then Samba (at least) thinks
			 * the reason for zero bytes having been written
			 * must have been "access denied" from the local
			 * filesystem.  This cannot be easily worked
			 * around since the server behaviour is
			 * indistinguishable from actual access denied.
			 * An incomplete workaround: attempt a 2 byte write
			 * from "offset-1".  (That may require reading at
			 * offset-1 first.)  The flaw is that reading or
			 * writing at offset-1 could cause an
			 * unrelated error (due to a byte range lock
			 * for instance) and we can't presume the
			 * order servers check errors in.
			 */
		    case ERRbadaccess:
			return EACCES;
		    case ERRbadshare:
			return EBUSY;
		    case ERRbadfid:
			return EBADF;
		    case ERRbadmcb:
			return EIO;
		    case ERRnomem:
			return ENOMEM;	/* actually remote no mem... */
		    case ERRbadmem:
			return EFAULT;
		    case ERRbaddata:
			return E2BIG;
		    case ERRbaddrive:
		    case ERRnotready:	/* nt */
			return ENXIO;
		    case ERRdiffdevice:
			return EXDEV;
		    case ERRnofiles:
			return 0;	/* eeof ? */
			return ETXTBSY;
		    case ERRlock:
			return EDEADLK;
		    case ERRfilexists:
			return EEXIST;
		    case ERRinvalidname:	/* samba maps as noent */
			return ENOENT;
		    case ERRdirnotempty:		/* samba */
			return ENOTEMPTY;
		    case ERRnotlocked:
			return 0; /* 0 since bsd unlocks on any close */
		    case ERRrename:
			return EEXIST;
		    case ERRmoredata:
			return EMOREDATA;
		}
		break;
	    case ERRSRV:
		switch (eno) {
		    case ERRerror:
			return EINVAL;
		    case ERRbadpw:
			return EAUTH;
		    case ERRaccess:
		    case ERRbaduid:
			return EACCES;
		    case ERRinvnid:
			return ENETRESET;
		    case ERRinvnetname:
			SMBERROR("NetBIOS name is invalid\n");
			return EAUTH;
		    case ERRbadtype:		/* reserved and returned */
			return EIO;
		    case ERRacctexpired: /* NT: account exists but disabled */
			return EPERM;
		}
		break;
	    case ERRHRD:
		switch (eno) {
		    case ERRnowrite:
			return EROFS;
		    case ERRbadunit:
			return ENODEV;
		    case ERRbadreq:
			return EBADRPC;
		    case ERRbadshare:
			return ETXTBSY;
		    case ERRlock:
			return EDEADLK;
		    case ERRdiskfull:
			return EFBIG;
		    case ERRnotready:
		    case ERRbadcmd:
		    case ERRdata:
		    case ERRgeneral:
			return EIO;
		    default:
			SMBERROR("Unmapped DOS error %d:%d\n", eclass, eno);
			return EIO;
		}
		break;
	}
	SMBERROR("Unmapped DOS error %d:%d\n", eclass, eno);
	return EBADRPC;
}


int
smb_put_dmem(struct mbchain *mbp, struct smb_vc *vcp, const char *src,
	int size, int caseopt, int *lenp)
{
	#pragma unused(caseopt)
	char convbuf[512];
	char *dst;
	size_t inlen, outlen;
	int error;

	if (size == 0)
		return 0;
        if (vcp->vc_toserver == NULL) {
                error =  mb_put_mem(mbp, src, size, MB_MSYSTEM);
                if (!error && lenp)
                        *lenp += size;
                return error;
        }
	inlen = size;
	outlen = sizeof(convbuf);
	dst = convbuf;

	error = iconv_conv(vcp->vc_toserver, &src, &inlen, &dst, &outlen);
	if (error)
		return error;
	outlen = sizeof(convbuf) - outlen;
	if (SMB_UNICODE_STRINGS(vcp))
		mb_put_padbyte(mbp);
	error = mb_put_mem(mbp, (c_caddr_t)convbuf, outlen, MB_MSYSTEM);
        if (!error && lenp)
                *lenp += outlen;
	return error;
}


int
smb_put_dstring(struct mbchain *mbp, struct smb_vc *vcp, const char *src,
	int caseopt)
{
	int error;

	error = smb_put_dmem(mbp, vcp, src, strlen(src), caseopt, NULL);
	if (error)
		return error;
	if (SMB_UNICODE_STRINGS(vcp))
		return mb_put_uint16le(mbp, 0);
	return mb_put_uint8(mbp, 0);
}


int
smb_checksmp(void)
{
	/* APPLE:
	 * just return success...
	 * since kernel_sysctl is broken
	 * and hw_sysctl tries to copyout to user space
	 * and we are always SMP anyway
	 */
	return 0;
}
