/* CanvasPanel.java
 *
 * created: Sat Jun 17 2000
 *
 * This file is part of Artemis
 *
 * Copyright (C) 2000  Genome Research Limited
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Header: /nfs/disk222/yeastpub/Repository/powmap/diana/components/CanvasPanel.java,v 1.1 2000/06/19 11:08:44 kmr Exp $
 */

package diana.components;

import diana.Options;

import java.awt.*;
import java.awt.event.*;

/**
 *  This is a Panel that contains a Panel containing a Canvas.  Both Panels
 *  have BorderLayout.  The Canvas is added at "Center".
 *
 *  @author Kim Rutherford <kmr@sanger.ac.uk>
 *  @version $Id: CanvasPanel.java,v 1.1 2000/06/19 11:08:44 kmr Exp $
 **/

abstract public class CanvasPanel extends Panel {
  /**
   *  Create a new Panel (mid_panel) and a Canvas.
   **/
  public CanvasPanel () {
    setLayout(new BorderLayout ());

    final Font font = getDefaultFont ();

    setFont (font);
    setFontInfo ();

    createCanvas ();
  }

  /**
   *  Create a Panel (mid_panel) and a Canvas object.
   **/
  private void createCanvas () {
    mid_panel = new Panel ();

    mid_panel.setLayout (new BorderLayout());

    canvas = new Canvas ()
      {
      /**
       *  Set the offscreen buffer to null as part of invalidation.
       **/
      public void invalidate () {
        super.invalidate();
        offscreen = null;
      }

      /**
       *  Override update to *not* erase the background before painting
       */
      public void update (final Graphics g) {
        paint(g);
      }

      /**
       *  Paint the canvas.
       */
      public void paint (final Graphics g) {
        final int canvas_width = canvas.getSize ().width;
        final int canvas_height = canvas.getSize ().height;

        if (canvas_height <= 0 || canvas_width <= 0) {
          // there is no point drawing into a zero width canvas
          return;
        }

        if (!isVisible ()) {
          return;
        }

        if (offscreen == null) {
          offscreen = createImage (canvas_width,
                                   canvas_height);
        }

        Graphics og = offscreen.getGraphics ();
        og.setClip (0, 0, canvas_width, canvas_height);

        paintCanvas (og);

        g.drawImage (offscreen, 0, 0, null);
        g.dispose ();
      }
    };

    mid_panel.add (canvas, "Center");
    add (mid_panel, "Center");
  }

  /**
   *  Return the Canvas that was created by createCanvas ().
   **/
  protected Canvas getCanvas () {
    return canvas;
  }

  /**
   *  Returns the sub-Panel that contains the Canvas.
   **/
  protected Panel getMidPanel () {
    return mid_panel;
  }

  /**
   *  Called by canvas.paint () when
   **/
  abstract protected void paintCanvas (final Graphics graphics);

  /**
   *  Set font_width, font_height and font_base_line from the default font.
   **/
  protected void setFontInfo () {
    FontMetrics fm = getFontMetrics (getFont ());

    int [] font_widths = fm.getWidths ();

    font_width = -1;
    // get the maximum character width

    for ( int i = 0 ; i < font_widths.length ; ++i ) {
      if (font_widths[i] > font_width) {
        font_width = font_widths[i];
      }
    }

    font_height = fm.getHeight ();
    font_base_line = fm.getMaxAscent ();
  }

  /**
   *  Return the current default font (from Diana.options).
   **/
  public Font getDefaultFont () {
    return Options.getOptions ().getFont ();
  }

  /**
   *  Return the width of our font, as calculated by setFontInfo ().
   **/
  public int getFontWidth () {
    return font_width;
  }

  /**
   *  Return the height of our font, as calculated by setFontInfo ().
   **/
  public int getFontHeight () {
    return font_height;
  }

  /**
   *  Return the base line of our font, as calculated by setFontInfo ().
   **/
  public int getFontBaseLine () {
    return font_base_line;
  }

  /**
   *  Off screen image used for double buffering when drawing the canvas.
   **/
  private Image offscreen;

  /**
   *  Contains the canvas.
   **/
  private Panel mid_panel = null;

  /**
   *  The drawing area for this component.
   **/
  private Canvas canvas = null;

  /**
   *  The height of the font used in this component.
   **/
  private int font_height;

  /**
   *  The (maximum) width of the font used in this component.
   **/
  private int font_width;

  /**
   *  The base line of the font used in this component.
   **/
  private int font_base_line;
}
