/* ComparatorDialog.java
 *
 * created: Wed May 10 2000
 *
 * This file is part of Artemis
 *
 * Copyright (C) 2000  Genome Research Limited
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Header: /nfs/disk222/yeastpub/Repository/powmap/diana/components/ComparatorDialog.java,v 1.4 2000/09/18 12:37:49 kmr Exp $
 */

package diana.components;

import diana.*;

import uk.ac.sanger.pathogens.OutOfRangeException;
import uk.ac.sanger.pathogens.embl.EntryInformationException;

import java.awt.*;
import java.awt.event.*;
import java.io.*;

/**
 *  ComparatorDialog a dialog that allows the user to the choose two files to
 *  compare with a Comparator component and a file containing comparison data.
 *
 *  @author Kim Rutherford <kmr@sanger.ac.uk>
 *  @version $Id: ComparatorDialog.java,v 1.4 2000/09/18 12:37:49 kmr Exp $
 **/

public class ComparatorDialog extends Frame {
  /**
   *  Create a dialog that allow the user to the choose two files to compare
   *  and a file containing comparison data.
   **/
  public ComparatorDialog () {
    final Font default_font = Options.getOptions ().getFont ();

    setFont (default_font);

    GridBagLayout gridbag = new GridBagLayout();
    setLayout (gridbag);

    GridBagConstraints c = new GridBagConstraints();

    c.fill = GridBagConstraints.HORIZONTAL;
    c.anchor = GridBagConstraints.NORTH;
    c.weighty = 0;


    final Label first_sequence_label = new Label ("Subject sequence:");
    final Panel first_sequence_panel = new Panel ();
    first_sequence_panel.setLayout (new FlowLayout (FlowLayout.LEFT));
    first_sequence_panel.add (first_sequence_label);
    c.gridwidth = 3;
    gridbag.setConstraints (first_sequence_panel, c);
    add (first_sequence_panel);

    final TextField first_sequence_text = new TextField ("", 28);
    gridbag.setConstraints (first_sequence_text, c);
    add (first_sequence_text);

    final Button first_sequence_button = new Button ("Choose ...");

    first_sequence_button.addActionListener (new ActionListener () {
      public void actionPerformed (ActionEvent _) {
        final FileDialog file_dialog =
          new FileDialog (ComparatorDialog.this,
                          "Choose first sequence ...", FileDialog.LOAD);

        file_dialog.setVisible (true);

        if (file_dialog.getFile () != null) {
          first_sequence_text.setText (file_dialog.getDirectory () +
                                       file_dialog.getFile ());
        }
      }
    });

    c.gridwidth = GridBagConstraints.REMAINDER;
    gridbag.setConstraints (first_sequence_button, c);
    add (first_sequence_button);


    final Label second_sequence_label = new Label ("Query sequence:");
    final Panel second_sequence_panel = new Panel ();
    second_sequence_panel.setLayout (new FlowLayout (FlowLayout.LEFT));
    second_sequence_panel.add (second_sequence_label);
    c.gridwidth = 3;
    gridbag.setConstraints (second_sequence_panel, c);
    add (second_sequence_panel);

    final TextField second_sequence_text = new TextField ("", 28);
    gridbag.setConstraints (second_sequence_text, c);
    add (second_sequence_text);

    final Button second_sequence_button = new Button ("Choose ...");

    second_sequence_button.addActionListener (new ActionListener () {
      public void actionPerformed (ActionEvent _) {
        final FileDialog file_dialog =
          new FileDialog (ComparatorDialog.this,
                          "Choose second sequence ...", FileDialog.LOAD);

        file_dialog.setVisible (true);

        if (file_dialog.getFile () != null) {
          second_sequence_text.setText (file_dialog.getDirectory () +
                                        file_dialog.getFile ());
        }
      }
    });

    c.gridwidth = GridBagConstraints.REMAINDER;
    gridbag.setConstraints (second_sequence_button, c);
    add (second_sequence_button);


    final Label comparison_sequence_label =
      new Label ("Comparison data file:");
    final Panel comparison_sequence_panel = new Panel ();
    comparison_sequence_panel.setLayout (new FlowLayout (FlowLayout.LEFT));
    comparison_sequence_panel.add (comparison_sequence_label);
    c.gridwidth = 3;
    gridbag.setConstraints (comparison_sequence_panel, c);
    add (comparison_sequence_panel);

    final TextField comparison_sequence_text = new TextField ("", 28);
    gridbag.setConstraints (comparison_sequence_text, c);
    add (comparison_sequence_text);

    final Button comparison_sequence_button = new Button ("Choose ...");

    comparison_sequence_button.addActionListener (new ActionListener () {
      public void actionPerformed (ActionEvent _) {
        final FileDialog file_dialog =
          new FileDialog (ComparatorDialog.this,
                          "Choose comparison data ...", FileDialog.LOAD);

        file_dialog.setVisible (true);

        if (file_dialog.getFile () != null) {
          comparison_sequence_text.setText (file_dialog.getDirectory () +
                                            file_dialog.getFile ());
        }
      }
    });

    c.gridwidth = GridBagConstraints.REMAINDER;
    gridbag.setConstraints (comparison_sequence_button, c);
    add (comparison_sequence_button);


    final Button apply_button = new Button ("Apply");

    apply_button.addActionListener (new ActionListener () {
      public void actionPerformed (ActionEvent e) {
        doApply (first_sequence_text.getText (),
                 second_sequence_text.getText (),
                 comparison_sequence_text.getText ());
      }
    });


    final Button close_button = new Button ("Close");

    close_button.addActionListener (new ActionListener () {
      public void actionPerformed (ActionEvent e) {
        ComparatorDialog.this.dispose ();
      }
    });


    final FlowLayout flow_layout =
      new FlowLayout (FlowLayout.CENTER, 15, 5);

    final Panel close_and_apply_panel = new Panel (flow_layout);

    close_and_apply_panel.add (apply_button);
    close_and_apply_panel.add (close_button);

    gridbag.setConstraints (close_and_apply_panel, c);
    add (close_and_apply_panel);


    addWindowListener (new WindowAdapter () {
      public void windowClosing (WindowEvent event) {
        ComparatorDialog.this.dispose ();
      }
    });


    pack ();

    final Dimension screen = Toolkit.getDefaultToolkit().getScreenSize();
    setLocation (new Point ((screen.width - getSize ().width) / 2,
                            (screen.height - getSize ().height) / 2));
  }

  /**
   *  Attempt to call ActMain.makeComparator () for the given file names.
   **/
  private void doApply (final String file1, final String file2,
                        final String comparison_data) {

    if (file1.trim ().length () == 0) {
      new MessageDialog (this, "no file given for first sequence");
      return;
    }

    if (file2.trim ().length () == 0) {
      new MessageDialog (this, "no file given for second sequence");
      return;
    }

    if (comparison_data.trim ().length () == 0) {
      new MessageDialog (this, "no file given for comparison data");
      return;
    }

    final MessageFrame reading_message = new MessageFrame ("reading ...");

    try {
      ActMain.makeComparator (this, null, file1, file2, comparison_data);
      ComparatorDialog.this.dispose ();
    } catch (IOException e) {
      new MessageDialog (this, "error while reading: " + e.getMessage ());
    } catch (OutOfRangeException e) {
      new MessageDialog (this, "read failed: one of the features has an " +
                         "out of range location: " + e.getMessage ());
    } catch (EntryInformationException e) {
      new MessageDialog (this, "read failed: " + e.getMessage ());
    } finally {
      reading_message.dispose ();
    }
  }
}
