/* KeyChoice.java
 *
 * created: Mon Sep  6 1999
 *
 * This file is part of Artemis
 * 
 * Copyright (C) 1999  Genome Research Limited
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Header: /nfs/disk222/yeastpub/Repository/powmap/diana/components/KeyChoice.java,v 1.4 2000/02/25 16:24:50 kmr Exp $
 */

package diana.components;

import diana.*;

import uk.ac.sanger.pathogens.StringVector;
import uk.ac.sanger.pathogens.embl.Key;
import uk.ac.sanger.pathogens.embl.KeyVector;
import uk.ac.sanger.pathogens.embl.EntryInformation;

import java.awt.*;
import java.awt.event.*;

/**
 *  This component is a Choice component that shows the possible feature keys.
 *
 *  @author Kim Rutherford
 *  @version $Id: KeyChoice.java,v 1.4 2000/02/25 16:24:50 kmr Exp $
 **/

public class KeyChoice extends Panel {
  /**
   *  Create a new KeyChoice component with CDS as the default key.
   *  @param entry_information The object to get the list of possible
   *    keys from.
   **/
  public KeyChoice (final EntryInformation entry_information) {
    this (entry_information, Key.CDS);
  }

  /**
   *  Create a new KeyChoice component with the given key as the default.
   *  @param entry_information The object to get the list of possible
   *    keys from.
   **/
  public KeyChoice (final EntryInformation entry_information,
                    final Key default_key) {
    this.entry_information = entry_information;
    this.default_key = default_key;

    final Font font = Options.getOptions ().getFont ();
    setFont (font);

    key_chooser = new Choice ();

    common_keys_checkbox = new Checkbox ("Common Keys", true);

    common_keys_checkbox.addItemListener (new ItemListener () {
      public void itemStateChanged(ItemEvent e) {
        updateChoice (common_keys_checkbox.getState ());
      }
    });

    add (common_keys_checkbox);
    updateChoice (true);
  }

  /**
   *  Return the currently selected key.
   **/
  public Key getSelectedItem () {
    final Key key;

    key = new Key (key_chooser.getSelectedItem ());

    return key;
  }

  /**
   *  Set the selected Key.
   **/
  public void setKey (final Key new_key) {
    final int key_index = keyIndex (new_key);

    if (key_index == -1) {
      // add the key
      key_chooser.add (new_key.toString ());
      key_chooser.select (new_key.toString ());
    } else {
      key_chooser.select (key_index);
    }
  }

  /**
   *  Adds the specified item listener to receive item events from the Choice
   *  component of this KeyChoice.
   *  @param l The item listener.
   **/
  public void addItemListener(ItemListener l) {
    key_chooser.addItemListener (l);
  }


  /**
   *  Removes the specified item listener so that it no longer receives item
   *  events from the Choice component of this KeyChoice.
   *  @param l The item listener.
   **/
  public void removeItemListener(ItemListener l) {
    key_chooser.removeItemListener (l);
  }

  /**
   *  Update the key_chooser.
   **/
  private void updateChoice (final boolean use_common_keys) {
    removeAll ();

    while (key_chooser.getItemCount () > 0) {
      key_chooser.remove (key_chooser.getItemCount () - 1);
    }

    final Font font = Options.getOptions ().getFont ();
    key_chooser.setFont (font);

    add (key_chooser);
    add (common_keys_checkbox);
    

    final KeyVector keys;

    if (use_common_keys) {
      keys = getSortedCommonKeys ();
    } else {
      keys = getSortedUncommonKeys ();
    }

    for (int i = 0 ; i < keys.size () ; ++i) {
      key_chooser.add (keys.elementAt (i).toString ());
    }

    if (keyIndex (default_key) != -1) {
      setKey (default_key);
    }

    validate ();

    if (getParent () != null) {
      getParent ().validate ();
      if (getParent ().getParent () != null) {
        getParent ().getParent ().validate ();
      }
    }
  }

  /**
   *  Return a alphanumerically sorted vector containing the String
   *  representations of the common keys (those listed in the common_keys
   *  property of the options file).  If there is no common_keys option then
   *  all the legal keys are returned.
   **/
  private KeyVector getSortedCommonKeys () {
    final StringVector option_keys =
      Options.getOptions ().getOptionValues ("common_keys");

    if (option_keys == null) {
      return entry_information.getSortedValidKeys ();
    }

    option_keys.sort ();

    final KeyVector return_vector = new KeyVector ();

    for (int i = 0 ; i < option_keys.size () ; ++i) {
      return_vector.add (new Key (option_keys.elementAt (i)));
    }
    
    return return_vector;
  }

  /**
   *  Return a alphanumerically sorted vector containing the String
   *  representations of the uncommon keys (those not listed in the
   *  common_keys property of the options file).  If there is no common_keys
   *  option then no keys are returned.
   **/
  private KeyVector getSortedUncommonKeys () {
    final StringVector common_keys =
      Options.getOptions ().getOptionValues ("common_keys");

    final KeyVector possible_keys =
      entry_information.getSortedValidKeys ();

    if (common_keys == null || possible_keys == null) {
      return getSortedCommonKeys ();
    }

    final KeyVector return_vector = new KeyVector ();

    for (int i = 0 ; i < possible_keys.size () ; ++i) {
      if (!common_keys.contains (possible_keys.elementAt (i).toString ())) {
        return_vector.add (possible_keys.elementAt (i));
      }
    }
    
    return return_vector;
  }

  /**
   *  Return the index in the key_chooser component of the given Key.
   **/
  private int keyIndex (final Key key) {
    for (int i = 0 ; i < key_chooser.getItemCount () ; ++i) {
      if (key.toString ().equals (key_chooser.getItem (i))) {
        return i;
      }
    }
    return -1;
  }

  /**
   *  The Key that was passed to the constructor.
   **/
  private Key default_key = null;

  /**
   *  The Choice component that will show the feature keys.
   **/
  private Choice key_chooser = null;

  /**
   *  This toggle sets whether to show common keys or uncommon keys.
   **/
  private Checkbox common_keys_checkbox = null;

  /**
   *  The EntryInformation object that was passed to the constructor.
   **/
  private EntryInformation entry_information = null;
}
