/*
**  CXSC is a C++ library for eXtended Scientific Computing
**
**  Copyright (C) 1990-2000 Institut fuer Angewandte Mathematik,
**                          Universitaet Karlsruhe, Germany
**            (C) 2000-2001 Wiss. Rechnen/Softwaretechnologie
**                          Universitaet Wuppertal, Germany   
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Library General Public
**  License as published by the Free Software Foundation; either
**  version 2 of the License, or (at your option) any later version.
**
**  This library is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
**  Library General Public License for more details.
**
**  You should have received a copy of the GNU Library General Public
**  License along with this library; if not, write to the Free
**  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* CVS $Id: cidot.hpp,v 1.3 2001/08/09 11:37:52 cxsc Exp $ */

#ifndef CXSC_CIDOT_HPP_INCLUDED
#define CXSC_CIDOT_HPP_INCLUDED

#include <iostream>
#include <string>
#include "dot.hpp"
#include "idot.hpp"
#include "cdot.hpp"
#include "cinterva.hpp"

namespace cxsc {

class cidotprecision
{
   private:
      // ---- Datenelemente ---------------------------------------
      dotprecision reinf,resup,iminf,imsup;

   public:
      // ---- Constructors  ---------------------------------------
      cidotprecision() {}
               inline cidotprecision(const cidotprecision &) throw();
      
      explicit inline cidotprecision(const real &) throw();
      explicit inline cidotprecision(const dotprecision &) throw();
      explicit inline cidotprecision(const interval &) throw();
      explicit inline cidotprecision(const idotprecision &) throw();
      explicit inline cidotprecision(const complex &) throw();
      explicit inline cidotprecision(const cdotprecision &) throw();
      explicit inline cidotprecision(const cinterval &) throw();
      
      /*explicit inline cidotprecision(int n) { inf=n,sup=n; }
      explicit inline cidotprecision(int a,int b) { inf=a,sup=b; }
      explicit inline cidotprecision(const double &d) { inf=d,sup=d; }
      explicit inline cidotprecision(const double &a, const double &b) { inf=a,sup=b; }
      explicit inline cidotprecision(const dotprecision &r)   { inf=r,sup=r; }      
      */
            
      inline cidotprecision & operator= (const real & a)         throw() { reinf=resup=a; iminf=imsup=0.0; return *this; }
      inline cidotprecision & operator= (const complex & a)      throw() { reinf=resup=Re(a); iminf=imsup=Im(a); return *this; }
      inline cidotprecision & operator= (const interval & a)     throw() { reinf=Inf(a),resup=Sup(a),iminf=imsup=0.0; return *this; }
      inline cidotprecision & operator= (const cinterval & a)    throw() { reinf=Inf(Re(a)),resup=Sup(Re(a)),iminf=Inf(Im(a)),imsup=Sup(Im(a)); return *this; }
      inline cidotprecision & operator= (const dotprecision & a) throw() { reinf=resup=a; iminf=imsup=0.0; return *this; }
      inline cidotprecision & operator= (const cdotprecision & a)throw() { reinf=resup=Re(a),iminf=imsup=Im(a); return *this; }
      inline cidotprecision & operator= (const idotprecision & a)throw() { reinf=Inf(a),resup=Sup(a),iminf=imsup=0.0; return *this; }
      inline cidotprecision & operator= (const cidotprecision& a)throw() { reinf=a.reinf,resup=a.resup,iminf=a.iminf,imsup=a.imsup; return *this; }

      // ---- Destruktor    ----
      // ~cidotprecision() {} unnoetig

      // ---- Typwandlungen ----
      friend inline cidotprecision _cidotprecision(const complex &,const complex &) throw();
      friend inline cidotprecision _cidotprecision(const complex &,const real &) throw();
      friend inline cidotprecision _cidotprecision(const real &,const complex &) throw();
      friend inline cidotprecision _cidotprecision(const interval &,const interval &) throw();
      friend inline cidotprecision _cidotprecision(const interval &,const real &) throw();
      friend inline cidotprecision _cidotprecision(const real &,const interval &) throw();
      friend inline cidotprecision _cidotprecision(const real &) throw();
      friend inline cidotprecision _cidotprecision(const complex &) throw();
      friend inline cidotprecision _cidotprecision(const interval &) throw();
      friend inline cidotprecision _cidotprecision(const cinterval &) throw();
      
      friend inline cidotprecision _cidotprecision(const idotprecision &,const idotprecision &) throw();
      friend inline cidotprecision _cidotprecision(const cdotprecision &,const cdotprecision &) throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);
      friend inline cidotprecision _cidotprecision(const idotprecision &,const dotprecision &) throw();
      friend inline cidotprecision _cidotprecision(const cdotprecision &,const dotprecision &) throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);
      friend inline cidotprecision _cidotprecision(const dotprecision &,const idotprecision &) throw();
      friend inline cidotprecision _cidotprecision(const dotprecision &,const cdotprecision&) throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);
      friend inline cidotprecision _cidotprecision(const cdotprecision &) throw();
      friend inline cidotprecision _cidotprecision(const idotprecision &) throw();
      friend inline cidotprecision _cidotprecision(const dotprecision &) throw();
      
      
      friend inline cidotprecision _unchecked_cidotprecision(const complex &, const complex &) throw();
      friend inline cidotprecision _unchecked_cidotprecision(const complex &, const real &) throw();
      friend inline cidotprecision _unchecked_cidotprecision(const real &, const complex &) throw();
      
      friend inline cidotprecision _unchecked_cidotprecision(const cdotprecision &, const cdotprecision &) throw();
      friend inline cidotprecision _unchecked_cidotprecision(const cdotprecision &, const dotprecision &) throw();
      friend inline cidotprecision _unchecked_cidotprecision(const dotprecision &, const cdotprecision &) throw();

      // ---- Ausgabefunkt. ---------------------------------------
      friend std::istream& operator >> (std::istream& s, cidotprecision& a)       throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);
      friend std::ostream& operator << (std::ostream& s, const cidotprecision& a) throw();
      friend std::string&  operator >> (std::string&  s, cidotprecision& a)       throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);
      friend std::string&  operator << (std::string&  s, const cidotprecision& a) throw();
      friend void          operator >> (const std::string &s,cidotprecision& a)   throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);
      friend void          operator >> (const char *s       ,cidotprecision& a)   throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);

      // ---- Standardfunkt ---- (arithmetische Operatoren)
      friend inline cidotprecision operator -(cidotprecision) throw();
      friend inline cidotprecision operator +(const cidotprecision &) throw();

      friend inline cidotprecision operator +(const cidotprecision &,const cidotprecision &) throw();
      friend inline cidotprecision operator -(const cidotprecision &,const cidotprecision &) throw();
      friend inline cidotprecision operator |(const cidotprecision &,const cidotprecision &) throw();
      friend inline cidotprecision operator &(const cidotprecision &,const cidotprecision &) throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);

      friend inline cidotprecision operator |(const cdotprecision &,const cdotprecision &)  throw();

      friend inline cidotprecision & operator +=(cidotprecision &, const cidotprecision &) throw();
      friend inline cidotprecision & operator -=(cidotprecision &, const cidotprecision &) throw();
      friend inline cidotprecision & operator |=(cidotprecision &, const cidotprecision &) throw();
      friend inline cidotprecision & operator &=(cidotprecision &, const cidotprecision &) throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);

      // CID-R
      
      friend inline cidotprecision operator +(const cidotprecision &,const real &) throw();
      friend inline cidotprecision operator +(const real &,const cidotprecision &) throw();
      friend inline cidotprecision operator -(const cidotprecision &,const real &) throw();
      friend inline cidotprecision operator -(const real &,const cidotprecision &) throw();
      friend inline cidotprecision operator |(const cidotprecision &,const real &) throw();
      friend inline cidotprecision operator |(const real &,const cidotprecision &) throw();
      friend inline cidotprecision operator &(const cidotprecision &,const real &) throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);
      friend inline cidotprecision operator &(const real &,const cidotprecision &) throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);
      
      friend inline cidotprecision & operator +=(cidotprecision &, const real &) throw();
      friend inline cidotprecision & operator -=(cidotprecision &, const real &) throw();
      friend inline cidotprecision & operator |=(cidotprecision &, const real &) throw();
      friend inline cidotprecision & operator &=(cidotprecision &, const real &) throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);
      
      // CID-C

      friend inline cidotprecision operator +(const cidotprecision &,const complex &) throw();
      friend inline cidotprecision operator +(const complex &,const cidotprecision &) throw();
      friend inline cidotprecision operator -(const cidotprecision &,const complex &) throw();
      friend inline cidotprecision operator -(const complex &,const cidotprecision &) throw();
      friend inline cidotprecision operator |(const cidotprecision &,const complex &) throw();
      friend inline cidotprecision operator |(const complex &,const cidotprecision &) throw();
      friend inline cidotprecision operator &(const cidotprecision &,const complex &) throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);
      friend inline cidotprecision operator &(const complex &,const cidotprecision &) throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);
      

      friend inline cidotprecision & operator +=(cidotprecision &, const complex &) throw();
      friend inline cidotprecision & operator -=(cidotprecision &, const complex &) throw();
      friend inline cidotprecision & operator |=(cidotprecision &, const complex &) throw();
      friend inline cidotprecision & operator &=(cidotprecision &, const complex &) throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);
      
      // CID-I

      friend inline cidotprecision operator +(const cidotprecision &,const interval &) throw();
      friend inline cidotprecision operator +(const interval &,const cidotprecision &) throw();
      friend inline cidotprecision operator -(const cidotprecision &,const interval &) throw();
      friend inline cidotprecision operator -(const interval &,const cidotprecision &) throw();
      friend inline cidotprecision operator |(const cidotprecision &,const interval &) throw();
      friend inline cidotprecision operator |(const interval &,const cidotprecision &) throw();
      friend inline cidotprecision operator &(const cidotprecision &,const interval &) throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);
      friend inline cidotprecision operator &(const interval &,const cidotprecision &) throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);
      
      friend inline cidotprecision & operator +=(cidotprecision &, const interval &) throw();
      friend inline cidotprecision & operator -=(cidotprecision &, const interval &) throw();
      friend inline cidotprecision & operator |=(cidotprecision &, const interval &) throw();
      friend inline cidotprecision & operator &=(cidotprecision &, const interval &) throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);

      // CID-CI

      friend inline cidotprecision operator +(const cidotprecision &,const cinterval &) throw();
      friend inline cidotprecision operator +(const cinterval &,const cidotprecision &) throw();
      friend inline cidotprecision operator -(const cidotprecision &,const cinterval &) throw();
      friend inline cidotprecision operator -(const cinterval &,const cidotprecision &) throw();
      friend inline cidotprecision operator |(const cidotprecision &,const cinterval &) throw();
      friend inline cidotprecision operator |(const cinterval &,const cidotprecision &) throw();
      friend inline cidotprecision operator &(const cidotprecision &,const cinterval &) throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);
      friend inline cidotprecision operator &(const cinterval &,const cidotprecision &) throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);
      
      friend inline cidotprecision & operator +=(cidotprecision &, const cinterval &) throw();
      friend inline cidotprecision & operator -=(cidotprecision &, const cinterval &) throw();
      friend inline cidotprecision & operator |=(cidotprecision &, const cinterval &) throw();
      friend inline cidotprecision & operator &=(cidotprecision &, const cinterval &) throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);
      
      // CID-D
      
      friend inline cidotprecision operator +(const cidotprecision &,const dotprecision &) throw();
      friend inline cidotprecision operator +(const dotprecision &,const cidotprecision &) throw();
      friend inline cidotprecision operator -(const cidotprecision &,const dotprecision &) throw();
      friend inline cidotprecision operator -(const dotprecision &,const cidotprecision &) throw();
      friend inline cidotprecision operator |(const cidotprecision &,const dotprecision &) throw();
      friend inline cidotprecision operator |(const dotprecision &,const cidotprecision &) throw();
      friend inline cidotprecision operator &(const cidotprecision &,const dotprecision &) throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);
      friend inline cidotprecision operator &(const dotprecision &,const cidotprecision &) throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);
      
      friend inline cidotprecision & operator +=(cidotprecision &, const dotprecision &) throw();
      friend inline cidotprecision & operator -=(cidotprecision &, const dotprecision &) throw();
      friend inline cidotprecision & operator |=(cidotprecision &, const dotprecision &) throw();
      friend inline cidotprecision & operator &=(cidotprecision &, const dotprecision &) throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);
      
      // CID-CD

      friend inline cidotprecision operator +(const cidotprecision &,const cdotprecision &) throw();
      friend inline cidotprecision operator +(const cdotprecision &,const cidotprecision &) throw();
      friend inline cidotprecision operator -(const cidotprecision &,const cdotprecision &) throw();
      friend inline cidotprecision operator -(const cdotprecision &,const cidotprecision &) throw();
      friend inline cidotprecision operator |(const cidotprecision &,const cdotprecision &) throw();
      friend inline cidotprecision operator |(const cdotprecision &,const cidotprecision &) throw();
      friend inline cidotprecision operator &(const cidotprecision &,const cdotprecision &) throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);
      friend inline cidotprecision operator &(const cdotprecision &,const cidotprecision &) throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);
      

      friend inline cidotprecision & operator +=(cidotprecision &, const cdotprecision &) throw();
      friend inline cidotprecision & operator -=(cidotprecision &, const cdotprecision &) throw();
      friend inline cidotprecision & operator |=(cidotprecision &, const cdotprecision &) throw();
      friend inline cidotprecision & operator &=(cidotprecision &, const cdotprecision &) throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);
      
      // CID-ID

      friend inline cidotprecision operator +(const cidotprecision &,const idotprecision &) throw();
      friend inline cidotprecision operator +(const idotprecision &,const cidotprecision &) throw();
      friend inline cidotprecision operator -(const cidotprecision &,const idotprecision &) throw();
      friend inline cidotprecision operator -(const idotprecision &,const cidotprecision &) throw();
      friend inline cidotprecision operator |(const cidotprecision &,const idotprecision &) throw();
      friend inline cidotprecision operator |(const idotprecision &,const cidotprecision &) throw();
      friend inline cidotprecision operator &(const cidotprecision &,const idotprecision &) throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);
      friend inline cidotprecision operator &(const idotprecision &,const cidotprecision &) throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);
      
      friend inline cidotprecision & operator +=(cidotprecision &, const idotprecision &) throw();
      friend inline cidotprecision & operator -=(cidotprecision &, const idotprecision &) throw();
      friend inline cidotprecision & operator |=(cidotprecision &, const idotprecision &) throw();
      friend inline cidotprecision & operator &=(cidotprecision &, const idotprecision &) throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);

      // ---- Vergleichsop. ----
      friend inline bool operator !(const cidotprecision &) throw();
//             inline      operator void *() const throw();

      friend inline bool operator ==(const cidotprecision &,const cidotprecision &) throw();
      friend inline bool operator !=(const cidotprecision &,const cidotprecision &) throw();

      // CID-R
      
      friend inline bool operator== (const cidotprecision & a, const real & b)    throw();
      friend inline bool operator== (const real & a, const cidotprecision & b)    throw();
      friend inline bool operator!= (const cidotprecision & a, const real & b)    throw();
      friend inline bool operator!= (const real & a, const cidotprecision & b)    throw();

      // CID-C
      
      friend inline bool operator== (const cidotprecision & a, const complex & b)    throw();
      friend inline bool operator== (const complex & a, const cidotprecision & b)    throw();
      friend inline bool operator!= (const cidotprecision & a, const complex & b)    throw();
      friend inline bool operator!= (const complex & a, const cidotprecision & b)    throw();

      // CID-I
      
      friend inline bool operator== (const cidotprecision & a, const interval & b)    throw();
      friend inline bool operator== (const interval & a, const cidotprecision & b)    throw();
      friend inline bool operator!= (const cidotprecision & a, const interval & b)    throw();
      friend inline bool operator!= (const interval & a, const cidotprecision & b)    throw();

      // CID-CI
      
      friend inline bool operator== (const cidotprecision & a, const cinterval & b)    throw();
      friend inline bool operator== (const cinterval & a, const cidotprecision & b)    throw();
      friend inline bool operator!= (const cidotprecision & a, const cinterval & b)    throw();
      friend inline bool operator!= (const cinterval & a, const cidotprecision & b)    throw();
      
      // CID-D
      
      friend inline bool operator== (const cidotprecision & a, const dotprecision & b)    throw();
      friend inline bool operator== (const dotprecision & a, const cidotprecision & b)    throw();
      friend inline bool operator!= (const cidotprecision & a, const dotprecision & b)    throw();
      friend inline bool operator!= (const dotprecision & a, const cidotprecision & b)    throw();

      // CID-CD
      
      friend inline bool operator== (const cidotprecision & a, const cdotprecision & b)    throw();
      friend inline bool operator== (const cdotprecision & a, const cidotprecision & b)    throw();
      friend inline bool operator!= (const cidotprecision & a, const cdotprecision & b)    throw();
      friend inline bool operator!= (const cdotprecision & a, const cidotprecision & b)    throw();

      // CID-ID
      
      friend inline bool operator== (const cidotprecision & a, const idotprecision & b)    throw();
      friend inline bool operator== (const idotprecision & a, const cidotprecision & b)    throw();
      friend inline bool operator!= (const cidotprecision & a, const idotprecision & b)    throw();
      friend inline bool operator!= (const idotprecision & a, const cidotprecision & b)    throw();

      // ---- Set Operators ----
      friend inline bool operator  <(const cidotprecision &,const cidotprecision &) throw();
      friend inline bool operator  >(const cidotprecision &,const cidotprecision &) throw();
      friend inline bool operator <=(const cidotprecision &,const cidotprecision &) throw();
      friend inline bool operator >=(const cidotprecision &,const cidotprecision &) throw();

      // CID-R

      friend inline bool operator  <(const real &,const cidotprecision &) throw();
      friend inline bool operator  >(const real &,const cidotprecision &) throw();
      friend inline bool operator <=(const real &,const cidotprecision &) throw();
      friend inline bool operator >=(const real &,const cidotprecision &) throw();

      friend inline bool operator  <(const cidotprecision &,const real &) throw();
      friend inline bool operator  >(const cidotprecision &,const real &) throw();
      friend inline bool operator <=(const cidotprecision &,const real &) throw();
      friend inline bool operator >=(const cidotprecision &,const real &) throw();

      // CID-C

      friend inline bool operator  <(const complex &,const cidotprecision &) throw();
      friend inline bool operator  >(const complex &,const cidotprecision &) throw();
      friend inline bool operator <=(const complex &,const cidotprecision &) throw();
      friend inline bool operator >=(const complex &,const cidotprecision &) throw();

      friend inline bool operator  <(const cidotprecision &,const complex &) throw();
      friend inline bool operator  >(const cidotprecision &,const complex &) throw();
      friend inline bool operator <=(const cidotprecision &,const complex &) throw();
      friend inline bool operator >=(const cidotprecision &,const complex &) throw();

      // CID-I

      friend inline bool operator  <(const interval &,const cidotprecision &) throw();
      friend inline bool operator  >(const interval &,const cidotprecision &) throw();
      friend inline bool operator <=(const interval &,const cidotprecision &) throw();
      friend inline bool operator >=(const interval &,const cidotprecision &) throw();

      friend inline bool operator  <(const cidotprecision &,const interval &) throw();
      friend inline bool operator  >(const cidotprecision &,const interval &) throw();
      friend inline bool operator <=(const cidotprecision &,const interval &) throw();
      friend inline bool operator >=(const cidotprecision &,const interval &) throw();

      // CID-CI

      friend inline bool operator  <(const cinterval &,const cidotprecision &) throw();
      friend inline bool operator  >(const cinterval &,const cidotprecision &) throw();
      friend inline bool operator <=(const cinterval &,const cidotprecision &) throw();
      friend inline bool operator >=(const cinterval &,const cidotprecision &) throw();

      friend inline bool operator  <(const cidotprecision &,const cinterval &) throw();
      friend inline bool operator  >(const cidotprecision &,const cinterval &) throw();
      friend inline bool operator <=(const cidotprecision &,const cinterval &) throw();
      friend inline bool operator >=(const cidotprecision &,const cinterval &) throw();

      // CID-D

      friend inline bool operator  <(const dotprecision &,const cidotprecision &) throw();
      friend inline bool operator  >(const dotprecision &,const cidotprecision &) throw();
      friend inline bool operator <=(const dotprecision &,const cidotprecision &) throw();
      friend inline bool operator >=(const dotprecision &,const cidotprecision &) throw();

      friend inline bool operator  <(const cidotprecision &,const dotprecision &) throw();
      friend inline bool operator  >(const cidotprecision &,const dotprecision &) throw();
      friend inline bool operator <=(const cidotprecision &,const dotprecision &) throw();
      friend inline bool operator >=(const cidotprecision &,const dotprecision &) throw();

      // CID-CD

      friend inline bool operator  <(const cdotprecision &,const cidotprecision &) throw();
      friend inline bool operator  >(const cdotprecision &,const cidotprecision &) throw();
      friend inline bool operator <=(const cdotprecision &,const cidotprecision &) throw();
      friend inline bool operator >=(const cdotprecision &,const cidotprecision &) throw();

      friend inline bool operator  <(const cidotprecision &,const cdotprecision &) throw();
      friend inline bool operator  >(const cidotprecision &,const cdotprecision &) throw();
      friend inline bool operator <=(const cidotprecision &,const cdotprecision &) throw();
      friend inline bool operator >=(const cidotprecision &,const cdotprecision &) throw();

      // CID-ID

      friend inline bool operator  <(const idotprecision &,const cidotprecision &) throw();
      friend inline bool operator  >(const idotprecision &,const cidotprecision &) throw();
      friend inline bool operator <=(const idotprecision &,const cidotprecision &) throw();
      friend inline bool operator >=(const idotprecision &,const cidotprecision &) throw();

      friend inline bool operator  <(const cidotprecision &,const idotprecision &) throw();
      friend inline bool operator  >(const cidotprecision &,const idotprecision &) throw();
      friend inline bool operator <=(const cidotprecision &,const idotprecision &) throw();
      friend inline bool operator >=(const cidotprecision &,const idotprecision &) throw();


      // ---- Funktionen    ----
      
      friend inline cdotprecision   Inf(const cidotprecision&) throw();
      friend inline cdotprecision   Sup(const cidotprecision&) throw();
      friend inline cidotprecision& SetInf(cidotprecision&, const cdotprecision&) throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);
      friend inline cidotprecision& SetSup(cidotprecision&, const cdotprecision&) throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);
      friend inline cidotprecision& SetInf(cidotprecision&, const dotprecision&) throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);
      friend inline cidotprecision& SetSup(cidotprecision&, const dotprecision&) throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);
      friend inline cidotprecision& SetInf(cidotprecision&, const complex&) throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);
      friend inline cidotprecision& SetSup(cidotprecision&, const complex&) throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);
      friend inline cidotprecision& SetInf(cidotprecision&, const real&) throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);
      friend inline cidotprecision& SetSup(cidotprecision&, const real&) throw(ERROR_CIDOTPRECISION_EMPTY_INTERVAL);
      friend inline cidotprecision& UncheckedSetInf(cidotprecision&, const cdotprecision&) throw();
      friend inline cidotprecision& UncheckedSetSup(cidotprecision&, const cdotprecision&) throw();
      friend inline cidotprecision& UncheckedSetInf(cidotprecision&, const dotprecision&) throw();
      friend inline cidotprecision& UncheckedSetSup(cidotprecision&, const dotprecision&) throw();
      friend inline cidotprecision& UncheckedSetInf(cidotprecision&, const complex&) throw();
      friend inline cidotprecision& UncheckedSetSup(cidotprecision&, const complex&) throw();
      friend inline cidotprecision& UncheckedSetInf(cidotprecision&, const real&) throw();
      friend inline cidotprecision& UncheckedSetSup(cidotprecision&, const real&) throw();
   
      friend inline idotprecision   Re(const cidotprecision &) throw();
      friend inline idotprecision   Im(const cidotprecision &) throw();
      
      friend inline const dotprecision & InfRe(const cidotprecision &) throw();
      friend inline const dotprecision & InfIm(const cidotprecision &) throw();
      friend inline const dotprecision & SupRe(const cidotprecision &) throw();
      friend inline const dotprecision & SupIm(const cidotprecision &) throw(); 
      
      friend inline       dotprecision & InfRe(cidotprecision &) throw();
      friend inline       dotprecision & InfIm(cidotprecision &) throw();
      friend inline       dotprecision & SupRe(cidotprecision &) throw();
      friend inline       dotprecision & SupIm(cidotprecision &) throw(); 
      
      friend inline cidotprecision& SetRe(cidotprecision&, const idotprecision&) throw();
      friend inline cidotprecision& SetIm(cidotprecision&, const idotprecision&) throw();
      friend inline cidotprecision& SetRe(cidotprecision&, const dotprecision&) throw();
      friend inline cidotprecision& SetIm(cidotprecision&, const dotprecision&) throw();
      friend inline cidotprecision& SetRe(cidotprecision&, const interval&) throw();
      friend inline cidotprecision& SetIm(cidotprecision&, const interval&) throw();
      friend inline cidotprecision& SetRe(cidotprecision&, const real&) throw();
      friend inline cidotprecision& SetIm(cidotprecision&, const real&) throw();   

      
      friend inline void rnd(const cidotprecision &,cinterval &) throw();
      friend inline cinterval rnd(const cidotprecision &) throw();
      
      friend        void accumulate  (cidotprecision&, const cinterval&, const cinterval&) throw();

      friend inline void accumulate  (cidotprecision&, const cinterval&, const interval&) throw();
      friend inline void accumulate  (cidotprecision&, const cinterval&, const complex&) throw();
      friend inline void accumulate  (cidotprecision&, const cinterval&, const real&) throw();
      
      friend inline void accumulate  (cidotprecision&, const interval &,const cinterval &) throw();
      friend inline void accumulate  (cidotprecision&, const complex &,const cinterval &) throw();
      friend inline void accumulate  (cidotprecision&, const real &,const cinterval&) throw();
      
      friend inline void accumulate  (cidotprecision&, const complex &,const interval &) throw();
      friend inline void accumulate  (cidotprecision&, const interval &,const complex &) throw();

      friend inline void accumulate  (cidotprecision&, const interval &,const interval &) throw();
      friend inline void accumulate  (cidotprecision&, const interval &,const real &) throw();
      friend inline void accumulate  (cidotprecision&, const real &,const interval &) throw();

      friend inline void accumulate  (cidotprecision&, const complex &,const complex &) throw();
      friend inline void accumulate  (cidotprecision&, const real &,const complex &) throw();
      friend inline void accumulate  (cidotprecision&, const complex &,const real &) throw();
      
      friend inline void accumulate  (cidotprecision&, const real &,const real &) throw();
};

//----------------------------------------------------------------------
// global verfuegbare CIDotprecision Akku's   

#define MAXCIDOTAKKU     (MAXDOTAKKU / 2)
extern cidotprecision cidotakku[MAXCIDOTAKKU];
  
//----------------------------------------------------------------------
} // namespace cxsc 

#include "cidot.inl"

#endif 

