/* DiaCanvas -- A technical drawing canvas.
 * Copyright (C) 1999, Arjan Molenaar
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/* diabaseline.h
 * -------------
 * Base class for line like objects (lines, orthogonal lines, ...).
 * It draws lines between handles (so handles are used to mark line segments).
 *
 * NOTE: the DATA field of a connection point has a pointer to the first
 *       handle that is part of the line segment the CP is laying on.
 */

#ifndef __DIA_BASE_LINE_H__
#define __DIA_BASE_LINE_H__

#include <diacanvas/diaobject.h>

DIA_OPEN

#define DIA_BASE_LINE(x) ((DiaBaseLine*) x)

#define DIA_OBJECT_HANDLE_FIRST(obj) DIA_OBJECT_GET_HANDLE (obj, 0)
#define DIA_OBJECT_HANDLE_LAST(obj)  DIA_OBJECT_GET_HANDLE (obj, \
							    obj->handles->len - 1)

typedef struct _DiaBaseLine DiaBaseLine;
typedef struct _DiaBaseLineCP DiaBaseLineCP;

typedef enum
{
  /* bit 0 1 and are used by DIA_OBJECT_STATE_* */
  DIA_BASE_LINE_MOVE = 1 << 4,
  DIA_BASE_LINE_MOVE_HANDLE = 1 << 5,
  DIA_BASE_LINE_NEW = 1 << 6
} DiaBaseLineFlags;

/* This struct is connected to a CP */
struct _DiaBaseLineCP
{
  DiaHandle *handle; /* first handle of the line segment the CP is on. */
  gfloat     factor; /* This is the factor (cp - left) / (right - left). */
};

struct _DiaBaseLine
{
  DiaObject     object;
  
  DiaLineCaps   line_caps;
  DiaLineJoin   line_join;
  DiaLineStyle  line_style;
  gfloat        line_width;
  DiaColor      color;
  
  DiaHandle    *handle_moving;
};

DiaObjectType* dia_base_line_get_type ();

void dia_base_line_init (DiaBaseLine *base_line);

/* Note that the creation tool will use the first handle
 * (=the last added handle) to resize the added object
 */
DiaObject* dia_base_line_new (Point *pos);
     
void dia_base_line_insert_point (DiaBaseLine* line, Point* new_point,
				 Point *pred);

void dia_base_line_remove_point (DiaBaseLine *line, gint index);

/* Protected: only for use in children of DiaBaseLine. */
gfloat dia_base_line_distance_real (DiaObject* obj, Point* point,
				    Point *point_on_line);

void dia_base_line_update_connection_points (DiaBaseLine *line,
					     DiaHandle *handle);

gfloat dia_base_line_cp_calc_factor (Point *p1, Point *p2, Point *cp_point);

DIA_CLOSE

#endif /* __DIA_BASE_LINE_H__ */



