/* DiaCanvas -- A technical drawing canvas.
 * Copyright (C) 1999, Arjan Molenaar
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/* diabaseelement.h
 * ----------------
 * Base class for element like objects (boxes, UML classes, pictures, etc).
 * element typically have eight handles around them to allow resizes and show
 * that the object is selected.
 *
 * This class is "abstract"! it is not wise to create direct instances of this
 * object type. This class implements basic behaviour (like resizing) and some
 * basic things (like the eight handles around).
 */

#ifndef __DIA_BASE_ELEMENT_H__
#define __DIA_BASE_ELEMENT_H__

#include <diacanvas/diaobject.h> /* parent object */

DIA_OPEN

#define DIA_BASE_ELEMENT(x) ((DiaBaseElement*) x)

typedef struct _DiaBaseElement DiaBaseElement;
typedef struct _DiaBaseElementCP DiaBaseElementCP;

/* like this:  3--4--5
 *             |     |
 *             2     6
 *             |     |
 *             1--0--7
 * Now the placement tool will automatically pick the lower right corner
 * for its drag action.
 */
typedef enum
{
  DIA_BASE_ELEMENT_B  = 0,
  DIA_BASE_ELEMENT_BL = 1,
  DIA_BASE_ELEMENT_L  = 2,
  DIA_BASE_ELEMENT_TL = 3,
  DIA_BASE_ELEMENT_T  = 4,
  DIA_BASE_ELEMENT_TR = 5,
  DIA_BASE_ELEMENT_R  = 6,
  DIA_BASE_ELEMENT_BR = 7,
} DiaBaseElementDirection;

typedef enum
{
  /* bit 0-3 and are used by DIA_OBJECT_STATE_* */
  DIA_BASE_ELEMENT_MOVE = 1 << 4,
  DIA_BASE_ELEMENT_MOVE_HANDLE = 1 << 5,
  DIA_BASE_ELEMENT_NEW = 1 << 6 /* element is just added to the canvas */
} DiaBaseElementFlags;

/* This struct is connected to a CP */
struct _DiaBaseElementCP
{
  DiaBaseElementDirection direction;
  gfloat     factor; /* This is the factor (cp - left) / (right - left). */
  /* DIA_BASE_ELEMENT_TL for CP's on top,
   * DIA_BASE_ELEMENT_TR for CP's on right,
   * DIA_BASE_ELEMENT_BR for CP's on the bottom side and
   * DIA_BASE_ELEMENT_BL for CP's on the left.
   */
};


struct _DiaBaseElement
{
  DiaObject     object;

  //DiaHandle    *handle_moving;
  DiaBaseElementDirection direction;

  /* this is the real size of the element. (the bounding box can be larger
   * due to the thinkness of lines for example.) */
  Rectangle        size;
};

DiaObjectType* dia_base_element_get_type ();

void dia_base_element_init (DiaBaseElement *element);

void dia_base_element_resize (DiaBaseElement *element, gfloat width, gfloat height);

void dia_base_element_calc_size (DiaBaseElement *element);

DIA_CLOSE

#endif /* __DIA_BASE_ELEMENT_H__ */




