;;; epix.el --- A major mode for working with ePiX files.

;; Copyright (C) 2002 Jay Belanger

;; Author: Jay Belanger
;; Maintainer: Jay Belanger <belanger@truman.edu>

;; $Revision: 1.6 $
;; $Date: 2002/12/21 03:04:25 $
;; Keywords: epix

;; This program is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License as
;; published by the Free Software Foundation; either version 2 of
;; the License, or (at your option) any later version.
;;          
;; This program is distributed in the hope that it will be
;; useful, but WITHOUT ANY WARRANTY; without even the implied
;; warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
;; PURPOSE.  See the GNU General Public License for more details.
;;          
;; You should have received a copy of the GNU General Public
;; License along with this program; if not, write to the Free
;; Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
;; MA 02111-1307 USA
;;
;;
;; Please send suggestions and bug reports to <belanger@truman.edu>. 

;;; Commentary:

;; Quick intro

;; epix-mode is an extension of c++-mode with some extra commands for 
;; editing ePiX files.  The commands are
;;  C-cC-x    run epix on the current file (x for epiX, I guess)
;;  C-cC-l    run elaps on the current file (l for eLaps)
;; The output of the run can be seen by
;;  C-cC-r    show the output buffer  (r for Results)
;; and in case of errors, they can be jumped to, in sequence, with
;;  C-c`      go to the next error
;; The eps file created by elaps can be viewed with
;;  C-cC-v    view the eps file  (v for View)
;; An unwanted ePiX process can be killed with
;;  C-cC-k
;; A reminder of some ePiX commands will be given with 
;;  C-cC-h    get help on ePiX
;;
;; To install, put both epix.el and epix.info in the emacs load path.
;; Adding the line
;;   (autoload 'epix-mode "epix" "ePiX editing mode" t)
;; to your .emacs file will ensure that the command is always available.
;; Adding the line
;;   (setq auto-mode-alist (cons '("\\.xp" . epix-mode) auto-mode-alist))
;; will ensure that any file ending in .xp will be recognized as an ePiX
;; file, and started in epix-mode.
;;
;; Some variables that the user may wish to change (via customization
;; or otherwise) are
;;   epix-postscript-viewer  (default "gv")
;;     The command used to view postscript files.
;;   epix-mark-files-as-epix (default nil)
;;     Setting this to t will ensure that any file in ePiX mode
;;     will be marked with a /* -*-ePiX-* */, so that next time it
;;     is opened, it will be put in epix-mode.  The command
;;     epix-mark-file-as-epix will mark the current buffer.
;;  epix-insert-template-in-empty-buffer (default nil)
;;     Setting this to t will cause any empty buffer which is put in
;;     ePiX mode to have a skeleton of commands inserted into it.
;;     The skeleton can be inserted at any time with the command
;;     epix-insert-template
;;  epix-template
;;     This is the skeleton which will inserted.
;;     By default, it is
;;       #include \"epix.h\"
;;
;;       main() {
;;         bounding_box(P(,),P(,));
;;         unitlength(\"\");
;;         picture(P(,));
;;
;;         begin();
;;
;;         end();
;;       }


;;; Customization

(defgroup epix nil
  "ePiX mode"
  :prefix "epix-"
  :tag    "ePiX")

(defcustom epix-mark-files-as-epix nil
  "Non-nil means to make sure that any ePiX file is marked as such."
  :group 'epix
  :type 'boolean)

(defcustom epix-template 
"#include \"epix.h\"
using namespace std;
using namespace ePiX;

main() {
bounding_box(P(,),P(,));
unitlength(\"\");
picture(P(,));

begin();

end();
}
"
  "The information to enter into an empty ePiX buffer."
  :group 'epix
  :type 'string)

(defcustom epix-insert-template-in-empty-buffer nil
  "Non-nil means to insert the template into an empty ePiX buffer."
  :group 'epix
  :type 'boolean)

(defcustom epix-command-name "epix"
  "The name of the ePiX program."
  :group 'epix
  :type 'string)

(defcustom epix-command-args nil
  "Arguments to pass to epix"
  :group 'epix
  :type 'string)

(defcustom epix-elaps-command-name "elaps"
  "The name of the elaps program."
  :group 'epix
  :type 'string)

(defcustom epix-elaps-command-args nil
  "Arguments to pass to elaps"
  :group 'epix
  :type 'string)

(defcustom epix-postscript-viewer "gv"
  "The name of the program used to view postscript files."
  :group 'epix
  :type 'string)

;;; Some utility variables

(defvar epix-output-buffer nil)
(make-variable-buffer-local 'epix-output-buffer)

(defvar epix-error-point nil)
(make-variable-buffer-local 'epix-error-point)

(defvar epix-info-file (locate-library "epix.info"))

;;; Functions to run epix and friends

(defun epix-check-process ()
  "See if there is a process associated with the output buffer.
If there is and it is running, offer to kill it."
  (let ((epp)
        (epix-proc (get-buffer-process epix-output-buffer)))
    (if epix-proc (setq epp t) (setq epp nil))
    (if (and 
         epix-proc
         (y-or-n-p "There is an ePiX process running.  Kill it? "))
        (progn
          (kill-process epix-proc)
          (setq epp nil)))
    epp))

(defun epix-kill-process ()
  "Kill any epix process running."
  (interactive)
  (let ((epix-proc (get-buffer-process epix-output-buffer)))
    (if (and
         epix-proc
         (y-or-n-p "Really kill ePiX process? "))
        (kill-process epix-proc))))
         

(defun epix-check-buffer ()
  "See if the buffer has been modified since it was last saved.
If it has been, offer to save it."
  (if (and
       (buffer-modified-p)
       (y-or-n-p (concat "Save file " (buffer-file-name) "?")))
      (save-buffer)))

(defun epix-run-command (command file &optional args)
  "Create a process to run COMMAND on FILE.
Return the new process."
  (let ((buffer epix-output-buffer))
    (epix-check-buffer)
    (unless (epix-check-process)
      (set-buffer (get-buffer-create buffer))
      (setq epix-error-point nil)
      (erase-buffer)
      (insert "Running `" command "' on `" file "\n")
      (message "Type `C-c C-r' to display results of compilation.")
      (let ((process)
            (cmd))
        (if args
            (setq cmd 
                  (append 
                   (list 'start-process "ePiX" buffer "/bin/sh" command)
                   (split-string args)
                   (list file)))
          (setq cmd (list 'start-process "ePiX" buffer "/bin/sh" command file)))
        (setq process (eval cmd))
        (set-process-filter process 'epix-command-filter)
        (set-process-sentinel process 'epix-command-sentinel)
        (set-marker (process-mark process) (point-max))
        process))))
  
(defun epix-command-sentinel (process event)
  (if (string-match "finished" event)
      (message "ePiX process finished")))

(defun epix-command-filter (process string)
  "Filter to process normal output."
  (save-excursion
    (set-buffer (process-buffer process))
    (save-excursion
      (goto-char (process-mark process))
      (insert-before-markers string)
      (set-marker (process-mark process) (point))))
  (if (string-match "Compilation failed" string)
      (message (concat "ePiX errors in `" (buffer-name)
                       "'. Use C-c ` to display."))))
(defun epix-run-epix ()
  "Run epix on the current file."
  (interactive)
  (let ((file (file-name-nondirectory (buffer-file-name))))
    (epix-run-command epix-command-name file epix-command-args)))

(defun epix-run-elaps ()
  "Run elaps on the current file."
  (interactive)
  (let ((file (file-name-nondirectory (buffer-file-name))))
    (epix-run-command epix-elaps-command-name file epix-elaps-command-args)))

(defun epix-view-eps ()
  "View the eps output of the file."
  (interactive)
  (let ((filename (file-name-nondirectory (buffer-file-name)))
        (file nil))
    (if (file-name-extension filename)
        (setq file (concat (file-name-sans-extension filename) ".eps")))
    (unless (file-exists-p file)
      (setq file (concat filename ".eps")))
    (message file)
    (if (file-exists-p file)
        (call-process epix-postscript-viewer nil epix-output-buffer nil file)
      (message (concat "No file " file ".  Run elaps first.")))))

;;; Dealing with output

(defun epix-show-output-buffer ()
  "Show the epix output buffer."
  (interactive)
  (let ((buf (current-buffer)))
    (if (get-buffer epix-output-buffer)
        (progn
          (pop-to-buffer epix-output-buffer t)
          (bury-buffer buf)
	  (goto-char (point-max))
	  (recenter (/ (window-height) 2))
	  (pop-to-buffer buf))
      (message "No output buffer."))))

(defun epix-find-error (arg)
  "Go to the next ePiX error."
  (interactive "P")
  (if arg 
      (save-excursion
        (set-buffer epix-output-buffer)
        (setq epix-error-point nil)))
  (let ((ln)
        (col)
        (epix-error-end)
        (buf (current-buffer)))
    (switch-to-buffer-other-window epix-output-buffer)
    (widen)
    (if epix-error-point
        (goto-char epix-error-point)
      (goto-char (point-min)))
    (if (re-search-forward ":\\([0-9]+\\):\\([0-9]+\\):" (point-max) t)
        (progn
          (setq epix-error-point (point))
          (setq ln (string-to-int (match-string 1)))
          (setq col (string-to-int (match-string 2)))
          (save-excursion
            (if (re-search-forward ":[0-9]+:[0-9]+:" (point-max) t)
                (setq epix-error-end (line-beginning-position))
              (if (search-forward "Compilation failed" (point-max) t)
                  (setq epix-error-end (line-beginning-position))
                (setq epix-error-end (point-max)))))
          (recenter)
          (beginning-of-line)
          (if (looking-at "epix: Compiling...")
              (forward-char (length "epix: Compiling...")))
          (narrow-to-region (point) epix-error-end)
          (switch-to-buffer-other-window buf)
          (goto-line ln)
          (move-to-column col))
      (switch-to-buffer-other-window buf)
      (message "No more errors."))))


;;; Auxiliary functions

(defun epix-mark-file-as-epix ()
  "Mark the file as an ePiX buffer.
The next time the file is loaded, it will then be in ePiX mode"
  (interactive)
  (save-excursion
    (goto-char (point-min))
    (unless (looking-at ".*-\\*-ePiX-\\*-")
      (insert "/* -*-ePiX-*- */\n"))))

(defun epix-insert-template ()
  "Insert a template."
  (interactive)
  (let ((beg (point)))
    (insert epix-template)
    (indent-region beg (point) nil)
    (goto-char beg)
    (search-forward "P(")))

(defun epix-help ()
  "Read an ePiX info file"
  (interactive)
  (info-other-window epix-info-file))

;;; The ePiX mode

(define-derived-mode epix-mode c++-mode "ePiX"
  "ePiX mode is a major mode for editing ePiX files.
It is C++ mode with a couple of extra commands:

Run epix on the file being visited:  C-cC-x
Run elaps on the file being visited: C-cC-l
View the postscript output:          C-cC-v
Get help on ePiX:                    C-cC-h

In the case of errors:

Go to the (next) error:              C-c`

SUMMARY OF EPIX COMMANDS
------------------------

Preamble
--------
bounding_box(P(x_min,y_min),P(x_max,y_max));
picture(P(h_size,v_size));
unitlength(\"length\");

The actual picture commands need to be between
begin();
and
end();

Line styles
-----------
Widths:  plain (default), bold
Styles:  solid (default), dashed, dotted

Lines
-----
line(P(a,b),P(c,d));
Line(P(a,b),P(c,d));
Line(P(a,b),m);

The line goes from one point to the other,
the Line is that part of the entire line which lies inside the
bounding box.

Polygons
--------
triangle(P(a,b),P(c,d),P(e,f));
rect(P(a,b),P(c,d));
swatch(P(a,b),P(c,d));

Axes
----
h_axis(P(a,b),P(c,d),n);
v_axis(P(a,b),P(c,d),n);
h_axis_labels(P(a,b),P(c,d),n,P(u,v));
v_axis_labels(P(a,b),P(c,d),n,P(u,v));
h_axis_masklabels(P(a,b),P(c,d),n,P(u,v));
v_axis_masklabels(P(a,b),P(c,d),n,P(u,v));

The axis commands will put n+1 evenly spaced tick marks on the line. 
The labeling commands will put n+1 evenly spaced labels,
masklabels will put the label on an opaque white rectangle.

Points
------
spot(P(a,b)); dot(P(a,b)); ddot(P(a,b));
circ(P(a,b));
ring(P(a,b),r);

spot, dot and ddot are various sized (in decreasing order) dots.
A circ is a small circle with white interior.
A ring is a small circle with transparent interior, whose radius is r
true points.

Arrows
------
arrow(P(a,b),P(c,d));
dart(P(a,b),P(c,d));
boldarrow(P(a,b),P(c,d));
bolddart(P(a,b),P(c,d));

A dart is a small arrow.

Labels
------
label(P(a,b),P(u,v),\"label\");
label(P(a,b),\"label\");
label(P(a,b),P(u,v),\"label\",posn);
masklabel(P(a,b),P(u,v),\"label\");
masklabel(P(a,b),\"label\");
masklabel(P(a,b),P(u,v),\"label\",posn);

These commands put a label at the point P(a,b).
P(u,v) is an offset (in true points).
posn will specify the position of the label relative to the basepoint,
and can be c(enter), l(eft), r(ight), t(op), b(ottom) or an
appropriate pair.

Curves
------
ellipse(P(a,b),P(u,v));
ellipse_top/bottom/left/right(P(a,b),P(u,v));
ellipse_half(P(a,b),P(u,v),theta);

The point P(a,b) will be the center of the ellipse, whose axes are
determined by P(u,v).
ellipse_half will be rotated counterclockwise by theta degrees.

Arcs
----
arc(P(a,b),r,theta1,theta2);
arc_arrow(P(a,b),r,theta1,theta2);

The angles are measured in radians.

Splines
-------
spline(P(a,b),P(c,d),P(e,f));
spline(P(a,b),P(c,d),P(e,f),P(g,h));

Plotting
--------
plot(f,a,b,n);
clipplot(f,a,b,n);

These plot the function f from a to b using n+1 evenly spaced points. 
clipplot will clip the graph to lie inside the bounding box.
If f is a pair, or even f1,f2, then these will be parametric plots.

Calc plotting
-------------
plot_deriv(f,a,b,n);
plot_int(f,a,b,n);
plot_int(f,x0,a,b,n);

plot_deriv plots f'.
plot_int plots int_a^x f (or if x0 is given, int_x0^x f).

Tangents
--------
tan_line(f,t);
envelope(f,t_min,t_max,n);
tan_field(f,t_min,t_max,n);

tan_line will plot the tangent Line to the graph.
envelope will plot n+1 tangent Lines.
tan_field will plot n+1 tangent vectors.
Here, as above, f can be replaced by a pair or f1,f2.

Vector fields
-------------
slope_field(F,P(a,b),P(c,d),n1,n2);
dart_field(F,P(a,b),P(c,d),n1,n2);
vector_field(F,P(a,b),P(c,d),n1,n2);

The slope_field elements will have fixed length, no arrows.
The dart_field elements will have fixed length, small arrows.
The vector_field elements will have true length.

Fractals
--------
const int seed[] = {N, k1, k2, k3, ... , kn};
fractal(P(a,b),P(c,d),D,seed);

seed determines a path made up of equal space line segments, each of
which can point in the direction 2 pi k/N.  Each integer after N in seed
will specify the direction of the next segment.
fractal will recursively replace each segment in seed by a copy of the
original, up to a depth of D, and draw it from P(a,b) to P(c,d).

\\{epix-mode-map}"
  (interactive)
  (setq font-lock-defaults
        '((c++-font-lock-keywords 
           c++-font-lock-keywords-1
           c++-font-lock-keywords-2 
           c++-font-lock-keywords-3)
          nil nil ((?_ . "w")) beginning-of-defun
          (font-lock-mark-block-function . mark-defun)))
  (setq epix-output-buffer (concat "*" (buffer-file-name) " output*"))
  (when (and epix-insert-template-in-empty-buffer
           (= (point-min) (point-max)))
      (epix-insert-template)
      (goto-char (point-min)))
  (if epix-mark-files-as-epix
      (epix-mark-file-as-epix))
  (run-hooks 'epix-mode-hooks))

(define-key epix-mode-map "\C-c\C-k" 'epix-kill-process)
(define-key epix-mode-map "\C-c\C-x" 'epix-run-epix)
(define-key epix-mode-map "\C-c\C-l" 'epix-run-elaps)
(define-key epix-mode-map "\C-c\C-r" 'epix-show-output-buffer)
(define-key epix-mode-map "\C-c\C-v" 'epix-view-eps)
(define-key epix-mode-map "\C-c\C-h" 'epix-help)
(define-key epix-mode-map "\C-c`" 'epix-find-error)

;; Menu
(easy-menu-define epix-mode-menu epix-mode-map
  "ePiX mode menu"
  '("ePiX"
    ["Run ePiX" epix-run-epix t]
    ["Run elaps" epix-run-elaps t]
    ["View the eps file" epix-view-eps t]
    ["Get help on ePiX" epix-help t]
    ["Show the output buffer" epix-show-output-buffer t]))

(provide 'epix)
