//----------------------------------------------------------------------------
//
// Module:      JFrameDocumentHandler.java      
//
// Description: Handles JFrame elements
//
// FESI Copyright (c) Jean-Marc Lugrin, 2000
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.

// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//----------------------------------------------------------------------------

package FesiForm;

import org.xml.sax.*;
import FESI.jslib.*;

import javax.swing.*;
import java.awt.*;
import java.util.Vector;


public class JFrameDocumentHandler extends DefaultDocumentHandler 
                                   implements DocumentHandler {

    DocumentHandler previousDocumentHandler;
    String script = null;
    JFrame frame = null;
    String containerName;
    
    public JFrameDocumentHandler(String cn,
                                    DocumentHandler pdh, 
                                    GuiContext guiContext, 
                                    boolean trace) {
        super(guiContext, trace);
        this.containerName = cn;
        this.previousDocumentHandler = pdh;
        // Validate assumption
        if (!cn.equals("JFrame")) throw new IllegalArgumentException("** Unexpected container name for JFrame: " + cn);
    }
  
      
    // Receive notification of the beginning of an element.
    public void startElement(String name, AttributeList atts) throws SAXException {
        if (trace) System.out.println("JFRAME START_ELEMENT: <" + name + ">, " + 
                                         atts.getLength() + " attribute(s)");
        if (name.equals("JFrame")) {
            String id = getAttribute("id", atts);
            if (id==null) {
                throw new SAXException("Missing ID in JFrame");
            } else {
                frame = new JFrame();
                frame = (JFrame) createComponent(JFrame.class, atts);
                guiContext.putId(id, frame);
                processComponentAttributes(frame, atts);
                processTITLE(frame, atts);
                processLAYOUT(frame.getContentPane(), atts);
                processDEFAULTCLOSEOPERATION(frame, atts);
                script = saveScript(frame, atts);
                DocumentHandler dh = 
                    new ContentDocumentHandler("JFrame", this, frame, frame.getContentPane(), guiContext, trace, -1);
                guiContext.parser.setDocumentHandler(dh);
                dh.startElement(name, atts);
            }
        } else {
            throw new SAXException("Unexpected ELEMENT in frame ignored: " + name);
        }
    }

    // Receive notification of the end of an element. 
    public void endElement(String name) throws SAXException {
        if (trace) System.out.println("JFRAME END_ELEMENT: </" + name + ">");
        if (name.equals("JFrame")) {  
            processScript(frame, script);
            previousDocumentHandler.endElement(name);
        } else {
            throw new SAXException("Unexpected ELEMENT end in frame ignored: " + name);
        }
    }
    
     protected String processTITLE(JFrame cp, AttributeList atts) {
         String title = getAttribute("title", atts);
         if (title != null) {
             if (trace) System.out.println("  TITLE: " + title);
             cp.setTitle(title); 
         }
         return title;
     }


     protected int processDEFAULTCLOSEOPERATION(JFrame cp, AttributeList atts)  throws SAXException {
         String op = getAttribute("defaultCloseOperation", atts);
         int opCode = -1;
         if (op != null) {
             if (trace) System.out.println("  DEFAULTCLOSEOPERATION: " + op);
             opCode = operationByName(op); 
             cp.setDefaultCloseOperation(opCode);
         }
         return opCode;
     }
       
     int operationByName(String opName) throws SAXException {
         String cbn = opName.toLowerCase();
         if (cbn.equals("nothing")) {
             return JFrame.DO_NOTHING_ON_CLOSE;
         } else if (cbn.equals("hide")) { 
             return JFrame.HIDE_ON_CLOSE;
         } else if (cbn.equals("dispose")) {
             return JFrame.DISPOSE_ON_CLOSE;
         }
         throw new SAXException("Unrecognized on close operation: " + opName);
     }     

}