/*en    Laurent Constantin's network library (lcrzo)
                 lcrzo_conf module

  Functions herein obtains configuration of the local computer.
*/
/*fr    Bibliotheque reseau de Laurent Constantin (lcrzo)
                 Module lcrzo_conf

  Les fonctions presentes dans ce module permettent d'obtenir des
informations sur la machine locale.
*/


/*-------------------------------------------------------------*/
/*EN************************************************************
 * Note about lcrzo_conf :                                     *
 * Configuration obtained by this module is a copy of the      *
 * configuration of the local computer. Then the user can      *
 * modify it using "xx_add" and "xx_del" functions : the       *
 * configuration of lcrzo is modified, but not the             *
 * configuration of the computer.                              *
 * For examples the configuration of the local computer can    *
 * be (obtained with lcrzo_conf_print) :                       */
/*FR************************************************************
 * Note sur lcrzo_conf :                                       *
 * La configuration obtenue par ce module est une copie de la  *
 * configuration de la machine locale. L'utilisateur peut par  *
 * la suite la modifier avec les fonctions "xx_add" et         *
 * "xx_del" : la configuration de lcrzo est changee, mais pas  *
 * celle de l'ordinateur.                                      *
 * Par exemple la configuration de la machine locale peut etre *
 * (obtenu par lcrzo_conf_print) :                             */
/*
Devices
  device  ethernet          ip          /netmask        mtu
  lo      loopback          127.0.0.1   /255.0.0.0      3924 up
  eth0    00:00:33:AC:C2:24 192.168.1.1 /255.255.255.0  1500 up
  eth0:1  00:00:33:AC:C2:24 192.168.2.1 /255.255.255.0  1500 up,alias
  eth1    00:00:43:BC:D2:34 192.168.3.1 /255.255.255.0  1500 up
Arp
  eth0    00:00:33:AC:C2:24 192.168.1.1   (permanent)
  eth0:1  00:00:33:AC:C2:24 192.168.2.1   (permanent)
  eth1    00:00:43:BC:D2:34 192.168.3.1   (permanent)
  eth0    12:34:56:78:90:ab 192.168.1.254 (-1s)
Routes
  device  destination  /netmask         ip_source     gateway
  lo      127.0.0.1    /255.255.255.255 local_device                 0,up
  eth0    192.168.1.1  /255.255.255.255 local_device                 0,up
  eth0:1  192.168.2.1  /255.255.255.255 local_device                 0,up
  eth1    192.168.3.1  /255.255.255.255 local_device                 0,up
  eth0    192.168.1.0  /255.255.255.0   192.168.1.1                  0,up
  eth0:1  192.168.2.0  /255.255.255.0   192.168.2.1                  0,up
  eth1    192.168.3.0  /255.255.255.0   192.168.3.1                  0,up
  lo      127.0.0.0    /255.0.0.0       127.0.0.1                    0,up
  eth0    0.0.0.0      /0.0.0.0         192.168.1.1   192.168.1.254  0,up
 ***************************************************************/

/*EN************************************************************
 * Functions lcrzo_conf_xxx_del_system permit to remove        *
 * unwanted configuration of the local system. For example,    *
 *  before :                                                   *
 *    device eth0...                                           *
 *    device eth1...                                           *
 *  after lcrzo_conf_device_del_system("eth0", ...)            *
 *    device eth1...                                           *
 *                                                             *
 * Functions lcrzo_conf_xxx_add_user permit to add             *
 * a user configuration which can be prioritary (or not) over  *
 * system configuration. For example,                          *
 *  before :                                                   *
 *    device eth0 a...                                         *
 *    device eth1 b...                                         *
 *  after lcrzo_conf_devices_add_user(TRUE, "eth0", c...) and  *
 *  lcrzo_conf_devices_add_user(FALSE, "eth1", d...)           *
 *    device eth0 c...                                         *
 *    device eth1 b... (do not see user'choice because FALSE)  *
 *                                                             *
 * Functions lcrzo_conf_xxx_del_user permit to remove an       *
 * user added configuration.                                   *
 ***************************************************************/
/*FR************************************************************
 * Les fonctions lcrzo_conf_xxx_del_system permettent          *
 * d'effacer les configurations non voulues du systeme local.  *
 * Par exemple,                                                *
 *  avant :                                                    *
 *    device eth0...                                           *
 *    device eth1...                                           *
 *  apres lcrzo_conf_device_del_system("eth0", ...)            *
 *    device eth1...                                           *
 *                                                             *
 * Les fonctions lcrzo_conf_xxx_add_user permettent d'ajouter  *
 * une configuration d'utilisateur qui peut etre prioritaire   *
 * (ou non) sur la configuration systeme. Par exemple,         *
 *  avant :                                                    *
 *    device eth0 a...                                         *
 *    device eth1 b...                                         *
 *  apres lcrzo_conf_devices_add_user(TRUE, "eth0", c...) et   *
 *  lcrzo_conf_devices_add_user(FALSE, "eth1", d...)           *
 *    device eth0 c...                                         *
 *    device eth1 b... (ne voit pas le choix de l'utilisateur) *
 *                                                             *
 * Les fonctions lcrzo_conf_xxx_del_user permettent d'enlever  *
 * une configuration ajoutee par l'utilisateur.                *
 ***************************************************************/

/*-------------------------------------------------------------*/
/*en Name : lcrzo_conf_fprint
   Description :
     Print the configuration of lcrzo.
   Input parameter(s) :
     *pf : FILE where the function prints
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_conf_fprint
   Description :
     Affiche la configuration de lcrzo.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_conf_fprint(LCRZOFILE *pf);
#define lcrzo_conf_print() lcrzo_conf_fprint(stdout)


/*-------------------------------------------------------------*/
/*en Hardware types */
/*fr nom des devices physiques */
#define LCRZO_DEVICE_HWTYPE_UNKNOWN      0
#define LCRZO_DEVICE_HWTYPE_UNKNOWN_txt  "unknown"
#define LCRZO_DEVICE_HWTYPE_ETHER        1
#define LCRZO_DEVICE_HWTYPE_ETHER_txt    "ethernet"
#define LCRZO_DEVICE_HWTYPE_LOOPBACK     2
#define LCRZO_DEVICE_HWTYPE_LOOPBACK_txt "loopback"
#define LCRZO_DEVICE_HWTYPE_PPP          3
#define LCRZO_DEVICE_HWTYPE_PPP_txt      "ppp"
#define LCRZO_DEVICE_HWTYPE_PLIP         4
#define LCRZO_DEVICE_HWTYPE_PLIP_txt     "plip"
#define LCRZO_DEVICE_HWTYPE_SLIP         5
#define LCRZO_DEVICE_HWTYPE_SLIP_txt     "slip"

/*-------------------------------------------------------------*/
/*en Name : lcrzo_conf_devices_add_user
   Description :
     Add a device to the configuration of lcrzo.
   Input parameter(s) :
     priorityoversystem : if this configuration is prioritary
                          over the system configuration
     device : device name
     etha : associated Ethernet address
     ipl : associated IP address
     netmask : associated netmask
     mtu : associated mtu
     isup : if this device is up
     isanalias : if this device is an alias
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_conf_devices_add_user
   Description :
     Ajoute un device a la configuration de lcrzo.
   Parametre(s) d'entree :
     priorityoversystem : si cette configuration doit etre 
                          prioritaire sur celle du systeme
     device : nom du device
     etha : adresse Ethernet du device
     ipl : adresse IP du device
     netmask : masque reseau du device
     mtu : mtu du device
     isup : si le device est actif
     isanalias : si le device est un alias
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_conf_devices_add_user(lcrzo_bool priorityoversystem,
                                lcrzo_constdevice device,
				lcrzo_int16 hwtype,
			        lcrzo_constetha etha,
			        lcrzo_ipl ipl,
			        lcrzo_ipl netmask,
			        lcrzo_uint32 mtu,
			        lcrzo_bool isup,
			        lcrzo_bool isanalias,
				lcrzo_bool ispointtopoint,
				lcrzo_ipl pointtopointipl);

/*en Name : lcrzo_conf_devices_del_xxx
   Description :
     Delete a device from the configuration (system or user)
     of lcrzo.
   Input parameter(s) :
     device : device name to delete
     etha : associated Ethernet address
     ipl : associated IP address
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_conf_devices_del_xxx
   Description :
     Enleve un device de la configuration (systeme ou 
     utilisateur) de lcrzo.
   Parametre(s) d'entree :
     device : nom du device
     etha : adresse Ethernet du device
     ipl : adresse IP du device
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_conf_devices_del_system(lcrzo_constdevice device,
				  lcrzo_int16 hwtype,
			          lcrzo_constetha etha,
			          lcrzo_ipl ipl);
int lcrzo_conf_devices_del_user(lcrzo_constdevice device,
				lcrzo_int16 hwtype,
				lcrzo_constetha etha,
			        lcrzo_ipl ipl);

/*en Name : lcrzo_conf_devices_count
   Description :
     Count the number of devices
   Input parameter(s) :
   Input/output parameter(s) :
   Output parameter(s) :
     *pcount : number of devices
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_conf_devices_count
   Description :
     Compte le nombre de devices
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pcount : nombre de devices
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_conf_devices_count(lcrzo_int32 *pcount);

/*EN*******************************************
 * Note about positions :                     *
 * Warning : 0 is an invalid value            *
 * Here are possible values for a position :  *
 *  +1 : first item                           *
 *  +2 : second item                          *
 *  ...                                       *
 * Moreover, we can count from the end :      *
 *  -1 : last item                            *
 *  -2 : last but one item                    *
 *  ...                                       *
 **********************************************/
/*FR*******************************************
 * Note sur les positions :                   *
 * Attention : 0 n'est pas une valeur valide  *
 * Voici des valeurs de position possibles :  *
 *  +1 : premier element                      *
 *  +2 : deuxieme element                     *
 *  ...                                       *
 * De plus, on peut partir de la fin :        *
 *  -1 : dernier element                      *
 *  -2 : avant dernier element                *
 *  ...                                       *
 **********************************************/

/*en Name : lcrzo_conf_devices_value_pos
   Description :
     Get the device located at the wanted position
   Input parameter(s) :
     pos : position of the wanted device (cf. the above frame to
           know possible values)
   Input/output parameter(s) :
   Output parameter(s) :
     device : device name
     etha : associated Ethernet address
     *pipl : associated IP address
     *pnetmask : associated netmask
     *pmtu : associated mtu
     *pisup : if this device is up
     *pisanalias : if this device is an alias
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_conf_devices_value_pos
   Description :
     Fournit le device situe a la position voulue
   Parametre(s) d'entree :
     pos : position du device cherche (cf. le cadre
            ci-dessus pour connaitre les valeurs valides)
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     device : nom du device
     etha : adresse Ethernet du device
     *pipl : adresse IP du device
     *pnetmask : masque reseau du device
     *pmtu : mtu du device
     *pisup : si le device est actif
     *pisanalias : si le device est un alias
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_conf_devices_value_pos(lcrzo_int32 pos,
				 lcrzo_device device,
				 lcrzo_int16 *phwtype,
				 lcrzo_etha etha,
				 lcrzo_ipl *pipl,
				 lcrzo_ipl *pnetmask,
				 lcrzo_uint32 *pmtu,
				 lcrzo_bool *pisup,
				 lcrzo_bool *pisanalias,
				 lcrzo_bool *pispointtopoint,
				 lcrzo_ipl *ppointtopointipl);

/*en Name : lcrzo_conf_devices_fprint
   Description :
     Print the configuration of devices of lcrzo.
   Input parameter(s) :
     *pf : FILE where the function prints
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_conf_devices_fprint
   Description :
     Affiche la configuration des devices de lcrzo.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_conf_devices_fprint(LCRZOFILE *pf);
#define lcrzo_conf_devices_print() lcrzo_conf_devices_fprint(stdout)


/*-------------------------------------------------------------*/
/*en Name : lcrzo_conf_arp_add_user
   Description :
     Add an arp entry to the configuration of lcrzo.
   Input parameter(s) :
     priorityoversystem : if this configuration is prioritary
                          over the system configuration
     device : device name
     etha : Ethernet address
     ipl : IP address
     ispermanent : if this arp entry is permanent
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_conf_arp_add_user
   Description :
     Ajoute une entree ARP a la configuration de lcrzo.
   Parametre(s) d'entree :
     priorityoversystem : si cette configuration doit etre 
                          prioritaire sur celle du systeme
     device : nom du device
     etha : adresse Ethernet
     ipl : adresse IP
     ispermanent : si cette entree arp est permanente
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_conf_arp_add_user(lcrzo_bool priorityoversystem,
                            lcrzo_constdevice device,
		            lcrzo_constetha etha,
		            lcrzo_ipl ipl,
		            lcrzo_bool ispermanent);

/*en Name : lcrzo_conf_arp_del_xxx
   Description :
     Delete an arp entry from the configuration (system or user)
     of lcrzo.
   Input parameter(s) :
     device : device name
     etha : Ethernet address
     ipl : IP address
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_conf_arp_del_xxx
   Description :
     Enleve une entree ARP de la configuration (systeme ou 
     utilisateur) de lcrzo.
   Parametre(s) d'entree :
     device : nom du device
     etha : adresse Ethernet
     ipl : adresse IP
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_conf_arp_del_system(lcrzo_constdevice device,
		              lcrzo_constetha etha,
		              lcrzo_ipl ipl);
int lcrzo_conf_arp_del_user(lcrzo_constdevice device,
		            lcrzo_constetha etha,
		            lcrzo_ipl ipl);

/*en Name : lcrzo_conf_arp_count
   Description :
     Count the number of arp entries
   Input parameter(s) :
   Input/output parameter(s) :
   Output parameter(s) :
     *pcount : number of entries
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_conf_arp_count
   Description :
     Compte le nombre d'entrees arp
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pcount : nombre d'entrees
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_conf_arp_count(lcrzo_int32 *pcount);

/*en Name : lcrzo_conf_arp_value_pos
   Description :
     Get the arp entry located at the wanted position
   Input parameter(s) :
     pos : position of the wanted arp entry
   Input/output parameter(s) :
   Output parameter(s) :
     device : device name
     etha : Ethernet address
     *pipl : IP address
     *pispermanent : if this arp entry is permanent
     *ptimeset : time to which this arp entry was set
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_conf_arp_value_pos
   Description :
     Fournit l'entree arp situee a la position voulue
   Parametre(s) d'entree :
     pos : position de l'entree arp voulue
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     device : nom de device
     etha : adresse Ethernet
     *pipl : adresse IP
     *pispermanent : si cette entree est permanente
     *ptimeset : heure a laquelle cette entree a ete ajoutee
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_conf_arp_value_pos(lcrzo_int32 pos,
			     lcrzo_device device,
			     lcrzo_etha etha,
			     lcrzo_ipl *pipl,
			     lcrzo_bool *pispermanent,
			     lcrzo_uint32 *ptimeset);

/*en Name : lcrzo_conf_arp_fprint
   Description :
     Print the arp configuration of lcrzo.
   Input parameter(s) :
     *pf : FILE where the function prints
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_conf_arp_fprint
   Description :
     Affiche la configuration arp de lcrzo.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_conf_arp_fprint(LCRZOFILE *pf);
#define lcrzo_conf_arp_print() lcrzo_conf_arp_fprint(stdout)


/*-------------------------------------------------------------*/
/*en Name : lcrzo_conf_routes_add_user
   Description :
     Add a route to the configuration of lcrzo.
   Input parameter(s) :
     priorityoversystem : if this configuration is prioritary
                          over the system configuration
     device : device name
     dest/mask : IP address/netmask
     ipsource : source IP address to use (if 0, local device)
     gateway : gateway to use (if 0, dest is on LAN)
     isup : if this route is up
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_conf_routes_add_user
   Description :
     Ajoute une route a la configuration de lcrzo.
   Parametre(s) d'entree :
     priorityoversystem : si cette configuration doit etre 
                          prioritaire sur celle du systeme
     device : nom du device
     dest/mask : adresse IP/masque de la route voulue
     ipsource : adresse IP source a employer (si 0, device local)
     gateway : routeur a employer (si 0, dest est sur le LAN)
     isup : si cette route est active
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_conf_routes_add_user(lcrzo_bool priorityoversystem,
                               lcrzo_constdevice device, 
			       lcrzo_ipl dest,
			       lcrzo_ipl mask,
			       lcrzo_ipl ipsource,
			       lcrzo_ipl gateway,
                               lcrzo_uint32 metric,
			       lcrzo_bool isup);

/*en Name : lcrzo_conf_routes_del_xxx
   Description :
     Delete a route from the configuration (system or user)
     of lcrzo.
   Input parameter(s) :
     dest/mask : IP address/mask of the route
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_conf_routes_del_xxx
   Description :
     Enleve une route de la configuration (systeme ou 
     utilisateur) de lcrzo.
   Parametre(s) d'entree :
     dest/mask : adresse IP/masque de la route
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_conf_routes_del_system(lcrzo_ipl dest,
			         lcrzo_ipl mask,
			         lcrzo_ipl gateway);
int lcrzo_conf_routes_del_user(lcrzo_ipl dest,
			       lcrzo_ipl mask,
			       lcrzo_ipl gateway);

/*en Name : lcrzo_conf_routes_count
   Description :
     Count the number of routes
   Input parameter(s) :
   Input/output parameter(s) :
   Output parameter(s) :
     *pcount : number of routes
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_conf_routes_count
   Description :
     Compte le nombre de routes
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pcount : nombre de routes
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_conf_routes_count(lcrzo_int32 *pcount);

/*en Name : lcrzo_conf_routes_value_pos
   Description :
     Get the route located at the wanted position
   Input parameter(s) :
     pos : position of the wanted route
   Input/output parameter(s) :
   Output parameter(s) :
     device : device name
     *pdest : IP address
     *pmask : netmask
     *pipsource : source IP address to use
     *pgateway : gateway to use
     *pisup : if this route is up
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_conf_routes_value_pos
   Description :
     Fournit la route situee a la position voulue
   Parametre(s) d'entree :
     pos : position de la route voulue
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     device : nom de device
     *pdest : adresse IP destination
     *pmask : masque destination
     *pipsource : adresse IP source a employer
     *pgateway : routeur a employer
     *pisup : si cette route est active
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_conf_routes_value_pos(lcrzo_int32 pos,
				lcrzo_device device,
				lcrzo_ipl *pdest,
				lcrzo_ipl *pmask,
				lcrzo_ipl *pipsource,
				lcrzo_ipl *pgateway,
                                lcrzo_uint32 *pmetric,
				lcrzo_bool *pisup);

/*en Name : lcrzo_conf_routes_fprint
   Description :
     Print the routing configuration of lcrzo.
   Input parameter(s) :
     *pf : FILE where the function prints
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_conf_routes_fprint
   Description :
     Affiche la configuration de routage de lcrzo.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_conf_routes_fprint(LCRZOFILE *pf);
#define lcrzo_conf_routes_print() lcrzo_conf_routes_fprint(stdout)


/*-------------------------------------------------------------*/
/*EN************************************************************
 * Note about the following functions :                         *
 * The default values are correct for most applications.       *
 * You should modify their parameters only if you know         *
 * what you are doing.                                         *
 ***************************************************************/
/*FR************************************************************
 * Note sur les fonctions suivantes :                          *
 * Les valeurs par defaut sont correctes pour la majorite des  *
 * applications.                                               *
 * Vous ne devriez modifier les parametres que si vous savez   *
 * ce que vous faites.                                         *
 ***************************************************************/

/*en to allow the update of the configuration */
/*fr pour permettre la mise a jour de la configuration */
int lcrzo_conf_set_sysrenew_allow(lcrzo_bool yes);
int lcrzo_conf_set_sysrenew_allow_devices(lcrzo_bool yes);
int lcrzo_conf_set_sysrenew_allow_arpperm(lcrzo_bool yes);
int lcrzo_conf_set_sysrenew_allow_arpcache(lcrzo_bool yes);
int lcrzo_conf_set_sysrenew_allow_routes(lcrzo_bool yes);
int lcrzo_conf_get_sysrenew_allow(lcrzo_bool *pyes);
int lcrzo_conf_get_sysrenew_allow_devices(lcrzo_bool *pyes);
int lcrzo_conf_get_sysrenew_allow_arpperm(lcrzo_bool *pyes);
int lcrzo_conf_get_sysrenew_allow_arpcache(lcrzo_bool *pyes);
int lcrzo_conf_get_sysrenew_allow_routes(lcrzo_bool *pyes);

/*en lifetime of permanent entries before having to update them */
/*fr duree de vie des entrees permanentes au bout de laquelle
     celle-ci doivent etre mises a jour */
int lcrzo_conf_set_permentriesperiod(lcrzo_uint32 period);
int lcrzo_conf_get_permentriesperiod(lcrzo_uint32 *pperiod);

/*en lifetime of cached entries */
/*fr duree de vie des entrees mises en cache */
int lcrzo_conf_set_cacheentriestimeout(lcrzo_uint32 timeout);
int lcrzo_conf_get_cacheentriestimeout(lcrzo_uint32 *ptimeout);

/*-------------------------------------------------------------*/
/*en to force an update of the system configuration */
/*fr pour forcer la mise a jour de la configuration systeme */
int lcrzo_conf_renewnow_system(void);
int lcrzo_conf_devices_renewnow_system(void);
int lcrzo_conf_arpperm_renewnow_system(void);
int lcrzo_conf_arpcache_renewnow_system(void);
int lcrzo_conf_routes_renewnow_system(void);

/*-------------------------------------------------------------*/
/*en flush the system/user configuration */
/*fr purge la configuration systeme/utilisateur */
int lcrzo_conf_flush_system(void);
int lcrzo_conf_devices_flush_system(void);
int lcrzo_conf_arpperm_flush_system(void);
int lcrzo_conf_arpcache_flush_system(void);
int lcrzo_conf_routes_flush_system(void);
int lcrzo_conf_flush_user(void);
int lcrzo_conf_devices_flush_user(void);
int lcrzo_conf_arpperm_flush_user(void);
int lcrzo_conf_arpcache_flush_user(void);
int lcrzo_conf_routes_flush_user(void);


/*-------------------------------------------------------------*/
/*en Name : lcrzo_conf_route_to_host
   Description :
     Determine device, sources addresses and destination Ethernet
     address to use to send an IP packet to a host.
   Input parameter(s) :
     hostipl : host to reach
   Input/output parameter(s) :
   Output parameter(s) :
     device : device to use
     srcetha : Ethernet source address to use
     destetha : Ethernet destination address to use
     *psrcipl : IP source address to use
     *prouteripl : IP address of the router (or 0 if hostipl is
                   on the local LAN)
   Normal return values :
     LCRZO_ERR_OK : ok (set: all ; *prouteripl==0 if hostipl
                    is on the LAN)
     LCRZO_ERR_OKROUTELOCALDEV : hostipl is the address of a 
                                 local device (set : device)
     LCRZO_ERR_OKROUTEHOSTUNREACH : hostipl in on LAN, but we 
                                    can't get its Ethernet address
				    (set : device, srcetha, *psrcipl,
				    *prouteripl==0)
     LCRZO_ERR_OKROUTEGWUNREACH : hostipl in behind a router, but we 
                                  can't get the Ethernet address of
                                  the router (set : device,
				  srcetha, *psrcipl, *prouteripl)
     LCRZO_ERR_OKROUTENOTETHER : Ethernet is not used to reach
                                 destination (set : device, 
				 *psrcipl, *prouteripl)
     LCRZO_ERR_OKROUTENOTFOUND : the routing table doesn't contain
                                 a route to hostipl (set : none)
*/
/*fr Nom : lcrzo_conf_route_to_host
   Description :
     Determine le device, les adresses source et l'adresse Ethernet
     destination a employer pour envoyer un paquet IP vers une machine.
   Parametre(s) d'entree :
     hostipl : machine a joindre
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     device : device a employer
     srcetha : adresse ethernet source a employer
     destetha : adresse ethernet destination a employer
     *psrcipl : adresse IP source a employer
     *prouteripl : adresse IP du routeur (ou 0 si hostipl est sur le 
                   LAN local)
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok (affecte: tous ; *prouteripl==0 si hostipl
                    est sur le LAN)
     LCRZO_ERR_OKROUTELOCALDEV : hostipl est l'adresse d'un device
                                 local (affecte : device)
     LCRZO_ERR_OKROUTEHOSTUNREACH : hostipl est sur le LAN, mais on 
                                    n'a pas pu obtenir son adresse
                                    Ethernet (affecte : device,
				    srcetha, *psrcipl, *prouteripl==0)
     LCRZO_ERR_OKROUTEGWUNREACH : hostipl est derriere un routeur, 
                                  mais on n'a pas pu obtenir l'adresse
                                  Ethernet du routeur (affecte : device,
				  srcetha, *psrcipl, *prouteripl)
     LCRZO_ERR_OKROUTENOTETHER : Ethernet n'est pas utilise pour
                                 atteindre la destination (affecte :
				 device, *psrcipl, *prouteripl)
     LCRZO_ERR_OKROUTENOTFOUND : la table de routage ne contient pas 
                                 d'entree vers hostipl (affecte :
				 aucun)
*/
int lcrzo_conf_route_to_host(lcrzo_ipl hostipl,
			     lcrzo_device device,
			     lcrzo_etha srcetha,
			     lcrzo_etha destetha,
			     lcrzo_ipl *psrcipl,
			     lcrzo_ipl *prouteripl);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_conf_route_to_host_ip
   Description :
     Determine device and addresses to use to send an IP packet
     to a host.
   Input parameter(s) :
     hostipl : host to reach
   Input/output parameter(s) :
   Output parameter(s) :
     device : device to use (for example to sniff)
     *psrcipl : IP source address to use
     *prouteripl : IP address of the router (or 0 if hostipl is
                   on the local LAN)
   Normal return values :
     LCRZO_ERR_OK : ok (set: all ; *prouteripl==0 if hostipl
                    is on the LAN)
     LCRZO_ERR_OKROUTELOCALDEV : hostipl is the address of a 
                                 local device (set : device)
     LCRZO_ERR_OKROUTENOTFOUND : the routing table doesn't contain
                                 a route to hostipl (set : none)
*/
/*fr Nom : lcrzo_conf_route_to_host_ip
   Description :
     Determine le device et les adresses a employer pour envoyer
     un paquet IP vers une machine.
   Parametre(s) d'entree :
     hostipl : machine a joindre
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     device : device a employer (par exemple pour sniffer)
     *psrcipl : adresse IP source a employer
     *prouteripl : adresse IP du routeur (ou 0 si hostipl est sur le 
                   LAN local)
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok (affecte: tous ; *prouteripl==0 si hostipl
                    est sur le LAN)
     LCRZO_ERR_OKROUTELOCALDEV : hostipl est l'adresse d'un device
                                 local (affecte : device)
     LCRZO_ERR_OKROUTENOTFOUND : la table de routage ne contient pas 
                                 d'entree vers hostipl (affecte :
				 aucun)
*/
int lcrzo_conf_route_to_host_ip(lcrzo_ipl hostipl,
				lcrzo_device device,
				lcrzo_ipl *psrcipl,
				lcrzo_ipl *prouteripl);

/*-------------------------------------------------------------*/
/*en for debugging purposes only */
/*fr juste pour le debogage */
int lcrzo_conf_debug_fprint(LCRZOFILE *pf);
#define lcrzo_conf_debug_print() lcrzo_conf_debug_fprint(stdout)


