/*en    Laurent Constantin's network library (lcrzo)
                 lcrzo_device module

  Functions herein allow to work on devices.
*/
/*fr    Bibliotheque reseau de Laurent Constantin (lcrzo)
                 Module lcrzo_device

  Les fonctions presentes dans ce module permettent de gerer des
  devices.
*/

/*en device name (ex. eth0, le0) */
/*fr nom de device ethernet (ex. eth0, le0) */
#define LCRZO_DEVICE_MAXBYTES 128
typedef lcrzo_int8 lcrzo_device[LCRZO_DEVICE_MAXBYTES+1]; /*+1 for/pour '\0'*/
typedef const lcrzo_device lcrzo_constdevice;

/*-------------------------------------------------------------*/
/*en this define permits to verify that there is no buffer overflow
   before setting a lcrzo_device */
/*fr ce define permet de verifier qu'il n'y ait pas
   de buffer overflow dans une chaine lcrzo_device*/
#define lcrzo_device_verifbof(device) if(strlen(device)>LCRZO_DEVICE_MAXBYTES)return(LCRZO_ERR_PAOVERFLOW)

/*-------------------------------------------------------------*/
/*en Name : lcrzo_device_init
   Description :
     Initialize an lcrzo_device address
   Input parameter(s) :
     indevice : string containing the host name
   Input/output parameter(s) :
   Output parameter(s) :
     outdevice : lcrzo_device set with indevice
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : all the data can't be put in deviceout
*/
/*fr Nom : lcrzo_device_init
   Description :
     Initialise une adresse sous la forme lcrzo_device
   Parametre(s) d'entree :
     indevice : chaine contenant un nom de machine
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     outdevice : lcrzo_device affecte par indevice 
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : toutes les donnees entrees n'ont pas
                                 ete mises dans deviceout 
*/
int lcrzo_device_init(lcrzo_constdevice indevice,
		      lcrzo_device outdevice);
/*en idem, but convert " eth0:1 " to "eth0" */
/*fr idem, mais convertit " eth0:1 " en "eth0" */
int lcrzo_device_init_noalias(lcrzo_constdevice indevice,
			      lcrzo_device outdevice);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_device_equal
   Description :
     Verify if two lcrzo_device are equal
   Input parameter(s) :
     device1 : lcrzo_device to compare with device2
     device2 : lcrzo_device to compare with device1
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     0 : if device1 and device2 are different
     1 : if device1 and device2 are equal
*/
/*fr Nom : lcrzo_device_equal
   Description :
     Verifie l'egalite de deux lcrzo_device
   Parametre(s) d'entree :
     device1 : lcrzo_device a comparer avec device2
     device2 : lcrzo_device a comparer avec device1
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     0 : si device1 et device2 sont differents
     1 : si device1 et device2 sont egaux
*/
lcrzo_bool lcrzo_device_equal(lcrzo_constdevice device1,
		       lcrzo_constdevice device2);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_device_cmp
   Description :
     Compare two lcrzo_device
   Input parameter(s) :
     device1 : lcrzo_device to compare with device2
     device2 : lcrzo_device to compare with device1
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     -1 : device1<device2
      0 : if device1 and device2 are equal
     +1 : device1>device2
*/
/*fr Nom : lcrzo_device_cmp
   Description :
     Compare deux lcrzo_device
   Parametre(s) d'entree :
     device1 : lcrzo_device a comparer avec device2
     device2 : lcrzo_device a comparer avec device1
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     -1 : device1<device2
      0 : si device1 et device2 sont egaux
     +1 : device1>device2
*/
lcrzo_int8 lcrzo_device_cmp(lcrzo_constdevice device1,
		     lcrzo_constdevice device2);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_device_initdefault
   Description :
     Initialize a default lcrzo_device address
   Input parameter(s) :
   Input/output parameter(s) :
   Output parameter(s) :
     device : lcrzo_device set with "" (default value)
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_device_initdefault
   Description :
     Initialise une adresse lcrzo_device avec la valeur par defaut
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     device : lcrzo_device affecte par "" (valeur par defaut)
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_device_initdefault(lcrzo_device device);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_mtu_init_device
   Description :
     Get the MTU (Maxi. transmission Unit) of a device
   Input parameter(s) :
     device : device of which we get the MTU
   Input/output parameter(s) :
   Output parameter(s) :
     *pmtu : MTU of device 
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_mtu_init_device
   Description :
     Fournit le mtu associe au device
   Parametre(s) d'entree :
     device : device dont on veut connaitre le MTU
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pmtu : MTU de device
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_mtu_init_device(lcrzo_constdevice device,
			  lcrzo_uint32 *pmtu);

/*en Name : lcrzo_etha_init_device
   Description :
     Get the Ethernet address of a device
   Input parameter(s) :
     device : device of which we get the Ethernet address
   Input/output parameter(s) :
   Output parameter(s) :
     etha : lcrzo_etha of device 
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_etha_init_device
   Description :
     Fournit l'adresse Ethernet associee au device
   Parametre(s) d'entree :
     device : device dont on veut connaitre l'adresse Ethernet
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     etha : lcrzo_etha de device
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_etha_init_device(lcrzo_constdevice device,
			   lcrzo_etha etha);

/*en Name : lcrzo_ipl_init_device
   Description :
     Get the IP address of a device
   Input parameter(s) :
     device : device of which we get the IP address
   Input/output parameter(s) :
   Output parameter(s) :
     ipl : lcrzo_ipl of device 
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ipl_init_device
   Description :
     Fournit l'adresse IP associee au device
   Parametre(s) d'entree :
     device : device dont on veut connaitre l'adresse IP
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     ipl : lcrzo_ipl de device
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ipl_init_device(lcrzo_constdevice device,
			  lcrzo_ipl *pipl,
			  lcrzo_ipl *pmask);

/*en Name : lcrzo_hs_init_device
   Description :
     Get the hostname of a device
   Input parameter(s) :
     device : device of which we get the hostname
   Input/output parameter(s) :
   Output parameter(s) :
     hs : lcrzo_hs of device 
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_hs_init_device
   Description :
     Fournit le nom de machine associee au device
   Parametre(s) d'entree :
     device : device dont on veut connaitre le nom de machine
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     hs : lcrzo_hs de device
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_hs_init_device(lcrzo_constdevice device,
			 lcrzo_hs hs);

/*-------------------------------------------------------------*/
/*en we define 6 types of devices : */
/*fr on definit 6 types de devices : */
/* Note : those types are almost obsolete. Use :
     lcrzo_conf_devices_count
     lcrzo_conf_devices_value_pos
   which are more powerful
*/
/*en down interfaces, up interfaces, down aliases, up aliases*/
/*fr interfaces montees ou non, alias montes ou non */
#define LCRZO_DEVICE_TYPE_ALL         1
/*en up interfaces, up aliases (no loopback)*/
/*fr interfaces montees, alias montes (sauf le loopback)*/
#define LCRZO_DEVICE_TYPE_ONLYVALIDIP 2
/*en up Ethernet interfaces (one for each network card)*/
/*fr interfaces Ethernet montees (une seule par carte reseau) */
#define LCRZO_DEVICE_TYPE_ONLYBOARDS  3
/*en loopback */
/*fr loopback */
#define LCRZO_DEVICE_TYPE_LOOPBACK    4
/*en up interfaces */
/*fr interfaces up */
#define LCRZO_DEVICE_TYPE_UP          5
/*en alias */
/*fr alias */
#define LCRZO_DEVICE_TYPE_ALIAS       6
/*en best interface for default choice */
/*fr meilleure interface pour les choix par defaut */
#define LCRZO_DEVICE_TYPE_BEST        7

/*en Name : lcrzo_device_istype
   Description :
     Verify if a device is of given type
   Input parameter(s) :
     device : device to check
     type : type to check (LCRZO_DEVICE_TYPE_ALL, ONLYVALIDIP, etc.)
   Input/output parameter(s) :
   Output parameter(s) :
     *pyesitistype : 1 if device has type 'type'
                     0 if device hasn't type 'type'
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_device_istype
   Description :
     Verifie si un device est d'un type donne
   Parametre(s) d'entree :
     device : device a verifier
     type : type a verifier (LCRZO_DEVICE_TYPE_ALL, ONLYVALIDIP, etc.)
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pyesitistype : 1 si device est de type 'type'
                     0 si device n'est pas de type 'type'
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_device_istype(lcrzo_constdevice device,
			lcrzo_int16 type,
			lcrzo_bool *pyesitistype);

/*en Name : lcrzo_device_count
   Description :
     Count the number of devices of given type
   Input parameter(s) :
     type : type to count (LCRZO_DEVICE_TYPE_ALL, ONLYVALIDIP, etc.)
   Input/output parameter(s) :
   Output parameter(s) :
     *pcount : number of devices which have type 'type'
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_device_count
   Description :
     Compte le nombre de devices d'un type donne
   Parametre(s) d'entree :
     type : type a compter (LCRZO_DEVICE_TYPE_ALL, ONLYVALIDIP, etc.)
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pcount : nombre de devices de type 'type'
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_device_count(lcrzo_int16 type,
		       lcrzo_int32 *pcount);

/*EN*******************************************
 * Note about positions :                     *
 * Warning : 0 is an invalid value            *
 * Here are possible values for a position :  *
 *  +1 : first item                           *
 *  +2 : second item                          *
 *  ...                                       *
 * Moreover, we can count from the end :      *
 *  -1 : last item                            *
 *  -2 : last but one item                    *
 *  ...                                       *
 **********************************************/
/*FR*******************************************
 * Note sur les positions :                   *
 * Attention : 0 n'est pas une valeur valide  *
 * Voici des valeurs de position possibles :  *
 *  +1 : premier element                      *
 *  +2 : deuxieme element                     *
 *  ...                                       *
 * De plus, on peut partir de la fin :        *
 *  -1 : dernier element                      *
 *  -2 : avant dernier element                *
 *  ...                                       *
 **********************************************/

/*en Name : lcrzo_device_value_pos
   Description :
     Get a device of a given type
   Input parameter(s) :
     type : type we want (LCRZO_DEVICE_TYPE_ALL, ONLYVALIDIP, etc.)
     pos : position of the wanted device (cf. the above frame to
           know possible values)
   Input/output parameter(s) :
   Output parameter(s) :
     device : device number 'pos' and which have type 'type'
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_device_value_pos
   Description :
     Fournit le device d'un type donne, et situe a la position
     voulue
   Parametre(s) d'entree :
     type : type cherche (LCRZO_DEVICE_TYPE_ALL, ONLYVALIDIP, etc.)
     pos : position du device cherche (cf. le cadre ci-dessus pour
           connaitre les valeurs valides)
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     device : device numero 'pos' et de type 'type'
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_device_value_pos(lcrzo_int16 type,
			   lcrzo_int32 pos,
			   lcrzo_device device);

/*en Name : lcrzo_device_value_first
   Description :
     Get the first device of a given type
   Input parameter(s) :
     type : type we want (LCRZO_DEVICE_TYPE_ALL, ONLYVALIDIP, etc.)
   Input/output parameter(s) :
   Output parameter(s) :
     device : first device which have type 'type'
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_device_value_first
   Description :
     Fournit le premier device d'un type donne
   Parametre(s) d'entree :
     type : type cherche (LCRZO_DEVICE_TYPE_ALL, ONLYVALIDIP, etc.)
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     device : premier device de type 'type'
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_device_value_first(lcrzo_int16 type,
			     lcrzo_device device);

/*en Name : lcrzo_device_value_last
   Description :
     Get the last device of a given type
   Input parameter(s) :
     type : type we want (LCRZO_DEVICE_TYPE_ALL, ONLYVALIDIP, etc.)
   Input/output parameter(s) :
   Output parameter(s) :
     device : last device which have type 'type'
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_device_value_last
   Description :
     Fournit le dernier device d'un type donne
   Parametre(s) d'entree :
     type : type cherche (LCRZO_DEVICE_TYPE_ALL, ONLYVALIDIP, etc.)
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     device : dernier device de type 'type'
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_device_value_last(lcrzo_int16 type,
			    lcrzo_device device);

/*en Name : lcrzo_device_loop_range
   Description :
     Call a function for each device of a given type, and in a
     given position range.
   Input parameter(s) :
     type : type we want (LCRZO_DEVICE_TYPE_ALL, ONLYVALIDIP, etc.)
     startpos : start position for the loop (cf. the above frame to
                know possible values)
     endpos : end position for the loop (cf. the above frame to
              know possible values)
     pfunc : memory address of the function which will be called
             for each device in the range [startpos->endpos]. For
             each call, the first parameter ('device') will be set
             with the device, and the second parameter ('pinfos')
             will be set with the optional parameter below.
   Input/output parameter(s) :
     pinfos : optional parameter (can be NULL) which will be
              used as the second parameter for *pfunc. This may be
              used to send information to *pfunc.
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_device_loop_range
   Description :
     Appelle une fonction pour les devices d'un type donne, et
     situe dans un intervalle donne.
   Parametre(s) d'entree :
     type : type cherche (LCRZO_DEVICE_TYPE_ALL, ONLYVALIDIP, etc.)
     startpos : position de debut de la boucle (cf. le cadre
                ci-dessus pour connaitre les valeurs valides)
     endpos : position de fin de la boucle (cf. le cadre
              ci-dessus pour connaitre les valeurs valides)
     pfunc : adresse memoire de la fonction appelee pour chaque
             device de l'intervalle [startpos->endpos]. Lors de
	     chaque appel, le premier parametre ('device') est 
	     affecte avec le device courant, et le deuxieme
	     parametre ('pinfos') est affecte avec la valeur 
	     de pinfos ci-dessous.
   Parametre(s) d'entree-sortie :
     pinfos : parametre optionnel (peut mettre NULL) qui sera
              passe en deuxieme parametre de pfunc. Il peut 
	      servir a envoyer des informations a pfunc.
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_device_loop_range(lcrzo_int16 type,
			    lcrzo_int32 startpos,
			    lcrzo_int32 endpos,
			    int (*pfunc)(lcrzo_device device,
					 void *pinfos),
			    void *pinfos);

/*en Name : lcrzo_device_loop_all
   Description :
     Call a function for every device of a given type.
   Input parameter(s) :
     type : type we want (LCRZO_DEVICE_TYPE_ALL, ONLYVALIDIP, etc.)
     pfunc : memory address of the function which will be called
             for every device. For each call, the first parameter
             ('device') will be set with the device, and the
	     second parameter ('pinfos') will be set with the 
	     optional parameter below.
   Input/output parameter(s) :
     pinfos : optional parameter (can be NULL) which will be
              used as the second parameter for *pfunc. This may be
              used to send information to *pfunc.
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_device_loop_all
   Description :
     Appelle une fonction pour tous les devices d'un type donne.
   Parametre(s) d'entree :
     type : type cherche (LCRZO_DEVICE_TYPE_ALL, ONLYVALIDIP, etc.)
     pfunc : adresse memoire de la fonction appelee pour chaque
             device. Lors de chaque appel, le premier parametre
	     ('device') est affecte avec le device courant, et 
	     le deuxieme parametre ('pinfos') est affecte avec
	     la valeur de pinfos ci-dessous.
   Parametre(s) d'entree-sortie :
     pinfos : parametre optionnel (peut mettre NULL) qui sera
              passe en deuxieme parametre de pfunc. Il peut 
	      servir a envoyer des informations a pfunc.
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_device_loop_all(lcrzo_int16 type,
			  int (*pfunc)(lcrzo_device device,
				       void *pinfos),
			  void *pinfos);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_device_init_stdin
   Description :
     Initialize an lcrzo_device with the address entered by the 
     user through the keyboard
   Input parameter(s) :
     message : message to print before
     defaultdevice : default lcrzo_device to use if user enters nothing
   Input/output parameter(s) :
   Output parameter(s) :
     device : lcrzo_device set with user's string
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_device_init_stdin
   Description :
     Initialise une adresse par une valeur entree au clavier
   Parametre(s) d'entree :
     message : message a afficher avant de demander a entrer
     defaultdevice : lcrzo_device par defaut a utiliser si l'utilisateur
                     n'entre rien
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     device : lcrzo_device affecte par la valeur entree au clavier
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_device_init_stdin(lcrzo_conststring message,
			    lcrzo_constdevice defaultdevice,
			    lcrzo_device device);

