/*en    Laurent Constantin's network library (lcrzo)
                 lcrzo_icmp module

  Functions herein allow to create and decode icmp packets.
*/
/*fr  Bibliotheque reseau de Laurent Constantin (lcrzo)
                 Module lcrzo_icmp

  Les fonctions presentes dans ce module permettent de creer ou
  d'analyser des paquets icmp.
*/

/*-------------------------------------------------------------*/
#define LCRZO_ICMPTYPE_DESTUNREACH            3
#define LCRZO_ICMPCODE_DESTUNREACH_NET        0
#define LCRZO_ICMPCODE_DESTUNREACH_HOST       1
#define LCRZO_ICMPCODE_DESTUNREACH_PROTO      2
#define LCRZO_ICMPCODE_DESTUNREACH_PORT       3
#define LCRZO_ICMPCODE_DESTUNREACH_FRAG       4
#define LCRZO_ICMPCODE_DESTUNREACH_SR         5
#define LCRZO_ICMPCODE_DESTUNREACH_NETUNKNO   6
#define LCRZO_ICMPCODE_DESTUNREACH_HOSTUNKNO  7
#define LCRZO_ICMPCODE_DESTUNREACH_HOSTISOL   8
#define LCRZO_ICMPCODE_DESTUNREACH_NETPROHI   9
#define LCRZO_ICMPCODE_DESTUNREACH_HOSTPROHI 10
#define LCRZO_ICMPCODE_DESTUNREACH_NETTOS    11
#define LCRZO_ICMPCODE_DESTUNREACH_HOSTTOS   12
#define LCRZO_ICMPCODE_DESTUNREACH_PROHI     13
#define LCRZO_ICMPCODE_DESTUNREACH_PRECVIOL  14
#define LCRZO_ICMPCODE_DESTUNREACH_PRECCUT   15

/*en Name : lcrzo_packet_initm_xxxicmpdestunreach
   Description :
     Initialize a packet with {ICMP header, DestinationUnreachable}.
     Initialize a packet with {IP header, ICMP header,
     DestinationUnreachable}.
     Initialize a packet with {Ethernet header, IP header, 
     ICMP header, DestinationUnreachable}.
   Input parameter(s) :
     hdrleth : Ethernet header
     hdrlip : IP header
     ipopt : IP options
     ipoptsize : IP options size
     code : icmp code
     ippacket : ip packet which generated this ICMP error
     ippacketsize : size of ippacket
   Input/output parameter(s) :
   Output parameter(s) :
     *pdataout : output array, which is allocated.
     *pdataoutsize : reached size of *pdataout
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_packet_initm_xxxicmpdestunreach
   Description :
     Initialise un paquet avec {entete ICMP, DestinationUnreachable}.
     Initialise un paquet avec {entete IP, entete ICMP,
     DestinationUnreachable}.
     Initialise un paquet avec {entete Ethernet, entete IP, 
     entete ICMP, DestinationUnreachable}.
   Parametre(s) d'entree :
     hdrleth : entete Ethernet
     hdrlip : entete IP
     ipopt : options IP
     ipoptsize : taille des options IP
     code : code icmp
     ippacket : paquet IP ayant genere cette erreur ICMP
     ippacketsize : taille de ippacket
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pdataout : tableau de sortie qui est alloue
     *pdataoutsize : taille atteinte par *pdataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_packet_initm_icmpdestunreach(lcrzo_uint8 code,
				       lcrzo_constdata ippacket, 
				       lcrzo_uint16 ippacketsize,
				       lcrzo_data *pdataout,
				       lcrzo_int32 *pdataoutsize);
int lcrzo_packet_initm_ipopticmpdestunreach(lcrzo_hdrlip hdrlip,
					    lcrzo_constipopt ipopt,
					    lcrzo_uint8 ipoptsize,
					    lcrzo_uint8 code,
					    lcrzo_constdata ippacket, 
					    lcrzo_uint16 ippacketsize,
					    lcrzo_data *pdataout,
					    lcrzo_int32 *pdataoutsize);
int lcrzo_packet_initm_ethipopticmpdestunreach(lcrzo_hdrleth hdrleth,
					       lcrzo_hdrlip hdrlip,
					       lcrzo_constipopt ipopt,
					       lcrzo_uint8 ipoptsize,
					       lcrzo_uint8 code,
					       lcrzo_constdata ippacket, 
					       lcrzo_uint16 ippacketsize,
				               lcrzo_data *pdataout,
					       lcrzo_int32 *pdataoutsize);

/*-------------------------------------------------------------*/
#define LCRZO_ICMPTYPE_TIMEEXCEED        11
#define LCRZO_ICMPCODE_TIMEEXCEED_TTL     0
#define LCRZO_ICMPCODE_TIMEEXCEED_FRAG    1

/*en Name : lcrzo_packet_initm_xxxicmptimeexceed
   Description :
     Initialize a packet with {ICMP header, TimeExceeded}.
     Initialize a packet with {IP header, ICMP header,
     TimeExceeded}.
     Initialize a packet with {Ethernet header, IP header, 
     ICMP header, TimeExceeded}.
   Input parameter(s) :
     hdrleth : Ethernet header
     hdrlip : IP header
     ipopt : IP options
     ipoptsize : IP options size
     code : icmp code
     ippacket : ip packet which generated this ICMP error
     ippacketsize : size of ippacket
   Input/output parameter(s) :
   Output parameter(s) :
     *pdataout : output array, which is allocated.
     *pdataoutsize : reached size of *pdataout
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_packet_initm_xxxicmptimeexceed
   Description :
     Initialise un paquet avec {entete ICMP, TimeExceeded}.
     Initialise un paquet avec {entete IP, entete ICMP,
     TimeExceeded}.
     Initialise un paquet avec {entete Ethernet, entete IP, 
     entete ICMP, TimeExceeded}.
   Parametre(s) d'entree :
     hdrleth : entete Ethernet
     hdrlip : entete IP
     ipopt : options IP
     ipoptsize : taille des options IP
     code : code icmp
     ippacket : paquet IP ayant genere cette erreur ICMP
     ippacketsize : taille de ippacket
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pdataout : tableau de sortie qui est alloue
     *pdataoutsize : taille atteinte par *pdataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_packet_initm_icmptimeexceed(lcrzo_uint8 code,
				      lcrzo_constdata ippacket, 
				      lcrzo_uint16 ippacketsize,
				      lcrzo_data *pdataout,
				      lcrzo_int32 *pdataoutsize);
int lcrzo_packet_initm_ipopticmptimeexceed(lcrzo_hdrlip hdrlip,
					   lcrzo_constipopt ipopt,
					   lcrzo_uint8 ipoptsize,
					   lcrzo_uint8 code,
					   lcrzo_constdata ippacket, 
					   lcrzo_uint16 ippacketsize,
					   lcrzo_data *pdataout,
					   lcrzo_int32 *pdataoutsize);
int lcrzo_packet_initm_ethipopticmptimeexceed(lcrzo_hdrleth hdrleth,
					      lcrzo_hdrlip hdrlip,
					      lcrzo_constipopt ipopt,
					      lcrzo_uint8 ipoptsize,
					      lcrzo_uint8 code,
					      lcrzo_constdata ippacket, 
					      lcrzo_uint16 ippacketsize,
					      lcrzo_data *pdataout,
					      lcrzo_int32 *pdataoutsize);

/*-------------------------------------------------------------*/
/*en idem for ParameterProblem (cf. rfc 792, 1108) */
/*fr idem avec ParameterProblem (cf. rfc 792, 1108) */
#define LCRZO_ICMPTYPE_PARAPROB           12 
#define LCRZO_ICMPCODE_PARAPROB_POINTER    0
#define LCRZO_ICMPCODE_PARAPROB_MISSOPT    1
#define LCRZO_ICMPCODE_PARAPROB_BADLEN     2

int lcrzo_packet_initm_icmpparaprob(lcrzo_uint8 code,
				    lcrzo_uint8 pointer,
				    lcrzo_constdata ippacket, 
				    lcrzo_uint16 ippacketsize,
				    lcrzo_data *pdataout,
				    lcrzo_int32 *pdataoutsize);
int lcrzo_packet_initm_ipopticmpparaprob(lcrzo_hdrlip hdrlip,
					 lcrzo_constipopt ipopt,
					 lcrzo_uint8 ipoptsize,
					 lcrzo_uint8 code,
					 lcrzo_uint8 pointer,
					 lcrzo_constdata ippacket, 
					 lcrzo_uint16 ippacketsize,
					 lcrzo_data *pdataout,
					 lcrzo_int32 *pdataoutsize);
int lcrzo_packet_initm_ethipopticmpparaprob(lcrzo_hdrleth hdrleth,
					    lcrzo_hdrlip hdrlip,
					    lcrzo_constipopt ipopt,
					    lcrzo_uint8 ipoptsize,
					    lcrzo_uint8 code,
					    lcrzo_uint8 pointer,
					    lcrzo_constdata ippacket, 
					    lcrzo_uint16 ippacketsize,
					    lcrzo_data *pdataout,
					    lcrzo_int32 *pdataoutsize);

/*-------------------------------------------------------------*/
/*en idem for SourceQuench (cf. rfc 792) */
/*fr idem avec SourceQuench (cf. rfc 792) */
#define LCRZO_ICMPTYPE_SRCQUENCH    4
int lcrzo_packet_initm_icmpsrcquench(lcrzo_constdata ippacket, 
				     lcrzo_uint16 ippacketsize,
				     lcrzo_data *pdataout,
				     lcrzo_int32 *pdataoutsize);
int lcrzo_packet_initm_ipopticmpsrcquench(lcrzo_hdrlip hdrlip,
					  lcrzo_constipopt ipopt,
					  lcrzo_uint8 ipoptsize,
					  lcrzo_constdata ippacket, 
					  lcrzo_uint16 ippacketsize,
					  lcrzo_data *pdataout,
					  lcrzo_int32 *pdataoutsize);
int lcrzo_packet_initm_ethipopticmpsrcquench(lcrzo_hdrleth hdrleth,
					     lcrzo_hdrlip hdrlip,
					     lcrzo_constipopt ipopt,
					     lcrzo_uint8 ipoptsize,
					     lcrzo_constdata ippacket, 
					     lcrzo_uint16 ippacketsize,
					     lcrzo_data *pdataout,
					     lcrzo_int32 *pdataoutsize);

/*-------------------------------------------------------------*/
/*en idem for Redirect (cf. rfc 792) */
/*fr idem avec Redirect (cf. rfc 792) */
#define LCRZO_ICMPTYPE_REDIRECT         5
#define LCRZO_ICMPCODE_REDIRECT_NET     0
#define LCRZO_ICMPCODE_REDIRECT_HOST    1
#define LCRZO_ICMPCODE_REDIRECT_NETTOS  2
#define LCRZO_ICMPCODE_REDIRECT_HOSTTOS 3
int lcrzo_packet_initm_icmpredirect(lcrzo_uint8 code,
				    lcrzo_ipl gw,
				    lcrzo_constdata ippacket, 
				    lcrzo_uint16 ippacketsize,
				    lcrzo_data *pdataout,
				    lcrzo_int32 *pdataoutsize);
int lcrzo_packet_initm_ipopticmpredirect(lcrzo_hdrlip hdrlip,
					 lcrzo_constipopt ipopt,
					 lcrzo_uint8 ipoptsize,
					 lcrzo_uint8 code,
					 lcrzo_ipl gw,
					 lcrzo_constdata ippacket, 
					 lcrzo_uint16 ippacketsize,
					 lcrzo_data *pdataout,
					 lcrzo_int32 *pdataoutsize);
int lcrzo_packet_initm_ethipopticmpredirect(lcrzo_hdrleth hdrleth,
					    lcrzo_hdrlip hdrlip,
					    lcrzo_constipopt ipopt,
					    lcrzo_uint8 ipoptsize,
					    lcrzo_uint8 code,
					    lcrzo_ipl gw,
					    lcrzo_constdata ippacket, 
					    lcrzo_uint16 ippacketsize,
					    lcrzo_data *pdataout,
					    lcrzo_int32 *pdataoutsize);

/*-------------------------------------------------------------*/
/*en idem for Echo (cf. rfc 792) */
/*fr idem avec Echo (cf. rfc 792) */
#define LCRZO_ICMPTYPE_ECHOREQ      8
#define LCRZO_ICMPTYPE_ECHOREP      0
int lcrzo_packet_initm_icmpecho(lcrzo_uint8 type,
				lcrzo_uint16 id,
				lcrzo_uint16 seqnum,
				lcrzo_constdata echodata, 
				lcrzo_uint16 echodatasize,
			        lcrzo_data *pdataout,
				lcrzo_int32 *pdataoutsize);
int lcrzo_packet_initm_ipopticmpecho(lcrzo_hdrlip hdrlip,
				     lcrzo_constipopt ipopt,
				     lcrzo_uint8 ipoptsize,
				     lcrzo_uint8 type,
				     lcrzo_uint16 id,
				     lcrzo_uint16 seqnum,
				     lcrzo_constdata echodata, 
				     lcrzo_uint16 echodatasize,
				     lcrzo_data *pdataout,
				     lcrzo_int32 *pdataoutsize);
int lcrzo_packet_initm_ethipopticmpecho(lcrzo_hdrleth hdrleth,
					lcrzo_hdrlip hdrlip,
					lcrzo_constipopt ipopt,
					lcrzo_uint8 ipoptsize,
					lcrzo_uint8 type,
					lcrzo_uint16 id,
					lcrzo_uint16 seqnum,
					lcrzo_constdata echodata, 
					lcrzo_uint16 echodatasize,
					lcrzo_data *pdataout,
					lcrzo_int32 *pdataoutsize);

/*-------------------------------------------------------------*/
/*en idem for Timestamp (cf. rfc 792) */
/*fr idem avec Timestamp (cf. rfc 792) */
#define LCRZO_ICMPTYPE_TIMESTAMPREQ  13 
#define LCRZO_ICMPTYPE_TIMESTAMPREP  14 
int lcrzo_packet_initm_icmptimestamp(lcrzo_uint8 type,
				     lcrzo_uint16 id,
				     lcrzo_uint16 seqnum,
				     lcrzo_uint32 originatetimestamp,
				     lcrzo_uint32 receivetimestamp,
				     lcrzo_uint32 transmittimestamp,
				     lcrzo_data *pdataout,
				     lcrzo_int32 *pdataoutsize);
int lcrzo_packet_initm_ipopticmptimestamp(lcrzo_hdrlip hdrlip,
					  lcrzo_constipopt ipopt,
					  lcrzo_uint8 ipoptsize,
					  lcrzo_uint8 type,
					  lcrzo_uint16 id,
					  lcrzo_uint16 seqnum,
					  lcrzo_uint32 originatetimestamp,
					  lcrzo_uint32 receivetimestamp,
					  lcrzo_uint32 transmittimestamp,
					  lcrzo_data *pdataout,
					  lcrzo_int32 *pdataoutsize);
int lcrzo_packet_initm_ethipopticmptimestamp(lcrzo_hdrleth hdrleth,
					     lcrzo_hdrlip hdrlip,
					     lcrzo_constipopt ipopt,
					     lcrzo_uint8 ipoptsize,
					     lcrzo_uint8 type,
					     lcrzo_uint16 id,
					     lcrzo_uint16 seqnum,
					     lcrzo_uint32 originatetimestamp,
					     lcrzo_uint32 receivetimestamp,
					     lcrzo_uint32 transmittimestamp,
					     lcrzo_data *pdataout,
					     lcrzo_int32 *pdataoutsize);

/*-------------------------------------------------------------*/
/*en idem for Information (cf. rfc 792) */
/*fr idem avec Information (cf. rfc 792) */
#define LCRZO_ICMPTYPE_INFOREQ     15
#define LCRZO_ICMPTYPE_INFOREP     16
int lcrzo_packet_initm_icmpinfo(lcrzo_uint8 type,
				lcrzo_uint16 id,
				lcrzo_uint16 seqnum,
				lcrzo_data *pdataout,
				lcrzo_int32 *pdataoutsize);
int lcrzo_packet_initm_ipopticmpinfo(lcrzo_hdrlip hdrlip,
				     lcrzo_constipopt ipopt,
				     lcrzo_uint8 ipoptsize,
				     lcrzo_uint8 type,
				     lcrzo_uint16 id,
				     lcrzo_uint16 seqnum,
				     lcrzo_data *pdataout,
				     lcrzo_int32 *pdataoutsize);
int lcrzo_packet_initm_ethipopticmpinfo(lcrzo_hdrleth hdrleth,
					lcrzo_hdrlip hdrlip,
					lcrzo_constipopt ipopt,
					lcrzo_uint8 ipoptsize,
					lcrzo_uint8 type,
					lcrzo_uint16 id,
					lcrzo_uint16 seqnum,
					lcrzo_data *pdataout,
					lcrzo_int32 *pdataoutsize);


/*-------------------------------------------------------------*/
/*-------------------------------------------------------------*/
/*en Name : lcrzo_packet_decodem_xxxicmpdestunreach
   Description :
     Check if a packet contains {ICMP header, DestUnreachable}.
     Check if a packet contains {IP header, ICMP header, 
     DestUnreachable}.
     Check if a packet contains {Ethernet header, IP header, 
     ICMP header, DestUnreachable}.
   Input parameter(s) :
     xxpacket : packet to check
     xxpacketsize : size of packet
   Input/output parameter(s) :
   Output parameter(s) :
     *phdreth : Ethernet header
     *phdrlip : IP header
     ipopt : IP options
     *pipoptsize : size of ipopt
     *pcode : icmp code
     pbadpacket : ip packet which generated this ICMP error
     pbadpacketsize : size of ippacket
   Normal return values :
     LCRZO_ERR_OK : packet decoded
     LCRZO_ERR_OKNOTDECODED : packet not decoded
*/
/*fr Nom : lcrzo_packet_decodem_xxxicmpdestunreach
   Description :
     Verifie si un packet contient {entete ICMP, DestUnreachable}.
     Verifie si un packet contient {entete IP, entete ICMP, 
     DestUnreachable}.
     Verifie si un packet contient {entete Ethernet, entete IP, 
     entete ICMP, DestUnreachable}.
   Parametre(s) d'entree :
     xxpacket : paquet a verifier
     xxpacketsize : taille de packet
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *phdrleth : entete Ethernet
     *phdrlip : entete IP
     ipopt : options IP
     *pipoptsize : taille de ipopt
     *pcode : code icmp
     *pbadpacket : paquet IP ayant genere cette erreur ICMP
     *pbadpacketsize : taille de ippacket
   Valeurs de retour normales :
     LCRZO_ERR_OK : paquet decode
     LCRZO_ERR_OKNOTDECODED : paquet non decode
*/
int lcrzo_packet_decodem_icmpdestunreach(lcrzo_constdata icmppacket,
					 lcrzo_int32 icmppacketsize,
					 lcrzo_uint8 *pcode,
					 lcrzo_data *pbadpacket, 
					 lcrzo_uint16 *pbadpacketsize);

int lcrzo_packet_decodem_ipopticmpdestunreach(lcrzo_constdata ippacket,
					      lcrzo_int32 ippacketsize,
					      lcrzo_hdrlip *phdrlip,
					      lcrzo_ipopt ipopt,
					      lcrzo_uint8 *pipoptsize,
					      lcrzo_uint8 *pcode,
					      lcrzo_data *pbadpacket, 
					      lcrzo_uint16 *pbadpacketsize);
int lcrzo_packet_decodem_ethipopticmpdestunreach(lcrzo_constdata ethpacket,
						 lcrzo_int32 ethpacketsize,
						 lcrzo_hdrleth *phdrleth,
						 lcrzo_hdrlip *phdrlip,
						 lcrzo_ipopt ipopt,
						 lcrzo_uint8 *pipoptsize,
						 lcrzo_uint8 *pcode,
						 lcrzo_data *pbadpacket, 
						 lcrzo_uint16 *pbadpacketsize);

/*-------------------------------------------------------------*/
/*en idem for TimeExceeded (cf. rfc 792 for a field explanation) */
/*fr idem avec TimeExceeded (cf. rfc 792 pour une description 
  des champs) */
int lcrzo_packet_decodem_icmptimeexceed(lcrzo_constdata icmppacket,
					lcrzo_int32 icmppacketsize,
					lcrzo_uint8 *pcode,
					lcrzo_data *pbadpacket, 
					lcrzo_uint16 *pbadpacketsize);

int lcrzo_packet_decodem_ipopticmptimeexceed(lcrzo_constdata ippacket,
					     lcrzo_int32 ippacketsize,
					     lcrzo_hdrlip *phdrlip,
					     lcrzo_ipopt ipopt,
					     lcrzo_uint8 *pipoptsize,
					     lcrzo_uint8 *pcode,
					     lcrzo_data *pbadpacket, 
					     lcrzo_uint16 *pbadpacketsize);
int lcrzo_packet_decodem_ethipopticmptimeexceed(lcrzo_constdata ethpacket,
						lcrzo_int32 ethpacketsize,
						lcrzo_hdrleth *phdrleth,
						lcrzo_hdrlip *phdrlip,
						lcrzo_ipopt ipopt,
						lcrzo_uint8 *pipoptsize,
						lcrzo_uint8 *pcode,
						lcrzo_data *pbadpacket, 
						lcrzo_uint16 *pbadpacketsize);

/*-------------------------------------------------------------*/
/*en idem for ParameterProblem (cf. rfc 792 for a field explanation) */
/*fr idem avec ParameterProblem (cf. rfc 792 pour une description 
  des champs) */
int lcrzo_packet_decodem_icmpparaprob(lcrzo_constdata icmppacket,
				      lcrzo_int32 icmppacketsize,
				      lcrzo_uint8 *pcode,
				      lcrzo_uint8 *ppointer,
				      lcrzo_data *pbadpacket, 
				      lcrzo_uint16 *pbadpacketsize);

int lcrzo_packet_decodem_ipopticmpparaprob(lcrzo_constdata ippacket,
					   lcrzo_int32 ippacketsize,
					   lcrzo_hdrlip *phdrlip,
					   lcrzo_ipopt ipopt,
					   lcrzo_uint8 *pipoptsize,
					   lcrzo_uint8 *pcode,
					   lcrzo_uint8 *ppointer,
					   lcrzo_data *pbadpacket, 
					   lcrzo_uint16 *pbadpacketsize);
int lcrzo_packet_decodem_ethipopticmpparaprob(lcrzo_constdata ethpacket,
					      lcrzo_int32 ethpacketsize,
					      lcrzo_hdrleth *phdrleth,
					      lcrzo_hdrlip *phdrlip,
					      lcrzo_ipopt ipopt,
					      lcrzo_uint8 *pipoptsize,
					      lcrzo_uint8 *pcode,
					      lcrzo_uint8 *ppointer,
					      lcrzo_data *pbadpacket, 
					      lcrzo_uint16 *pbadpacketsize);

/*-------------------------------------------------------------*/
/*en idem for SourceQuench (cf. rfc 792 for a field explanation) */
/*fr idem avec SourceQuench (cf. rfc 792 pour une description 
  des champs) */
int lcrzo_packet_decodem_icmpsrcquench(lcrzo_constdata icmppacket,
				       lcrzo_int32 icmppacketsize,
				       lcrzo_data *pbadpacket, 
				       lcrzo_uint16 *pbadpacketsize);

int lcrzo_packet_decodem_ipopticmpsrcquench(lcrzo_constdata ippacket,
					    lcrzo_int32 ippacketsize,
					    lcrzo_hdrlip *phdrlip,
					    lcrzo_ipopt ipopt,
					    lcrzo_uint8 *pipoptsize,
					    lcrzo_data *pbadpacket, 
					    lcrzo_uint16 *pbadpacketsize);
int lcrzo_packet_decodem_ethipopticmpsrcquench(lcrzo_constdata ethpacket,
					       lcrzo_int32 ethpacketsize,
					       lcrzo_hdrleth *phdrleth,
					       lcrzo_hdrlip *phdrlip,
					       lcrzo_ipopt ipopt,
					       lcrzo_uint8 *pipoptsize,
					       lcrzo_data *pbadpacket, 
					       lcrzo_uint16 *pbadpacketsize);

/*-------------------------------------------------------------*/
/*en idem for Redirect (cf. rfc 792 for a field explanation) */
/*fr idem avec Redirect (cf. rfc 792 pour une description 
  des champs) */
int lcrzo_packet_decodem_icmpredirect(lcrzo_constdata icmppacket,
				      lcrzo_int32 icmppacketsize,
				      lcrzo_uint8 *pcode,
				      lcrzo_ipl *pgw,
				      lcrzo_data *pbadpacket, 
				      lcrzo_uint16 *pbadpacketsize);

int lcrzo_packet_decodem_ipopticmpredirect(lcrzo_constdata ippacket,
					   lcrzo_int32 ippacketsize,
					   lcrzo_hdrlip *phdrlip,
					   lcrzo_ipopt ipopt,
					   lcrzo_uint8 *pipoptsize,
					   lcrzo_uint8 *pcode,
					   lcrzo_ipl *pgw,
					   lcrzo_data *pbadpacket, 
					   lcrzo_uint16 *pbadpacketsize);
int lcrzo_packet_decodem_ethipopticmpredirect(lcrzo_constdata ethpacket,
					      lcrzo_int32 ethpacketsize,
					      lcrzo_hdrleth *phdrleth,
					      lcrzo_hdrlip *phdrlip,
					      lcrzo_ipopt ipopt,
					      lcrzo_uint8 *pipoptsize,
					      lcrzo_uint8 *pcode,
					      lcrzo_ipl *pgw,
					      lcrzo_data *pbadpacket, 
					      lcrzo_uint16 *pbadpacketsize);

/*-------------------------------------------------------------*/
/*en idem for Echo (cf. rfc 792 for a field explanation) */
/*fr idem avec Echo (cf. rfc 792 pour une description 
  des champs) */
int lcrzo_packet_decodem_icmpecho(lcrzo_constdata icmppacket,
				  lcrzo_int32 icmppacketsize,
				  lcrzo_uint8 *ptype,
				  lcrzo_uint16 *pid,
				  lcrzo_uint16 *pseqnum,
				  lcrzo_data *pechodata, 
				  lcrzo_uint16 *pechodatasize);

int lcrzo_packet_decodem_ipopticmpecho(lcrzo_constdata ippacket,
				       lcrzo_int32 ippacketsize,
				       lcrzo_hdrlip *phdrlip,
				       lcrzo_ipopt ipopt,
				       lcrzo_uint8 *pipoptsize,
				       lcrzo_uint8 *ptype,
				       lcrzo_uint16 *pid,
				       lcrzo_uint16 *pseqnum,
				       lcrzo_data *pechodata, 
				       lcrzo_uint16 *pechodatasize);

int lcrzo_packet_decodem_ethipopticmpecho(lcrzo_constdata ethpacket,
					  lcrzo_int32 ethpacketsize,
					  lcrzo_hdrleth *phdrleth,
					  lcrzo_hdrlip *phdrlip,
					  lcrzo_ipopt ipopt,
					  lcrzo_uint8 *pipoptsize,
					  lcrzo_uint8 *ptype,
					  lcrzo_uint16 *pid,
					  lcrzo_uint16 *pseqnum,
					  lcrzo_data *pechodata, 
					  lcrzo_uint16 *pechodatasize);

/*-------------------------------------------------------------*/
/*en idem for Timestamp (cf. rfc 792 for a field explanation) */
/*fr idem avec Timestamp (cf. rfc 792 pour une description 
  des champs) */
int lcrzo_packet_decodem_icmptimestamp(lcrzo_constdata icmppacket,
				       lcrzo_int32 icmppacketsize,
				       lcrzo_uint8 *ptype,
				       lcrzo_uint16 *pid,
				       lcrzo_uint16 *pseqnum,
				       lcrzo_uint32 *poriginatetimestamp,
				       lcrzo_uint32 *preceivetimestamp,
				       lcrzo_uint32 *ptransmittimestamp);

int lcrzo_packet_decodem_ipopticmptimestamp(lcrzo_constdata ippacket,
					    lcrzo_int32 ippacketsize,
					    lcrzo_hdrlip *phdrlip,
					    lcrzo_ipopt ipopt,
					    lcrzo_uint8 *pipoptsize,
					    lcrzo_uint8 *ptype,
					    lcrzo_uint16 *pid,
					    lcrzo_uint16 *pseqnum,
					    lcrzo_uint32 *poriginatetimestamp,
					    lcrzo_uint32 *preceivetimestamp,
					    lcrzo_uint32 *ptransmittimestamp);

int lcrzo_packet_decodem_ethipopticmptimestamp(lcrzo_constdata ethpacket,
					       lcrzo_int32 ethpacketsize,
					       lcrzo_hdrleth *phdrleth,
					       lcrzo_hdrlip *phdrlip,
					       lcrzo_ipopt ipopt,
					       lcrzo_uint8 *pipoptsize,
					       lcrzo_uint8 *ptype,
					       lcrzo_uint16 *pid,
					       lcrzo_uint16 *pseqnum,
					     lcrzo_uint32 *poriginatetimestamp,
					     lcrzo_uint32 *preceivetimestamp,
				             lcrzo_uint32 *ptransmittimestamp);

/*-------------------------------------------------------------*/
/*en idem for Info (cf. rfc 792 for a field explanation) */
/*fr idem avec Info (cf. rfc 792 pour une description 
  des champs) */
int lcrzo_packet_decodem_icmpinfo(lcrzo_constdata icmppacket,
				  lcrzo_int32 icmppacketsize,
				  lcrzo_uint8 *ptype,
				  lcrzo_uint16 *pid,
				  lcrzo_uint16 *pseqnum);

int lcrzo_packet_decodem_ipopticmpinfo(lcrzo_constdata ippacket,
				       lcrzo_int32 ippacketsize,
				       lcrzo_hdrlip *phdrlip,
				       lcrzo_ipopt ipopt,
				       lcrzo_uint8 *pipoptsize,
				       lcrzo_uint8 *ptype,
				       lcrzo_uint16 *pid,
				       lcrzo_uint16 *pseqnum);

int lcrzo_packet_decodem_ethipopticmpinfo(lcrzo_constdata ethpacket,
					  lcrzo_int32 ethpacketsize,
					  lcrzo_hdrleth *phdrleth,
					  lcrzo_hdrlip *phdrlip,
					  lcrzo_ipopt ipopt,
					  lcrzo_uint8 *pipoptsize,
					  lcrzo_uint8 *ptype,
					  lcrzo_uint16 *pid,
					  lcrzo_uint16 *pseqnum);

/*-------------------------------------------------------------*/
/*en These functions try to decode errors :
   LCRZO_ICMPTYPE_DESTUNREACH
   LCRZO_ICMPTYPE_TIMEEXCEED
   LCRZO_ICMPTYPE_PARAPROB
   LCRZO_ICMPTYPE_SRCQUENCH
   LCRZO_ICMPTYPE_REDIRECT
 Output parameters represents needed parameters for each type.
*/
/*fr Ces fonctions essaient de decoder des erreurs :
   LCRZO_ICMPTYPE_DESTUNREACH
   LCRZO_ICMPTYPE_TIMEEXCEED
   LCRZO_ICMPTYPE_PARAPROB
   LCRZO_ICMPTYPE_SRCQUENCH
   LCRZO_ICMPTYPE_REDIRECT
 Les parametres de sortie sont l'union de tous les parametres de 
 chacun des types.
*/
int lcrzo_packet_decodem_icmperror(lcrzo_constdata icmppacket,
				   lcrzo_int32 icmppacketsize,
				   lcrzo_uint8 *ptype,
				   lcrzo_uint8 *pcode,
				   lcrzo_uint8 *ppointer,
				   lcrzo_ipl *pgw,
				   lcrzo_data *pbadpacket, 
				   lcrzo_uint16 *pbadpacketsize);

int lcrzo_packet_decodem_ipopticmperror(lcrzo_constdata ippacket,
					lcrzo_int32 ippacketsize,
					lcrzo_hdrlip *phdrlip,
					lcrzo_ipopt ipopt,
					lcrzo_uint8 *pipoptsize,
					lcrzo_uint8 *ptype,
					lcrzo_uint8 *pcode,
					lcrzo_uint8 *ppointer,
					lcrzo_ipl *pgw,
					lcrzo_data *pbadpacket, 
					lcrzo_uint16 *pbadpacketsize);
int lcrzo_packet_decodem_ethipopticmperror(lcrzo_constdata ethpacket,
					   lcrzo_int32 ethpacketsize,
					   lcrzo_hdrleth *phdrleth,
					   lcrzo_hdrlip *phdrlip,
					   lcrzo_ipopt ipopt,
					   lcrzo_uint8 *pipoptsize,
					   lcrzo_uint8 *ptype,
					   lcrzo_uint8 *pcode,
					   lcrzo_uint8 *ppointer,
					   lcrzo_ipl *pgw,
					   lcrzo_data *pbadpacket, 
					   lcrzo_uint16 *pbadpacketsize);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_ip64bits_initm_xxx
   Description :
     Initialize a packet an IP header an 64 bits of data.
     This kind of packets are needed for errors (cf. rfc 792).
   Input parameter(s) :
     hdrlip : IP header
     ipdata : IP data
     ipdatasize : size of ipdata
     hdrudp : udp header
     hdrtcp : tcp header
     hdrlicmp : icmp header
     icmpbyte4..7 : four bytes of icmp data (because icmp header 
                    has only 4bytes=32bits)
   Input/output parameter(s) :
   Output parameter(s) :
     *pdataout : output array, which is allocated.
     *pdataoutsize : reached size of *pdataout
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ip64bits_initm_xxx
   Description :
     Initialise un paquet avec un entete IP et 64 bits de 
     donnees.
     Ce type de paquet est employe pour les erreurs (cf. rfc 792).
   Parametre(s) d'entree :
     hdrlip : entete IP
     ipdata : donnees IP
     ipdatasize : taille de ipdata
     hdrudp : entete udp
     hdrtcp : entete tcp
     hdrlicmp : entete icmp
     icmpbyte4..7 : quatre octets de donnees icmp (car un 
                    entete icmp n'a que 4octets=32bits)
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pdataout : tableau de sortie qui est alloue
     *pdataoutsize : taille atteinte par *pdataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ip64bits_initm_ipdata(lcrzo_hdrlip hdrip,
				lcrzo_constdata ipdata, 
				lcrzo_uint16 ipdatasize,
				lcrzo_data *pdataout,
				lcrzo_int32 *pdataoutsize);
int lcrzo_ip64bits_initm_ipudp(lcrzo_hdrlip hdrip,
			       lcrzo_hdrludp hdrudp,
			       lcrzo_data *pdataout,
			       lcrzo_int32 *pdataoutsize);
int lcrzo_ip64bits_initm_iptcp(lcrzo_hdrlip hdrip,
			       lcrzo_hdrltcp hdrtcp,
			       lcrzo_data *pdataout,
			       lcrzo_int32 *pdataoutsize);
int lcrzo_ip64bits_initm_ipicmp(lcrzo_hdrlip hdrlip,
				lcrzo_hdrlicmp hdrlicmp,
				lcrzo_uint8 icmpbyte4,	
				lcrzo_uint8 icmpbyte5,
				lcrzo_uint8 icmpbyte6,
				lcrzo_uint8 icmpbyte7,
				lcrzo_data *pdataout,
				lcrzo_int32 *pdataoutsize);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_ip64bits_decodem_ipdata
   Description :
     Try to decode ip packet contained in ip64bits.
   Input parameter(s) :
     ip64bits : ip header and at least 64 bits of ip data
     ip64bitssize : size of ip64bits
   Input/output parameter(s) :
   Output parameter(s) :
     *phdrip : decoded IP header
     *pipdata : array, which is allocated, containing ip data
     *pipdatasize : reached size of *pipdata
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKNOTDECODED : packet not decoded
*/
/*fr Nom : lcrzo_ip64bits_decodem_ipdata
   Description :
     Essaie de decoder un paquet IP contenu dans ip64bits.
   Parametre(s) d'entree :
     ip64bits : entete IP and au moins 64 bits de donnees IP
     ip64bitssize : taille de ip64bits
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *phdrip : entete IP decode
     *pipdata : tableau qui est alloue et contient les donnees IP
     *pipdatasize : taille atteinte par *pipdata
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKNOTDECODED : paquet non decode
*/
int lcrzo_ip64bits_decodem_ipdata(lcrzo_constdata ip64bits,
				  lcrzo_int32 ip64bitssize,
				  lcrzo_hdrlip *phdrip,
				  lcrzo_data *pipdata, 
				  lcrzo_uint16 *pipdatasize);

/*en Name : lcrzo_ip64bits_decodem_ipudp
   Description :
     Try to decode ip+udp packet contained in ip64bits.
   Input parameter(s) :
     ip64bits : ip header and at least 64 bits of ip data
     ip64bitssize : size of ip64bits
   Input/output parameter(s) :
   Output parameter(s) :
     *phdrip : decoded IP header
     *pudpsport : udp source port
     *pudpdport : udp destination port
     *pudplen : udp len
     *pudpcheck : udp checksum
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKNOTDECODED : packet not decoded
*/
/*fr Nom : lcrzo_ip64bits_decodem_ipudp
   Description :
     Essaie de decoder un paquet IP+UDP contenu dans ip64bits.
   Parametre(s) d'entree :
     ip64bits : entete IP and au moins 64 bits de donnees IP
     ip64bitssize : taille de ip64bits
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *phdrip : entete IP decode
     *pudpsport : port source udp
     *pudpdport : port destination udp
     *pudplen : len udp
     *pudpcheck : checksum udp
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKNOTDECODED : paquet non decode
*/
int lcrzo_ip64bits_decodem_ipudp(lcrzo_constdata ip64bits,
				 lcrzo_int32 ip64bitssize,
				 lcrzo_hdrlip *phdrip,
				 lcrzo_uint16 *pudpsport,
				 lcrzo_uint16 *pudpdport,
				 lcrzo_uint16 *pudplen,
				 lcrzo_uint16 *pudpcheck);

/*en Name : lcrzo_ip64bits_decodem_iptcp
   Description :
     Try to decode ip+tcp packet contained in ip64bits.
   Input parameter(s) :
     ip64bits : ip header and at least 64 bits of ip data
     ip64bitssize : size of ip64bits
   Input/output parameter(s) :
   Output parameter(s) :
     *phdrip : decoded IP header
     *ptcpsport : tcp source port
     *ptcpdport : tcp destination port
     *pseqnum : tcp sequence number
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKNOTDECODED : packet not decoded
*/
/*fr Nom : lcrzo_ip64bits_decodem_iptcp
   Description :
     Essaie de decoder un paquet IP+TCP contenu dans ip64bits.
   Parametre(s) d'entree :
     ip64bits : entete IP and au moins 64 bits de donnees IP
     ip64bitssize : taille de ip64bits
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *phdrip : entete IP decode
     *ptcpsport : port source tcp
     *ptcpdport : port destination tcp
     *pseqnum : numero de sequence tcp
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKNOTDECODED : paquet non decode
*/
int lcrzo_ip64bits_decodem_iptcp(lcrzo_constdata ip64bits,
				 lcrzo_int32 ip64bitssize,
				 lcrzo_hdrlip *phdrip,
				 lcrzo_uint16 *ptcpsport,
				 lcrzo_uint16 *ptcpdport,
				 lcrzo_uint32 *ptcpseqnum);

/*en Name : lcrzo_ip64bits_decodem_ipicmp
   Description :
     Try to decode ip+icmp packet contained in ip64bits.
   Input parameter(s) :
     ip64bits : ip header and at least 64 bits of ip data
     ip64bitssize : size of ip64bits
   Input/output parameter(s) :
   Output parameter(s) :
     *phdrip : decoded IP header
     *picmptype : icmp type
     *picmpcode : icmp code
     *picmpcheck : icmp checksum
     *picmpbyte4..7 : four bytes following ICMP header
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKNOTDECODED : packet not decoded
*/
/*fr Nom : lcrzo_ip64bits_decodem_ipicmp
   Description :
     Essaie de decoder un paquet IP+ICMP contenu dans ip64bits.
   Parametre(s) d'entree :
     ip64bits : entete IP and au moins 64 bits de donnees IP
     ip64bitssize : taille de ip64bits
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *phdrip : entete IP decode
     *picmptype : type icmp
     *picmpcode : code icmp
     *picmpcheck : checksum icmp
     *picmpbyte4..7 : quatre octets qui suivent l'entete ICMP
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKNOTDECODED : paquet non decode
*/
int lcrzo_ip64bits_decodem_ipicmp(lcrzo_constdata ip64bits,
				  lcrzo_int32 ip64bitssize,
				  lcrzo_hdrlip *phdrip,
				  lcrzo_uint8 *picmptype,
				  lcrzo_uint8 *picmpcode,
				  lcrzo_uint16 *picmpcheck,
				  lcrzo_uint8 *picmpbyte4,	
				  lcrzo_uint8 *picmpbyte5,
				  lcrzo_uint8 *picmpbyte6,
				  lcrzo_uint8 *picmpbyte7);

/*-------------------------------------------------------------*/
int lcrzo_string_initm_icmptypecode(lcrzo_uint8 type, 
				    lcrzo_uint8 code,
				    lcrzo_string *pstring);

/*-------------------------------------------------------------*/
/* supplementary ICMP types (defined, but corresponding functions
   are not yet implemented) */
#define LCRZO_ICMPTYPE_ALTHOSTAD     6 /* www.iana.org */
#define LCRZO_ICMPTYPE_ROUTERADVERT  9 /* rfc 1256 */
#define LCRZO_ICMPTYPE_ROUTERSELECT 10 /* rfc 1256 */
#define LCRZO_ICMPTYPE_ADMASKREQ    17 /* rfc 950 */
#define LCRZO_ICMPTYPE_ADMASKREP    18 /* rfc 950 */
#define LCRZO_ICMPTYPE_TRACEROUTE   30 /* rfc 1393 */
#define LCRZO_ICMPTYPE_DATACONVERR  31 /* rfc 1474 */
#define LCRZO_ICMPTYPE_MOBHOSTREDIR 32 /* www.iana.org */
#define LCRZO_ICMPTYPE_IPV6WHEREYOU 33 /* www.iana.org */
#define LCRZO_ICMPTYPE_IPV6IAMHERE  34 /* www.iana.org */
#define LCRZO_ICMPTYPE_MOBREGREQ    35 /* www.iana.org */
#define LCRZO_ICMPTYPE_MOBREGREP    36 /* www.iana.org */
#define LCRZO_ICMPTYPE_SKIP         39 /* www.iana.org */
#define LCRZO_ICMPTYPE_PHOTURIS     40 /* www.iana.org */

