/*en    Laurent Constantin's network library (lcrzo)
                 lcrzo_ipc module

  Functions herein allow two process to communicate.
*/
/*fr    Bibliotheque reseau de Laurent Constantin (lcrzo)
                 Module lcrzo_ipc

  Les fonctions presentes dans ce module permettent de communiquer
des donnees entres deux processus.
*/

/*EN*********************************************************
 * Note about an ipc :                                      *
 * A lcrzo_ipc is used to transmit data between two         *
 * processes.                                               *
 * For example, the first process can write an integer into *
 * the "tube", and the other process will read it.          *
 ************************************************************/
/*FR*********************************************************
 * Note sur les ipc :                                       *
 * Un lcrzo_ipc est utilise pour transmettre des donnees    *
 * entre deux processus.                                    *
 * Par exemple, le premier processus peut ecrire un entier  *
 * dans le "tube", et l'autre processus peut le lire.       *
 ************************************************************/

/*-------------------------------------------------------------*/
/*en definition of a lcrzo_ipc. You should not use its
   fields directly. */
/*fr definition d'un lcrzo_ipc. Vous ne devriez pas utiliser
   directement ses champs. */
typedef struct
{ lcrzo_int16 side;
  int readpipe[2];
  int writepipe[2];
  void *readhandle[2];
  void *writehandle[2];
  lcrzo_bool openforreading;
  lcrzo_bool openforwriting;
  lcrzo_bool dataend;
  lcrzo_data storeddata;
  lcrzo_int32 storeddatasize;
} lcrzo_ipc;

/*-------------------------------------------------------------*/
/*en Name : lcrzo_ipc_init
   Description :
     Initialize a lcrzo_ipc.
   Input parameter(s) :
   Input/output parameter(s) :
   Output parameter(s) :
     *pipc : lcrzo_ipc initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ipc_init
   Description :
     Initialise un lcrzo_ipc
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pipc : lcrzo_ipc initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ipc_init(lcrzo_ipc *pipc);

/*en A is the father process, and B the child */
/*fr A est le processus pere, et B le fils */
int lcrzo_ipc_currentprocessissidea(lcrzo_ipc *pipc);
int lcrzo_ipc_currentprocessissideb(lcrzo_ipc *pipc);

/*en Name : lcrzo_ipc_close_xxx
   Description :
     Close a lcrzo_ipc.
     Close the reading descriptor of a lcrzo_ipc.
     Close the writing descriptor of a lcrzo_ipc.
   Input parameter(s) :
     ipc : lcrzo_ipc to close
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ipc_close_xxx
   Description :
     Ferme un lcrzo_ipc.
     Ferme le descripteur de lecture d'un lcrzo_ipc.
     Ferme le descripteur d'ecriture d'un lcrzo_ipc.
   Parametre(s) d'entree :
     ipc : lcrzo_ipc a fermer
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ipc_close(lcrzo_ipc *pipc);
int lcrzo_ipc_close_read(lcrzo_ipc *pipc);
int lcrzo_ipc_close_write(lcrzo_ipc *pipc);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_ipc_write_uint8
   Description :
     Write an lcrzo_uint8 in a lcrzo_ipc tube.
   Input parameter(s) :
     ipc : lcrzo_ipc where the data is written
     uint8 : lcrzo_uint8 to write in ipc
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ipc_write_uint8
   Description :
     Ecrit un lcrzo_uint8 dans un tube lcrzo_ipc
   Parametre(s) d'entree :
     ipc : lcrzo_ipc ou les donnees sont ecrites
     uint8 : lcrzo_uint8 a ecrire dans ipc
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ipc_write_uint8(lcrzo_ipc *pipc,
			  lcrzo_uint8 uint8);

/*en Name : lcrzo_ipc_read_uint8
   Description :
     Read an lcrzo_uint8 from a lcrzo_ipc tube.
   Input parameter(s) :
     ipc : lcrzo_ipc from where the datas are read
     *pmaxwaitingtime : maximum waiting time (can be a lcrzo_time,
                        LCRZO_WAIT_NODELAY or LCRZO_WAIT_INFINITE)
   Input/output parameter(s) :
   Output parameter(s) :
     *puint8 : lcrzo_uint8 read from the ipc
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : the function received nothing
                               before pmaxwaitingtime, so it 
			       exited without reading data
*/
/*fr Nom : lcrzo_ipc_read_uint8
   Description :
     Lit un lcrzo_uint8 depuis un tube lcrzo_ipc
   Parametre(s) d'entree :
     ipc : lcrzo_ipc ou les donnees sont lues
     *pmaxwaitingtime : temps maximum d'attente (peut etre un
                        lcrzo_time, LCRZO_WAIT_NODELAY ou
			LCRZO_WAIT_INFINITE)
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *puint8 : lcrzo_uint8 lu depuis ipc
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : la fonction n'a pas recu de donnees
                               avant pmaxwaitingtime, donc on 
                               retourne sans donnees.
*/
int lcrzo_ipc_read_uint8(lcrzo_ipc *pipc, 
			 lcrzo_time *pmaxwaitingtime, 
			 lcrzo_uint8 *puint8);

/*en idem, but work on a lcrzo_uint16 */
/*fr idem, mais travaille sur un lcrzo_uint16 */
int lcrzo_ipc_write_uint16(lcrzo_ipc *pipc,
			   lcrzo_uint16 uint16);
int lcrzo_ipc_read_uint16(lcrzo_ipc *pipc, 
			  lcrzo_time *pmaxwaitingtime, 
			  lcrzo_uint16 *puint16);

/*en idem, but work on a lcrzo_uint32 */
/*fr idem, mais travaille sur un lcrzo_uint32 */
int lcrzo_ipc_write_uint32(lcrzo_ipc *pipc,
			   lcrzo_uint32 uint32);
int lcrzo_ipc_read_uint32(lcrzo_ipc *pipc, 
			  lcrzo_time *pmaxwaitingtime, 
			  lcrzo_uint32 *puint32);

/*en idem, but work on a lcrzo_hs */
/*fr idem, mais travaille sur un lcrzo_hs */
int lcrzo_ipc_write_hs(lcrzo_ipc *pipc,
		       lcrzo_consths hs);
int lcrzo_ipc_read_hs(lcrzo_ipc *pipc, 
		      lcrzo_time *pmaxwaitingtime, 
		      lcrzo_hs hs);

/*en idem, but work on a lcrzo_ips */
/*fr idem, mais travaille sur un lcrzo_ips */
int lcrzo_ipc_write_ips(lcrzo_ipc *pipc,
			lcrzo_constips ips);
int lcrzo_ipc_read_ips(lcrzo_ipc *pipc, 
		       lcrzo_time *pmaxwaitingtime, 
		       lcrzo_ips ips);

/*en idem, but work on a lcrzo_ipa */
/*fr idem, mais travaille sur un lcrzo_ipa */
int lcrzo_ipc_write_ipa(lcrzo_ipc *pipc,
			lcrzo_constipa ipa);
int lcrzo_ipc_read_ipa(lcrzo_ipc *pipc, 
		       lcrzo_time *pmaxwaitingtime, 
		       lcrzo_ipa ipa);

/*en idem, but work on a lcrzo_ipl */
/*fr idem, mais travaille sur un lcrzo_ipl */
int lcrzo_ipc_write_ipl(lcrzo_ipc *pipc,
			lcrzo_ipl ipl);
int lcrzo_ipc_read_ipl(lcrzo_ipc *pipc, 
		       lcrzo_time *pmaxwaitingtime, 
		       lcrzo_ipl *pipl);

/*en idem, but work on a lcrzo_eths */
/*fr idem, mais travaille sur un lcrzo_eths */
int lcrzo_ipc_write_eths(lcrzo_ipc *pipc,
			 lcrzo_consteths eths);
int lcrzo_ipc_read_eths(lcrzo_ipc *pipc, 
			lcrzo_time *pmaxwaitingtime, 
			lcrzo_eths eths);

/*en idem, but work on a lcrzo_etha */
/*fr idem, mais travaille sur un lcrzo_etha */
int lcrzo_ipc_write_etha(lcrzo_ipc *pipc,
			 lcrzo_constetha etha);
int lcrzo_ipc_read_etha(lcrzo_ipc *pipc, 
			lcrzo_time *pmaxwaitingtime, 
			lcrzo_etha etha);

/*en idem, but work on a lcrzo_device */
/*fr idem, mais travaille sur un lcrzo_device */
int lcrzo_ipc_write_device(lcrzo_ipc *pipc,
			   lcrzo_constdevice device);
int lcrzo_ipc_read_device(lcrzo_ipc *pipc, 
			  lcrzo_time *pmaxwaitingtime, 
			  lcrzo_device device);

/*-------------------------------------------------------------*/
/*en idem, but work on a bloc composed of
   device_name + Ethernet_address + IP_address + port_number */
/*fr idem, mais travaille sur un bloc compose de
   nom_de_device + adresse_Ethernet + adresse_IP + numero_de_port */
int lcrzo_ipc_write_deviceethaiplport(lcrzo_ipc *pipc,
				      lcrzo_constdevice device,
				      lcrzo_constetha etha,
				      lcrzo_ipl ipl,
				      lcrzo_uint16 port);
int lcrzo_ipc_read_deviceethaiplport(lcrzo_ipc *pipc, 
				     lcrzo_time *pmaxwaitingtime, 
			             lcrzo_device device,
				     lcrzo_etha etha,
				     lcrzo_ipl *pipl,
				     lcrzo_uint16 *pport);

/*en idem, but work on a bloc composed of
   Ethernet_address + IP_address + port_number */
/*fr idem, mais travaille sur un bloc compose de
   adresse_Ethernet + adresse_IP + numero_de_port */
int lcrzo_ipc_write_ethaiplport(lcrzo_ipc *pipc,
				lcrzo_constetha etha,
				lcrzo_ipl ipl,
				lcrzo_uint16 port);
int lcrzo_ipc_read_ethaiplport(lcrzo_ipc *pipc, 
			       lcrzo_time *pmaxwaitingtime, 
			       lcrzo_etha etha,
			       lcrzo_ipl *pipl,
			       lcrzo_uint16 *pport);

/*en idem, but work on a bloc composed of
   IP_address + port_number */
/*fr idem, mais travaille sur un bloc compose de
   adresse_IP + numero_de_port */
int lcrzo_ipc_write_iplport(lcrzo_ipc *pipc,
			    lcrzo_ipl ipl,
			    lcrzo_uint16 port);
int lcrzo_ipc_read_iplport(lcrzo_ipc *pipc, 
			   lcrzo_time *pmaxwaitingtime, 
			   lcrzo_ipl *pipl,
			   lcrzo_uint16 *pport);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_ipc_write_msg
   Description :
     Write a data block in a lcrzo_ipc tube.
   Input parameter(s) :
     ipc : lcrzo_ipc where the data is written
     data : data to write
     datasize : size of data
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ipc_write_msg
   Description :
     Ecrit un bloc de donnees dans un tube lcrzo_ipc
   Parametre(s) d'entree :
     ipc : lcrzo_ipc ou les donnees sont ecrites
     data : donnees a ecrire
     datasize : taille de data
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ipc_write_msg(lcrzo_ipc *pipc,
			lcrzo_constdata data,
			lcrzo_int32 datasize);

/*en Name : lcrzo_ipc_write
   Description :
     Write a data flow in a lcrzo_ipc tube. 
     If datasize is too long, data will be written in several
     chunks (use lcrzo_ipc_write_msg if you want data to be written
     in one step).
  Input parameter(s) :
     ipc : lcrzo_ipc where the data is written
     data : data to write
     datasize : size of data
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ipc_write
   Description :
     Ecrit un flux de donnees dans un tube lcrzo_ipc
     Si datasize est trop long, les donnees sont ecrites en plusieurs
     fois (employez lcrzo_ipc_write_msg si vous desirez que les
     donnees soient ecrites en une seule fois).
   Parametre(s) d'entree :
     ipc : lcrzo_ipc ou les donnees sont ecrites
     data : donnees a ecrire
     datasize : taille de data
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ipc_write(lcrzo_ipc *pipc,
		    lcrzo_constdata data,
		    lcrzo_int32 datasize);

/*en Name : lcrzo_ipc_readm
   Description :
     Read data from a lcrzo_ipc tube.
   Input parameter(s) :
     ipc : lcrzo_ipc from where the datas are read
     *pmaxwaitingtime : maximum waiting time (can be a lcrzo_time,
                        LCRZO_WAIT_NODELAY or LCRZO_WAIT_INFINITE)
   Input/output parameter(s) :
   Output parameter(s) :
     *pdataout : pointer which will be malloced (so, the
                 memory will have to be freed by the
                 user with 'free(*pdataout)').
                 The allocated memory will be set with the read data
     *pdataoutsize : reached size of *pdataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : the function received nothing
                               before pmaxwaitingtime, so it 
			       exited without reading data
*/
/*fr Nom : lcrzo_ipc_readm
   Description :
     Lit des donnees depuis un tube lcrzo_ipc
   Parametre(s) d'entree :
     ipc : lcrzo_ipc ou les donnees sont lues
     *pmaxwaitingtime : temps maximum d'attente (peut etre un
                        lcrzo_time, LCRZO_WAIT_NODELAY ou
			LCRZO_WAIT_INFINITE)
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pdataout : pointeur qui sera alloue (la memoire
                 devra etre liberee par l'utilisateur 
                 avec 'free(*pdataout)').
                 Cette memoire allouee recevra les donnees lues
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : la fonction n'a pas recu de donnees
                               avant pmaxwaitingtime, donc on 
                               retourne sans donnees.
*/
int lcrzo_ipc_readm(lcrzo_ipc *pipc, 
		    lcrzo_time *pmaxwaitingtime, 
		    lcrzo_data *pdataout,
		    lcrzo_int32 *pdataoutsize);

/*en Name : lcrzo_ipc_readm_fixed
   Description :
     Read data from a lcrzo_ipc tube. Data read has a fixed size.
   Input parameter(s) :
     ipc : lcrzo_ipc from where the datas are read
     *pmaxwaitingtime : maximum waiting time (can be a lcrzo_time,
                        LCRZO_WAIT_NODELAY or LCRZO_WAIT_INFINITE)
     datasizetoread : exact size to read
   Input/output parameter(s) :
   Output parameter(s) :
     *pdataout : pointer which will be malloced (so, the
                 memory will have to be freed by the
                 user with 'free(*pdataout)').
                 The allocated memory will be set with the read data
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : the function received nothing
                               before pmaxwaitingtime, so it 
			       exited without reading data
*/
/*fr Nom : lcrzo_ipc_readm_fixed
   Description :
     Lit des donnees depuis un tube lcrzo_ipc
     Les donnees lues ont une taille constante.
   Parametre(s) d'entree :
     ipc : lcrzo_ipc ou les donnees sont lues
     *pmaxwaitingtime : temps maximum d'attente (peut etre un
                        lcrzo_time, LCRZO_WAIT_NODELAY ou
			LCRZO_WAIT_INFINITE)
     datasizetoread : taille exacte a lire
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pdataout : pointeur qui sera alloue (la memoire
                 devra etre liberee par l'utilisateur 
                 avec 'free(*pdataout)').
                 Cette memoire allouee recevra les donnees lues
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : la fonction n'a pas recu de donnees
                               avant pmaxwaitingtime, donc on 
                               retourne sans donnees.
*/
int lcrzo_ipc_readm_fixed(lcrzo_ipc *pipc, 
			  lcrzo_time *pmaxwaitingtime, 
			  lcrzo_int32 datasizetoread,
			  lcrzo_data *pdataout);

/*en Name : lcrzo_ipc_readm_line
   Description :
     Read a line (till \r\n) from a lcrzo_ipc tube.
   Input parameter(s) :
     ipc : lcrzo_ipc from where the datas are read
     *pmaxwaitingtime : maximum waiting time (can be a lcrzo_time,
                        LCRZO_WAIT_NODELAY or LCRZO_WAIT_INFINITE)
   Input/output parameter(s) :
   Output parameter(s) :
     *pdataout : pointer which will be malloced (so, the
                 memory will have to be freed by the
                 user with 'free(*pdataout)').
                 The allocated memory will be set with the read data
     *pdataoutsize : reached size of *pdataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : the function received nothing
                               before pmaxwaitingtime, so it 
			       exited without reading data
*/
/*fr Nom : lcrzo_ipc_readm_line
   Description :
     Lit une ligne (jusqu'a \r\n) depuis un tube lcrzo_ipc
   Parametre(s) d'entree :
     ipc : lcrzo_ipc ou les donnees sont lues
     *pmaxwaitingtime : temps maximum d'attente (peut etre un
                        lcrzo_time, LCRZO_WAIT_NODELAY ou
			LCRZO_WAIT_INFINITE)
     beblocking : si on veut bloquer jusqu'a recevoir des donnees
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pdataout : pointeur qui sera alloue (la memoire
                 devra etre liberee par l'utilisateur 
                 avec 'free(*pdataout)').
                 Cette memoire allouee recevra les donnees lues
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : la fonction n'a pas recu de donnees
                               avant pmaxwaitingtime, donc on 
                               retourne sans donnees.
*/
int lcrzo_ipc_readm_line(lcrzo_ipc *pipc, 
			 lcrzo_time *pmaxwaitingtime, 
			 lcrzo_data *pdataout,
			 lcrzo_int32 *pdataoutsize);
