/*en    Laurent Constantin's network library (lcrzo)
                 lcrzo_misc module

  Miscellaneous functions.
*/
/*fr    Bibliotheque reseau de Laurent Constantin (lcrzo)
                 Module lcrzo_misc

  Les fonctions presentes dans ce module assurent des taches diverses.
*/


/*-------------------------------------------------------------*/
/*en Name : lcrzo_init
   Description :
     Initialize lcrzo.
     You cannot call this function with "lcrzo_er(lcrzo_init());",
     but only with "lcrzo_init();", because errors message cannot
     be displayed before a successful end of lcrzo_init.
   Input parameter(s) :
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_init
   Description :
     Initialise lcrzo.
     Vous ne pouvez pas appeler cette fonction avec 
     "lcrzo_er(lcrzo_init());", mais uniquement avec "lcrzo_init();",
     car les messages d'erreur ne peuvent etre affiches avant un
     succes de lcrzo_init.
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_init(void);

/*en Name : lcrzo_close
   Description :
     Finish to use lcrzo's functions.
   Input parameter(s) :
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_close
   Description :
     Arrete d'utiliser les fonctions de lcrzo.
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_close(void);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_init_forthreadonly
   Description :
     Start to use lcrzo's functions.
     This function is useful only under Windows, and if you
     do not use lcrzo_process family. So generally, you should
     not need to use this function.
   Input parameter(s) :
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_init_forthreadonly
   Description :
     Commence a utiliser les fonctions de lcrzo.
     Cette fonction est utile uniquement sur Windows, et si vous
     n'utilisez pas la famille lcrzo_process. Donc, en general, 
     vous ne devriez pas avoir besoin de cette fonction.
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_init_forthreadonly(void);

/*en Name : lcrzo_close_forthreadonly
   Description :
     Finish to use lcrzo's functions in a thread under Windows.
   Input parameter(s) :
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_close_forthreadonly
   Description :
     Arrete d'utiliser les fonctions de lcrzo dans un thread
     de Windows.
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_close_forthreadonly(void);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_rand_init
   Description :
     Seed the random generator
   Input parameter(s) :
     seed : if 0, use /dev/random (or urandom, or time_pid)
            else use seed
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_rand_init
   Description :
     Initialise le generateur aleatoire
   Parametre(s) d'entree :
     seed : si 0, utilise /dev/random (ou urandom, ou time_pid)
            sinon utilise seed
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_rand_init(lcrzo_uint32 seed);

/*en Name : lcrzo_xintxx_init_rand
   Description :
     Generate a random number
   Input parameter(s) :
     min : minimum value
     max : maximum value
   Input/output parameter(s) :
   Output parameter(s) :
     *pnumber : random number generated
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_xintxx_init_rand
   Description :
     Genere un nombre aleatoire
   Parametre(s) d'entree :
     min : valeur minimale
     max : valeur maximale
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pnumber : alea genere
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_uint32_init_rand(lcrzo_uint32 min,
			   lcrzo_uint32 max, 
			   lcrzo_uint32 *pnumber);
int lcrzo_int32_init_rand(lcrzo_int32 min,
			  lcrzo_int32 max, 
			  lcrzo_int32 *pnumber);
int lcrzo_uint16_init_rand(lcrzo_uint16 min,
			   lcrzo_uint16 max, 
			   lcrzo_uint16 *pnumber);
int lcrzo_int16_init_rand(lcrzo_int16 min,
			  lcrzo_int16 max, 
			  lcrzo_int16 *pnumber);
int lcrzo_uint8_init_rand(lcrzo_uint8 min,
			  lcrzo_uint8 max, 
			  lcrzo_uint8 *pnumber);
int lcrzo_int8_init_rand(lcrzo_int8 min,
			 lcrzo_int8 max, 
			 lcrzo_int8 *pnumber);
int lcrzo_bool_init_rand(lcrzo_bool *pnumber);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_xintxx_init_text
   Description :
     Initialize an integer from a string.
   Input parameter(s) :
     str : string
   Input/output parameter(s) :
   Output parameter(s) :
     *pnumber : number
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_xintxx_init_text
   Description :
     Initialise un entier a partir d'une chaine texte
   Parametre(s) d'entree :
     str : chaine texte
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pnumber : nombre
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_uint32_init_text(lcrzo_conststring str,
			   lcrzo_uint32 *pnumber);
int lcrzo_int32_init_text(lcrzo_conststring str,
			  lcrzo_int32 *pnumber);
int lcrzo_uint16_init_text(lcrzo_conststring str,
			   lcrzo_uint16 *pnumber);
int lcrzo_int16_init_text(lcrzo_conststring str,
			  lcrzo_int16 *pnumber);
int lcrzo_uint8_init_text(lcrzo_conststring str,
			  lcrzo_uint8 *pnumber);
int lcrzo_int8_init_text(lcrzo_conststring str,
			 lcrzo_int8 *pnumber);
int lcrzo_bool_init_text(lcrzo_conststring str,
			 lcrzo_bool *pnumber);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_uint32_init_stdin
   Description :
     Ask user to enter an integer
   Input parameter(s) :
     message : message to print before
     min : minvalue which can be entered
     max : maxvalue which can be entered
     defaultnumber : default character
   Input/output parameter(s) :
   Output parameter(s) :
     *pnum : number chosen
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_uint32_init_stdin
   Description :
     Demande a l'utilisateur d'entrer un entier
   Parametre(s) d'entree :
     message : message a afficher avant
     min : valeur minimale
     max : valeur maximale
     defaultnumber : entier par defaut
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pnumber : entier choisi
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_uint32_init_stdin(lcrzo_conststring message,
			    lcrzo_uint32 min, 
			    lcrzo_uint32 max, 
			    lcrzo_uint32 defaultnumber, 
			    lcrzo_uint32 *pnumber);
int lcrzo_int32_init_stdin(lcrzo_conststring message,
			   lcrzo_int32 min, 
			   lcrzo_int32 max, 
			   lcrzo_int32 defaultnumber, 
			   lcrzo_int32 *pnumber);
int lcrzo_uint16_init_stdin(lcrzo_conststring message,
			    lcrzo_uint16 min, 
			    lcrzo_uint16 max, 
			    lcrzo_uint16 defaultnumber, 
			    lcrzo_uint16 *pnumber);
int lcrzo_int16_init_stdin(lcrzo_conststring message,
			   lcrzo_int16 min, 
			   lcrzo_int16 max, 
			   lcrzo_int16 defaultnumber, 
			   lcrzo_int16 *pnumber);
int lcrzo_uint8_init_stdin(lcrzo_conststring message,
			   lcrzo_uint8 min, 
			   lcrzo_uint8 max, 
			   lcrzo_uint8 defaultnumber, 
			   lcrzo_uint8 *pnumber);
int lcrzo_int8_init_stdin(lcrzo_conststring message,
			  lcrzo_int8 min, 
			  lcrzo_int8 max, 
			  lcrzo_int8 defaultnumber, 
			  lcrzo_int8 *pnumber);
int lcrzo_bool_init_stdin(lcrzo_conststring message,
			  lcrzo_bool defaultnumber, 
			  lcrzo_bool *pnumber);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_uint32_fprint
   Description :
     Print a lcrzo_uint32.
   Input parameter(s) :
     *pf : FILE where the function prints
     startstring : string to print before the lcrzo_uint32
     number : lcrzo_uint32 to print
     fmt : format for the integer. For example "%d", "%08X", etc.
           Note : the format must not contain a 'l' ("%ld"). This
           'l' is automatically added is needed.
     endstring : string to print after the lcrzo_uint32
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_uint32_fprint
   Description :
     Ecrit un lcrzo_uint32.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
     startstring : chaine a ecrire avant le lcrzo_uint32
     number : lcrzo_uint32 a ecrire
     fmt : format pour l'entier. Par exemple "%d", "%08X", etc.
           Note : le format ne doit pas contenir le 'l' ("%ld").
           Ce 'l' est ajoute automatiquement.
     endstring : chaine a ecrire apres le lcrzo_uint32
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_uint32_fprint(LCRZOFILE *pf,
			lcrzo_conststring startstring,
			lcrzo_uint32 number,
			lcrzo_conststring fmt,
			lcrzo_conststring endstring);
int lcrzo_int32_fprint(LCRZOFILE *pf,
		       lcrzo_conststring startstring,
		       lcrzo_int32 number,
		       lcrzo_conststring fmt,	
		       lcrzo_conststring endstring);
int lcrzo_uint16_fprint(LCRZOFILE *pf,
			lcrzo_conststring startstring,
			lcrzo_uint16 number,
			lcrzo_conststring fmt,
			lcrzo_conststring endstring);
int lcrzo_int16_fprint(LCRZOFILE *pf,
		       lcrzo_conststring startstring,
		       lcrzo_int16 number,
		       lcrzo_conststring fmt,
		       lcrzo_conststring endstring);
int lcrzo_uint8_fprint(LCRZOFILE *pf,
		       lcrzo_conststring startstring,
		       lcrzo_uint8 number,
		       lcrzo_conststring fmt,
		       lcrzo_conststring endstring);
int lcrzo_int8_fprint(LCRZOFILE *pf,
		      lcrzo_conststring startstring,
		      lcrzo_int8 number,
		      lcrzo_conststring fmt,
		      lcrzo_conststring endstring);
int lcrzo_bool_fprint(LCRZOFILE *pf,
		      lcrzo_conststring startstring,
		      lcrzo_bool number,
		      lcrzo_conststring endstring);

#define lcrzo_uint32_print(a,b,c,d) lcrzo_uint32_fprint(stdout,a,b,c,d)
#define lcrzo_int32_print(a,b,c,d) lcrzo_int32_fprint(stdout,a,b,c,d)
#define lcrzo_uint16_print(a,b,c,d) lcrzo_uint16_fprint(stdout,a,b,c,d)
#define lcrzo_int16_print(a,b,c,d) lcrzo_int16_fprint(stdout,a,b,c,d)
#define lcrzo_uint8_print(a,b,c,d) lcrzo_uint8_fprint(stdout,a,b,c,d)
#define lcrzo_int8_print(a,b,c,d) lcrzo_int8_fprint(stdout,a,b,c,d)
#define lcrzo_bool_print(a,b,c) lcrzo_bool_fprint(stdout,a,b,c)


/*-------------------------------------------------------------*/
/*en Name : lcrzo_stream_flush
   Description :
     Flush the stdin buffer
   Input parameter(s) :
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_stream_flush
   Description :
     Vide le buffer d'entree
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_stream_flush(LCRZOFILE *pstream);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_stdin_press
   Description :
     Ask user to press a key
   Input parameter(s) :
     message : message to print before
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_stdin_press
   Description :
     Demande a l'utilisateur de presser une touche
   Parametre(s) d'entree :
     message : message a afficher avant
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_stdin_press(lcrzo_conststring message);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_stdin_char
   Description :
     Ask user to enter a character
   Input parameter(s) :
     message : message to print before
     allowedchar : string containing allowed characters
                   For example : "aAbBrR"
     defaultchar : default character
   Input/output parameter(s) :
   Output parameter(s) :
     *pchar : character chosen
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_stdin_char
   Description :
     Demande a l'utilisateur d'entrer un caractere
   Parametre(s) d'entree :
     message : message a afficher avant
     allowedchar : chaine contenant les caracteres autorises
                   Par exemple : "aAbBrR"
     defaultchar : caractere par defaut
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pchar : caractere choisi
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_stdin_char(lcrzo_conststring message,
		     lcrzo_constdata allowedchar,
		     lcrzo_uint8 defaultchar,
		     lcrzo_uint8 *pchar);

/*-------------------------------------------------------------*/
/*en the following defines convert between host and network
   representation of "short int" et "long int" :
     lcrzo_ntohl : network to host long
     lcrzo_ntohs : network to host short
     lcrzo_htonl : host to network long
     lcrzo_htons : host to network short
*/
/*fr les defines suivants assurent la conversion entre les
   representations machine et reseau des "short int" et "long int" :
     lcrzo_ntohl : network to host long
     lcrzo_ntohs : network to host short
     lcrzo_htonl : host to network long
     lcrzo_htons : host to network short
*/
#if LCRZODEF_ENDIANLITTLE==1
 /*i386*/
 #define lcrzo_ntohl(x) ((lcrzo_uint32)((((lcrzo_uint32)(x) & 0x000000ffU) << 24) | (((lcrzo_uint32)(x) & 0x0000ff00U) <<  8) | (((lcrzo_uint32)(x) & 0x00ff0000U) >>  8) | (((lcrzo_uint32)(x) & 0xff000000U) >> 24)))
 #define lcrzo_ntohs(x) ((lcrzo_uint16)((((lcrzo_uint16)(x) & 0x00ff) << 8) | (((lcrzo_uint16)(x) & 0xff00) >> 8)))
 #define lcrzo_htonl(x) lcrzo_ntohl(x)
 #define lcrzo_htons(x) lcrzo_ntohs(x)
#elif LCRZODEF_ENDIANBIG==1
 /*68000*/
 #define lcrzo_ntohl(x)        (x)
 #define lcrzo_ntohs(x)        ((lcrzo_uint16)(x))
 #define lcrzo_htonl(x)        (x)
 #define lcrzo_htons(x)        ((lcrzo_uint16)(x))
#else
 #error  "Complete Endian"
 #error  "Completer le cas de Endian"
#endif

/*-------------------------------------------------------------*/
/*en Name : lcrzo_compilparam_check
   Description :
     Check compilation parameters.
   Input parameter(s) :
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_compilparam_check
   Description :
     Verifie les conditions de compilation de la librairie
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_compilparam_check(void);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_compilparam_get
   Description :
     Get compilation parameters.
   Input parameter(s) :
     wantedparameter : wanted parameter
   Input/output parameter(s) :
   Output parameter(s) :
     *pparametervalue : value of this parameter
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_compilparam_get
   Description :
     Obtient les conditions de compilation de la librairie
   Parametre(s) d'entree :
     wantedparameter : parametre voulu
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pparametervalue : valeur de ce parametre
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_compilparam_get(lcrzo_conststring wantedparameter,
		    lcrzo_string *pparametervalue);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_compilparam_fprint
   Description :
     Print compilation parameters.
   Input parameter(s) :
     *pf : FILE where the function prints
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_compilparam_fprint
   Description :
     Affiche les conditions de compilation de la librairie
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_compilparam_fprint(LCRZOFILE *pf);
#define lcrzo_compilparam_print() lcrzo_compilparam_fprint(stdout)

/*-------------------------------------------------------------*/
#define LCRZO_WINDOWS_SYSTEMVERSION_unknown   0
#define LCRZO_WINDOWS_SYSTEMVERSION_31       31
#define LCRZO_WINDOWS_SYSTEMVERSION_95       95
#define LCRZO_WINDOWS_SYSTEMVERSION_98       98
#define LCRZO_WINDOWS_SYSTEMVERSION_me       99
#define LCRZO_WINDOWS_SYSTEMVERSION_nt350   350
#define LCRZO_WINDOWS_SYSTEMVERSION_nt351   351
#define LCRZO_WINDOWS_SYSTEMVERSION_nt4     400
#define LCRZO_WINDOWS_SYSTEMVERSION_2k     2000
#define LCRZO_WINDOWS_SYSTEMVERSION_xp     2001
#define LCRZO_WINDOWS_SYSTEMVERSION_net    2002
/*en Name : lcrzo_windows_getsystemversion
   Description :
     Get system version.
   Input parameter(s) :
   Input/output parameter(s) :
   Output parameter(s) :
     *psystem : system type (LCRZO_WINDOWS_SYSTEMVERSION_xx)
     *pspmajor : major version of service pack
     *pspminor : minor version of service pack
     *pbuild : build number
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_windows_getsystemversion
   Description :
     Obtient le type du systeme.
   Parametre(s) d'entree :
     *psystem : type du systeme (LCRZO_WINDOWS_SYSTEMVERSION_xx)
     *pspmajor : version majeure du service pack
     *pspminor : version mineure du service pack
     *pbuild : build number
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_windows_getsystemversion(lcrzo_uint32 *psystemversion,
				   lcrzo_uint32 *pspmajor,
				   lcrzo_uint32 *pspminor,
				   lcrzo_uint32 *pbuild);
