/*en    Laurent Constantin's network library (lcrzo)
                 lcrzo_time module

  Functions herein allow to measure time.
*/
/*fr    Bibliotheque reseau de Laurent Constantin (lcrzo)
                 Module lcrzo_time

  Les fonctions presentes dans ce module assurent la gestion
du temps.
*/

/*-------------------------------------------------------------*/
/*en definition of a lcrzo_time. You should not use its
   fields directly. */
/*fr definition d'un lcrzo_time. Vous ne devriez pas utiliser
   directement ses champs. */
typedef struct
{ lcrzo_uint32 sec;
  lcrzo_uint32 usec;  /* cannot be over 999999 */
} lcrzo_time;
typedef const lcrzo_time lcrzo_consttime;

/*-------------------------------------------------------------*/
/*en Name : lcrzo_time_sleep_xsec
   Description :
     Wait for a number of seconds, milliseconds or microseconds.
     The waited time might be greater than user value.
   Input parameter(s) :
     sec : number of seconds to wait
     msec : number of milliseconds to wait
     usec : number of microseconds to wait
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_time_sleep_xsec
   Description :
     Attend sec secondes, msec millisecondes ou usec microsecondes.
     La duree attendue peut etre plus grande que celle voulue.
   Parametre(s) d'entree :
     sec : nombre de secondes a attendre
     msec : nombre de millisecondes a attendre
     usec : nombre de microsecondes a attendre
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_time_sleep_sec(lcrzo_uint32 sec);
int lcrzo_time_sleep_msec(lcrzo_uint32 msec);
int lcrzo_time_sleep_usec(lcrzo_uint32 usec);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_time_pause_xsec
   Description :
     Wait for a number of seconds, milliseconds or microseconds.
     The waited time is wanted to be as precise as possible.
   Input parameter(s) :
     sec : number of seconds to wait
     msec : number of milliseconds to wait
     usec : number of microseconds to wait
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_time_pause_xsec
   Description :
     Attend sec secondes, msec millisecondes ou usec microsecondes.
     La duree attendue essaie d'etre la plus precise possible.
   Parametre(s) d'entree :
     sec : nombre de secondes a attendre
     msec : nombre de millisecondes a attendre
     usec : nombre de microsecondes a attendre
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_time_pause_sec(lcrzo_uint32 sec);
int lcrzo_time_pause_msec(lcrzo_uint32 msec);
int lcrzo_time_pause_usec(lcrzo_uint32 usec);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_time_fprint
   Description :
     Print a lcrzo_time.
   Input parameter(s) :
     *pf : FILE where the function prints
     startstring : string to print before the lcrzo_time
     *ptime : lcrzo_time to print
     endstring : string to print after the lcrzo_time
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_time_fprint
   Description :
     Affiche un lcrzo_time.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
     startstring : chaine a ecrire avant le lcrzo_time
     time : lcrzo_time a afficher
     endstring : chaine a ecrire apres le lcrzo_time
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_time_fprint(LCRZOFILE *pf,
		      lcrzo_conststring startstring,
		      lcrzo_consttime *ptime,
		      lcrzo_conststring endstring);

/*en idem, but on stdout */
/*fr idem, mais sur stdout */
#define lcrzo_time_print(s,t,e) lcrzo_time_fprint(stdout, s,t,e)

/*-------------------------------------------------------------*/
/*en Name : lcrzo_time_init_currenttime
   Description :
     Initialize a lcrzo_time with the current time.
   Input parameter(s) :
   Input/output parameter(s) :
   Output parameter(s) :
     *ptime : lcrzo_time set to the current time
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_time_init_currenttime
   Description :
     Initialise un lcrzo_time avec l'heure courante.
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *ptime : lcrzo_time affecte a l'heure courante
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_time_init_currenttime(lcrzo_time *ptime);

/*en Name : lcrzo_time_init_sec
   Description :
     Initialize a lcrzo_time with a number of seconds.
   Input parameter(s) :
     sec : number of seconds
   Input/output parameter(s) :
   Output parameter(s) :
     *ptime : lcrzo_time set
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_time_init_sec
   Description :
     Initialise un lcrzo_time avec un nombre de secondes.
   Parametre(s) d'entree :
     sec : nombre de secondes
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *ptime : lcrzo_time affecte
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_time_init_sec(lcrzo_uint32 sec,
			lcrzo_time *ptime);

/*en Name : lcrzo_time_init_msec
   Description :
     Initialize a lcrzo_time with a number of milliseconds.
   Input parameter(s) :
     usec : number of milliseconds
   Input/output parameter(s) :
   Output parameter(s) :
     *ptime : lcrzo_time set
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_time_init_msec
   Description :
     Initialise un lcrzo_time avec un nombre de millisecondes.
   Parametre(s) d'entree :
     usec : nombre de millisecondes
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *ptime : lcrzo_time affecte
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_time_init_msec(lcrzo_uint32 msec,
			 lcrzo_time *ptime);

/*en Name : lcrzo_time_init_usec
   Description :
     Initialize a lcrzo_time with a number of micro-seconds.
   Input parameter(s) :
     usec : number of micro-seconds
   Input/output parameter(s) :
   Output parameter(s) :
     *ptime : lcrzo_time set
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_time_init_usec
   Description :
     Initialise un lcrzo_time avec un nombre de micro-secondes.
   Parametre(s) d'entree :
     usec : nombre de micro-secondes
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *ptime : lcrzo_time affecte
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_time_init_usec(lcrzo_uint32 usec,
			 lcrzo_time *ptime);

/*en Name : lcrzo_time_init_secmsecusec
   Description :
     Initialize a lcrzo_time.
   Input parameter(s) :
     sec : number of seconds
     msec : number of milli-seconds
     usec : number of micro-seconds
   Input/output parameter(s) :
   Output parameter(s) :
     *ptime : lcrzo_time set
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_time_init_secmsecusec
   Description :
     Initialise un lcrzo_time.
   Parametre(s) d'entree :
     sec : nombre de secondes
     msec : nombre de milli-secondes
     usec : nombre de micro-secondes
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *ptime : lcrzo_time affecte
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_time_init_secusec(lcrzo_uint32 sec,
			    lcrzo_uint32 usec,
			    lcrzo_time *ptime);
int lcrzo_time_init_msecusec(lcrzo_uint32 msec,
			     lcrzo_uint32 usec,
			     lcrzo_time *ptime);
int lcrzo_time_init_secmsec(lcrzo_uint32 sec,
			    lcrzo_uint32 msec,
			    lcrzo_time *ptime);
int lcrzo_time_init_secmsecusec(lcrzo_uint32 sec,
				lcrzo_uint32 msec,
				lcrzo_uint32 usec,
				lcrzo_time *ptime);

/*en initialize another lcrzo_time */
/*fr initialise un autre lcrzo_time */
int lcrzo_time_init_time(lcrzo_consttime *ptimein,
			 lcrzo_time *ptimeout);

/*en get seconds, milliseconds and microseconds stored in a lcrzo_time */
/*fr obtient les secondes, millisecondes et microsecondes stockees dans
  un lcrzo_time */
/*en  function              obtained ranges */
/*fr  fonction              intervalles obtenus */
/*
    lcrzo_time_get_sec          [0..2^32]
    lcrzo_time_get_msec         [0..2^32]
    lcrzo_time_get_usec         [0..2^32]
    lcrzo_time_get_secusec      [0..2^32], [0-999999]
    lcrzo_time_get_secmsec      [0..2^32], [0-999]
    lcrzo_time_get_msecusec     [0..2^32], [0-999]
    lcrzo_time_get_secmsecusec  [0..2^32], [0-999], [0-999]
*/
int lcrzo_time_get_sec(lcrzo_consttime *ptime,
		       lcrzo_uint32 *psec);
int lcrzo_time_get_msec(lcrzo_consttime *ptime,
			lcrzo_uint32 *pmsec);
int lcrzo_time_get_usec(lcrzo_consttime *ptime,
			lcrzo_uint32 *pusec);
int lcrzo_time_get_secusec(lcrzo_consttime *ptime,
			   lcrzo_uint32 *psec,
			   lcrzo_uint32 *pusec);
int lcrzo_time_get_secmsec(lcrzo_consttime *ptime,
			   lcrzo_uint32 *psec,
			   lcrzo_uint32 *pmsec);
int lcrzo_time_get_msecusec(lcrzo_consttime *ptime,
			    lcrzo_uint32 *pmsec,
			    lcrzo_uint32 *pusec);
int lcrzo_time_get_secmsecusec(lcrzo_consttime *ptime,
			       lcrzo_uint32 *psec,
			       lcrzo_uint32 *pmsec,
			       lcrzo_uint32 *pusec);

/*en add seconds/milliseconds/microseconds/time to a lcrzo_time */
/*fr ajoute des secondes/millisecondes/microsecondes/time a un lcrzo_time */
int lcrzo_time_plus_sec(lcrzo_consttime *ptime,
			lcrzo_uint32 sec,
			lcrzo_time *presulttime);
int lcrzo_time_plus_msec(lcrzo_consttime *ptime,
			 lcrzo_uint32 msec,
			 lcrzo_time *presulttime);
int lcrzo_time_plus_usec(lcrzo_consttime *ptime,
			 lcrzo_uint32 usec,
			 lcrzo_time *presulttime);
/* *ppresulttime = *ptime1 + *ptime2 */
int lcrzo_time_plus_time(lcrzo_consttime *ptime1,
			 lcrzo_consttime *ptime2,
			 lcrzo_time *presulttime);

/*en substract seconds/milliseconds/microseconds/time to a lcrzo_time */
/*fr retranche des secondes/millisecondes/microsecondes/time a un lcrzo_time */
int lcrzo_time_minus_sec(lcrzo_consttime *ptime,
			 lcrzo_uint32 sec,
			 lcrzo_time *presulttime);
int lcrzo_time_minus_msec(lcrzo_consttime *ptime,
			  lcrzo_uint32 msec,
			  lcrzo_time *presulttime);
int lcrzo_time_minus_usec(lcrzo_consttime *ptime,
			  lcrzo_uint32 usec,
			  lcrzo_time *presulttime);
/* *ppresulttime = *ptime1 - *ptime2 */
int lcrzo_time_minus_time(lcrzo_consttime *ptime1,
			  lcrzo_consttime *ptime2,
			  lcrzo_time *presulttime);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_time_diffcurrent_sec
   Description :
     Compare a lcrzo_time to the current time.
   Input parameter(s) :
     *ptime : lcrzo_time to compare to the current time
   Input/output parameter(s) :
   Output parameter(s) :
     *psec : number of seconds between the two times
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_time_diffcurrent_sec
   Description :
     Compare un lcrzo_time a l'heure courante.
   Parametre(s) d'entree :
     *ptime : lcrzo_time a comparer a l'heure courante
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psec : nombre de secondes entre les deux heures
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
/*en  function              obtained ranges */
/*fr  fonction              intervalles obtenus */
/*
lcrzo_time_get_sec          [0..2^32]
lcrzo_time_get_msec         [0..2^32]
lcrzo_time_get_usec         [0..2^32]
lcrzo_time_get_secusec      [0..2^32], [0-999999]
lcrzo_time_get_secmsec      [0..2^32], [0-999]
lcrzo_time_get_msecusec     [0..2^32], [0-999]
lcrzo_time_get_secmsecusec  [0..2^32], [0-999], [0-999]
*/
int lcrzo_time_diffcurrent_sec(lcrzo_consttime *ptime,
			       lcrzo_uint32 *psec);
int lcrzo_time_diffcurrent_msec(lcrzo_consttime *ptime,
				lcrzo_uint32 *pmsec);
int lcrzo_time_diffcurrent_usec(lcrzo_consttime *ptime,
				lcrzo_uint32 *pusec);
int lcrzo_time_diffcurrent_secusec(lcrzo_consttime *ptime,
				   lcrzo_uint32 *psec,
				   lcrzo_uint32 *pusec);
int lcrzo_time_diffcurrent_secmsec(lcrzo_consttime *ptime,
				   lcrzo_uint32 *psec,
				   lcrzo_uint32 *pmsec);
int lcrzo_time_diffcurrent_msecusec(lcrzo_consttime *ptime,
				    lcrzo_uint32 *pmsec,
				    lcrzo_uint32 *pusec);
int lcrzo_time_diffcurrent_secmsecusec(lcrzo_consttime *ptime,
				       lcrzo_uint32 *psec,
				       lcrzo_uint32 *pmsec,
				       lcrzo_uint32 *pusec);

/*en Name : lcrzo_time_diffcurrent_time
   Description :
     Compare a lcrzo_time to the current time.
   Input parameter(s) :
     *ptime : lcrzo_time to compare to the current time
   Input/output parameter(s) :
   Output parameter(s) :
     *pdifftime : lcrzo_time containing the difference
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_time_diffcurrent_time
   Description :
     Compare un lcrzo_time a l'heure courante.
   Parametre(s) d'entree :
     *ptime : lcrzo_time a comparer a l'heure courante
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pdifftime : lcrzo_time contenant la difference
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_time_diffcurrent_time(lcrzo_consttime *ptime,
				lcrzo_time *pdifftime);
