/////////////////////////////////////////////////////////////////////////////
// File:            gnGBKSource.h
// Purpose:         Implements gnBaseSource for GenBank files
// Description:     
// Changes:        
// Version:         libGenome 0.1.0 
// Author:          Aaron Darling 
// Last Edited:     April 15, 2001, 10:34:50pm 
// Modified by:     
// Copyright:       (c) Aaron Darling 
// Licenses:        Proprietary 
/////////////////////////////////////////////////////////////////////////////

#ifndef _gnGBKSource_h_
#define _gnGBKSource_h_

#include "gn/gnDefs.h"

#include <string>
#include <fstream>
#include <vector>
#include "gn/gnFileSource.h"
#include "gn/gnFileContig.h"
#include "gn/gnSourceSpec.h"
#include "gn/gnSequence.h"

const uint32 SEQ_COLUMN_WIDTH = 80;
const uint32 SEQ_HEADER_NAME_LENGTH = 11;
const uint32 SEQ_SUBTAG_COLUMN = 5;
const uint32 SEQ_LOCUS_CIRCULAR_COLUMN = 43;
const uint32 SEQ_LOCUS_NAME_COLUMN = 13;
const uint32 SEQ_LOCUS_NAME_LENGTH = 10;
const uint32 SEQ_LOCUS_SIZE_LENGTH = 7;
const uint32 SEQ_LOCUS_DNATYPE_OFFSET = 33;
const uint32 SEQ_LOCUS_DNATYPE_LENGTH = 7;
const uint32 SEQ_LOCUS_DIVCODE_OFFSET = 52;
const uint32 SEQ_LOCUS_DIVCODE_LENGTH = 3;
const uint32 SEQ_LOCUS_DATE_OFFSET = 62;
const uint32 SEQ_LOCUS_DATE_LENGTH = 11;
const uint32 SEQ_FEATURE_LOC_OFFSET = 21;
const uint32 SEQ_BASES_INDEX_END = 9;

/**
 *	gnGBKSource is a GenBank sequence file reader.
 * This class reads and writes the GenBank file format.
 * gnGBKSource is used by gnSourceFactory to read files and should only be used
 * directly when writing out files in GBK file format by calling
 * gnGBKSource::Write( mySpec, "C:\mySeqFile.gbk");
 */

class GNDLLEXPORT gnGBKSource : public gnFileSource
{
public:
	/**
	 * Empty Constructor, does nothing.
	 */
	gnGBKSource();	
	/**
	 * Clone Constructor copies the specified gnGBKSource.
	 * @param s The gnGBKSource to copy.
	 */
	gnGBKSource( const gnGBKSource& s );
	/**
	 * Destructor, frees memory.
	 */
	~gnGBKSource();
	/**
	 * Returns an exact copy of this class.
	 */
	gnGBKSource* Clone() const;
// Contig Access methods	
	uint32 GetContigListLength() const;
	boolean HasContig( const string& name ) const;
	uint32 GetContigID( const string& name ) const;
	string GetContigName( const uint32 i ) const;
	gnSeqI GetContigSeqLength( const uint32 i ) const;

	boolean SeqRead( const gnSeqI start, char* buf, uint32& bufLen, const uint32 contigI=ALL_CONTIGS );

	/**
	 * Writes the specified gnSequence to a .GBK file named "filename".
	 * @param source The gnSequence to write out.
	 * @param filename The name of the file to write.
	 * @return True if successful, false otherwise.
	 */
	static boolean Write(gnSequence& seq, const string& filename);
	/**
	 * Writes the specified source to a .GBK file named "filename".
	 * @param source The source to write out.
	 * @param filename The name of the file to write.
	 * @return True if successful, false otherwise.
	 */
	static boolean Write(gnBaseSource *source, const string& filename);
	gnGenomeSpec *GetSpec() const;
	gnFileContig* GetFileContig( const uint32 contigI ) const;
private:
	boolean SeqSeek( const gnSeqI start, const uint32& contigI, uint64& startPos, uint64& readableBytes );
	boolean SeqStartPos( const gnSeqI start, gnFileContig& contig, uint64& startPos, uint64& readableBytes );
	boolean ParseStream( istream& fin );

	static string& Filler(uint32 length);
	static void FormatString(string& data, uint32 offset, uint32 width);
	static void WriteHeader(gnMultiSpec* spec, const string& hdr, ofstream& m_ofstream);
//	gnSeqI m_seqLength;
	
	gnGenomeSpec *m_spec;
	vector< gnFileContig* > m_contigList;	
};// class gnGBKSource
// Clone	
inline
gnGBKSource* gnGBKSource::Clone() const
{
	return new gnGBKSource( *this );
}
// Contig Access methods	
inline
uint32 gnGBKSource::GetContigListLength() const
{
	return m_contigList.size();
}
inline
boolean gnGBKSource::Write(gnBaseSource *source, const string& filename){
	gnSequence gns(*source->GetSpec());
	return Write(gns, filename);
}
inline
gnGenomeSpec *gnGBKSource::GetSpec() const{
	return m_spec->Clone();
}

#endif
	// _gnGBKSource_h_
