# Languages.py
#
# Moleskine: a source code editor for the GNOME desktop
#
# Copyright (c) 2000 - 2002   Michele Campeotto <micampe@micampe.it>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

from UserDict import *
import os.path
import string

from GTK import *
import gnome.config

import GTKSCINTILLA
import Moleskine

_CONFIG_FILE = '/Moleskine/LanguageModes/'

Lexers = {
    _('None')          : GTKSCINTILLA.LEXER_NULL,
    _('Python')        : GTKSCINTILLA.LEXER_PYTHON,
    _('C/C++')         : GTKSCINTILLA.LEXER_CPP,
    _('HTML')          : GTKSCINTILLA.LEXER_HTML,
    _('XML')           : GTKSCINTILLA.LEXER_XML,
    _('Perl')          : GTKSCINTILLA.LEXER_PERL,
    _('SQL')           : GTKSCINTILLA.LEXER_SQL,
    _('Visual Basic')  : GTKSCINTILLA.LEXER_VB,
    _('Properties')    : GTKSCINTILLA.LEXER_PROPERTIES,
    _('Error list')    : GTKSCINTILLA.LEXER_ERRORLIST,
    _('Makefile')      : GTKSCINTILLA.LEXER_MAKEFILE,
    _('Batch file')    : GTKSCINTILLA.LEXER_BATCH,
    _('LaTeX')         : GTKSCINTILLA.LEXER_LATEX,
    _('LUA')           : GTKSCINTILLA.LEXER_LUA,
    _('Diff')          : GTKSCINTILLA.LEXER_DIFF,
    _('Apache Config') : GTKSCINTILLA.LEXER_CONF,
    _('Pascal')        : GTKSCINTILLA.LEXER_PASCAL,
    _('Avenue')        : GTKSCINTILLA.LEXER_AVE,
    _('Ada')           : GTKSCINTILLA.LEXER_ADA,
    _('Lisp')          : GTKSCINTILLA.LEXER_LISP,
    _('Ruby')          : GTKSCINTILLA.LEXER_RUBY,
    _('Eiffel')        : GTKSCINTILLA.LEXER_EIFFEL,
    _('Eiffel KW')     : GTKSCINTILLA.LEXER_EIFFELKW,
    _('Tcl')           : GTKSCINTILLA.LEXER_TCL,
    _('NN Crontab')    : GTKSCINTILLA.LEXER_NNCRONTAB,
    _('Bullant')       : GTKSCINTILLA.LEXER_BULLANT,
    _('VB Script')     : GTKSCINTILLA.LEXER_VBSCRIPT,
    _('ASP')           : GTKSCINTILLA.LEXER_ASP,
    _('PHP')           : GTKSCINTILLA.LEXER_PHP,
    _('Baan')          : GTKSCINTILLA.LEXER_BAAN,
}

Styles = {
    GTKSCINTILLA.LEXER_PYTHON : {
        _('White space')               : GTKSCINTILLA.PYTHON_WHITE_SPACE,
        _('Comment line')              : GTKSCINTILLA.PYTHON_COMMENT_LINE,
        _('Number')                    : GTKSCINTILLA.PYTHON_NUMBER,
        _('String double quote')       : GTKSCINTILLA.PYTHON_STRING_DOUBLE,
        _('String single quote')       : GTKSCINTILLA.PYTHON_STRING_SINGLE,
        _('Keyword')                   : GTKSCINTILLA.PYTHON_KEYWORD,
        _('Long string double quote')  : GTKSCINTILLA.PYTHON_TRIPLE_DOUBLE,
        _('Long string single quote')  : GTKSCINTILLA.PYTHON_TRIPLE_SINGLE,
        _('Class name definition')     : GTKSCINTILLA.PYTHON_CLASS_STATEMENT,
        _('Function name definition')  : GTKSCINTILLA.PYTHON_DEF_STATEMENT,
        _('Operator')                  : GTKSCINTILLA.PYTHON_OPERATOR,
        _('Identifier')                : GTKSCINTILLA.PYTHON_IDENTIFIER,
        _('Comment block')             : GTKSCINTILLA.PYTHON_COMMENT_BLOCK,
        _('String EOL')                : GTKSCINTILLA.PYTHON_STRING_EOL },
    GTKSCINTILLA.LEXER_CPP : {
        _('White space')           : GTKSCINTILLA.CPP_WHITE_SPACE,
        _('Comment block')         : GTKSCINTILLA.CPP_COMMENT_BLOCK,
        _('Comment line')          : GTKSCINTILLA.CPP_COMMENT_LINE,
        _('Comment doc')           : GTKSCINTILLA.CPP_COMMENT_DOC,
        _('Number')                : GTKSCINTILLA.CPP_NUMBER,
        _('Keyword')               : GTKSCINTILLA.CPP_KEYWORD,
        _('String')                : GTKSCINTILLA.CPP_STRING,
        _('Character')             : GTKSCINTILLA.CPP_CHARACTER,
        _('UUID')                  : GTKSCINTILLA.CPP_UUID,
        _('Preprocessor')          : GTKSCINTILLA.CPP_PREPROCESSOR,
        _('Operator')              : GTKSCINTILLA.CPP_OPERATOR,
        _('Identifier')            : GTKSCINTILLA.CPP_IDENTIFIER,
        _('String not closed')     : GTKSCINTILLA.CPP_STRING_EOL,
        _('Verbatim String (C#)')  : GTKSCINTILLA.CPP_STRING_VERBATIM,
        _('Regular expression')    : GTKSCINTILLA.CPP_REGEX,
        _('Comment line doc')      : GTKSCINTILLA.CPP_COMMENT_LINE_DOC,
        _('Keyword 2')             : GTKSCINTILLA.CPP_KEYWORD2,
        _('Comment doc keyword')   : GTKSCINTILLA.CPP_COMMENT_DOC_KEYWORD,
        _('Comment doc error')     : GTKSCINTILLA.CPP_COMMENT_DOC_KEYWORD_ERROR },
    GTKSCINTILLA.LEXER_HTML : {
        _('Text')                  : GTKSCINTILLA.HTML_TEXT,
        _('Tag')                   : GTKSCINTILLA.HTML_TAG,
        _('Tag unknown')           : GTKSCINTILLA.HTML_TAG_UNKNOWN,
        _('Attribute')             : GTKSCINTILLA.HTML_ATTRIBUTE,
        _('Attribute unknown')     : GTKSCINTILLA.HTML_ATTRIBUTE_UNKNOWN,
        _('Number')                : GTKSCINTILLA.HTML_NUMBER,
        _('String double quote')   : GTKSCINTILLA.HTML_DOUBLE_STRING,
        _('String single quote')   : GTKSCINTILLA.HTML_SINGLE_STRING,
        _('Other')                 : GTKSCINTILLA.HTML_OTHER,
        _('Comment')               : GTKSCINTILLA.HTML_COMMENT,
        _('Entity')                : GTKSCINTILLA.HTML_ENTITY,
        _('XML tag ends (/>)')     : GTKSCINTILLA.HTML_TAG_END,
        _('Identifier start (<?)') : GTKSCINTILLA.HTML_XML_START,
        _('Identifier end (?>)')   : GTKSCINTILLA.HTML_XML_END,
        _('Script')                : GTKSCINTILLA.HTML_SCRIPT,
        _('ASP')                   : GTKSCINTILLA.HTML_ASP,
        _('ASP Alt')               : GTKSCINTILLA.HTML_ASP_AT,
        _('CDATA')                 : GTKSCINTILLA.HTML_CDATA,
        _('Question')              : GTKSCINTILLA.HTML_QUESTION,
        _('Value')                 : GTKSCINTILLA.HTML_VALUE,
        
        _('XCode comment')         : GTKSCINTILLA.HTML_XC_COMMENT,
        
        _('SGML default')           : GTKSCINTILLA.HTML_SGML_DEFAULT,
        _('SGML command')           : GTKSCINTILLA.HTML_SGML_COMMAND,
        _('SGML 1st param')         : GTKSCINTILLA.HTML_SGML_1ST_PARAM,
        _('SGML string double')     : GTKSCINTILLA.HTML_SGML_DOUBLE_STRING,
        _('SGML string single')     : GTKSCINTILLA.HTML_SGML_SIMPLE_STRING,
        _('SGML error')             : GTKSCINTILLA.HTML_SGML_ERROR,
        _('SGML special')           : GTKSCINTILLA.HTML_SGML_SPECIAL,
        _('SGML entity')            : GTKSCINTILLA.HTML_SGML_ENTITY,
        _('SGML comment')           : GTKSCINTILLA.HTML_SGML_COMMENT,
        _('SGML 1st param comment') : GTKSCINTILLA.HTML_SGML_1ST_PARAM_COMMENT,
        _('SGML block default')     : GTKSCINTILLA.HTML_SGML_BLOCK_DEFAULT,
        
        _('JS Start')                  : GTKSCINTILLA.HTML_JS_START,
        _('JS Default')                : GTKSCINTILLA.HTML_JS_DEFAULT,
        _('JS Comment block')          : GTKSCINTILLA.HTML_JS_COMMENT_BLOCK,
        _('JS Comment line')           : GTKSCINTILLA.HTML_JS_COMMENT_LINE,
        _('JS Comment doc')            : GTKSCINTILLA.HTML_JS_COMMENT_DOC,
        _('JS Number')                 : GTKSCINTILLA.HTML_JS_NUMBER,
        _('JS Word')                   : GTKSCINTILLA.HTML_JS_WORD,
        _('JS Keyword')                : GTKSCINTILLA.HTML_JS_KEYWORD,
        _('JS String double quote')    : GTKSCINTILLA.HTML_JS_STRING_DOUBLE,
        _('JS String single quote')    : GTKSCINTILLA.HTML_JS_STRING_SINGLE,
        _('JS Symbol')                 : GTKSCINTILLA.HTML_JS_SYMBOL,
        _('JS String EOL')             : GTKSCINTILLA.HTML_JS_EOL,
        
        _('PY Start')                      : GTKSCINTILLA.HTML_PY_START,
        _('PY Default')                    : GTKSCINTILLA.HTML_PY_DEFAULT,
        _('PY Comment line')               : GTKSCINTILLA.HTML_PY_COMMENT_LINE,
        _('PY Number')                     : GTKSCINTILLA.HTML_PY_NUMBER,
        _('PY String double quote')        : GTKSCINTILLA.HTML_PY_STRING_DOUBLE,
        _('PY String single quote')        : GTKSCINTILLA.HTML_PY_STRING_SINGLE,
        _('PY Keyword')                    : GTKSCINTILLA.HTML_PY_KEYWORD,
        _('PY Long string double quote')   : GTKSCINTILLA.HTML_PY_TRIPLE_SINGLE,
        _('PY Long string single quote')   : GTKSCINTILLA.HTML_PY_TRIPLE_DOUBLE,
        _('PY Class name definition')      : GTKSCINTILLA.HTML_PY_CLASS_STATEMENT,
        _('PY Function name definition')   : GTKSCINTILLA.HTML_PY_DEF_STATEMENT,
        _('PY Operator')                   : GTKSCINTILLA.HTML_PY_OPERATOR,
        _('PY Identifier')                 : GTKSCINTILLA.HTML_PY_IDENTIFIER,
        
        _('PHP Default')               : GTKSCINTILLA.HTML_PHP_DEFAULT,
        _('PHP String double quote')   : GTKSCINTILLA.HTML_PHP_STRING_DOUBLE,
        _('PHP String single quote')   : GTKSCINTILLA.HTML_PHP_STRING_SINGLE,
        _('PHP Keyword')               : GTKSCINTILLA.HTML_PHP_KEYWORD,
        _('PHP Number')                : GTKSCINTILLA.HTML_PHP_NUMBER,
        _('PHP Variable')              : GTKSCINTILLA.HTML_PHP_VARIABLE,
        _('PHP Comment block')         : GTKSCINTILLA.HTML_PHP_COMMENT_BLOCK,
        _('PHP Comment line')          : GTKSCINTILLA.HTML_PHP_COMMENT_LINE,
        _('PHP Variable in string')    : GTKSCINTILLA.HTML_PHP_STRING_VARIABLE,
        _('PHP Operator')              : GTKSCINTILLA.HTML_PHP_OPERATOR },
    GTKSCINTILLA.LEXER_XML : {
        _('Text')                  : GTKSCINTILLA.XML_DEFAULT,
        _('Tag')                   : GTKSCINTILLA.XML_TAG,
        _('Tag unknown')           : GTKSCINTILLA.XML_TAG_UNKNOWN,
        _('Attribute')             : GTKSCINTILLA.XML_ATTRIBUTE,
        _('Attribute unknown')     : GTKSCINTILLA.XML_ATTRIBUTE_UNKNOWN,
        _('Number')                : GTKSCINTILLA.XML_NUMBER,
        _('String double quote')   : GTKSCINTILLA.XML_DOUBLE_STRING,
        _('String single quote')   : GTKSCINTILLA.XML_SINGLE_STRING,
        _('Other')                 : GTKSCINTILLA.XML_OTHER,
        _('Comment')               : GTKSCINTILLA.XML_COMMENT,
        _('Entity')                : GTKSCINTILLA.XML_ENTITY,
        _('Tag ends (/>)')         : GTKSCINTILLA.XML_TAG_END,
        _('Identifier start (<?)') : GTKSCINTILLA.XML_START,
        _('Identifier end (?>)')   : GTKSCINTILLA.XML_SCRIPT,
        _('CDATA')                 : GTKSCINTILLA.XML_CDATA,
        _('Question')              : GTKSCINTILLA.XML_QUESTION },
    GTKSCINTILLA.LEXER_PERL : {
        _('White space')           : GTKSCINTILLA.PERL_WHITE_SPACE,
        _('Error')                 : GTKSCINTILLA.PERL_ERROR,
        _('Comment line')          : GTKSCINTILLA.PERL_COMMENT_LINE,
        _('POD')                   : GTKSCINTILLA.PERL_POD,
        _('Number')                : GTKSCINTILLA.PERL_NUMBER,
        _('Keyword')               : GTKSCINTILLA.PERL_KEYWORD,
        _('String')                : GTKSCINTILLA.PERL_STRING,
        _('Character')             : GTKSCINTILLA.PERL_CHARACTER,
        _('Punctuation')           : GTKSCINTILLA.PERL_PUNCTUATION,
        _('Preprocessor')          : GTKSCINTILLA.PERL_PREPROCESSOR,
        _('Operator')              : GTKSCINTILLA.PERL_OPERATOR,
        _('Identifier')            : GTKSCINTILLA.PERL_IDENTIFIER,
        _('Scalar')                : GTKSCINTILLA.PERL_SCALAR,
        _('Array')                 : GTKSCINTILLA.PERL_ARRAY,
        _('Hash')                  : GTKSCINTILLA.PERL_HASH,
        _('Symbol table')          : GTKSCINTILLA.PERL_SYMBOL_TABLE,
        _('Regular expression')    : GTKSCINTILLA.PERL_REGEX,
        _('Regex substitution')    : GTKSCINTILLA.PERL_REGSUBST,
        _('Long quote')            : GTKSCINTILLA.PERL_LONG_QUOTE,
        _('Backticks')             : GTKSCINTILLA.PERL_BACKTICKS,
        _('Data section')          : GTKSCINTILLA.PERL_DATA_SECTION,
        _('Here doc delimiter')    : GTKSCINTILLA.PERL_HERE_DELIM,
        _('Here doc Q')            : GTKSCINTILLA.PERL_HERE_Q,
        _('Here doc QQ')           : GTKSCINTILLA.PERL_HERE_QQ,
        _('Here doc QX')           : GTKSCINTILLA.PERL_HERE_QX,
        _('String Q')              : GTKSCINTILLA.PERL_STRING_Q,
        _('String QQ')             : GTKSCINTILLA.PERL_STRING_QQ,
        _('String QX')             : GTKSCINTILLA.PERL_STRING_QX,
        _('String QR')             : GTKSCINTILLA.PERL_STRING_QR,
        _('String QW')             : GTKSCINTILLA.PERL_STRING_QW },
    GTKSCINTILLA.LEXER_LATEX : {
        _('White space')   : GTKSCINTILLA.LATEX_WHITE_SPACE,
        _('Command')       : GTKSCINTILLA.LATEX_COMMAND,
        _('Tag')           : GTKSCINTILLA.LATEX_TAG,
        _('Math')          : GTKSCINTILLA.LATEX_MATH,
        _('Comment')       : GTKSCINTILLA.LATEX_COMMENT },
    GTKSCINTILLA.LEXER_LUA : {
        _('White space')       : GTKSCINTILLA.LUA_WHITE_SPACE,
        _('Comment block')     : GTKSCINTILLA.LUA_COMMENT_BLOCK,
        _('Comment line')      : GTKSCINTILLA.LUA_COMMENT_LINE,
        _('Comment doc')       : GTKSCINTILLA.LUA_COMMENT_DOC,
        _('Number')            : GTKSCINTILLA.LUA_NUMBER,
        _('Keyword')           : GTKSCINTILLA.LUA_KEYWORD,
        _('String')            : GTKSCINTILLA.LUA_STRING,
        _('Character')         : GTKSCINTILLA.LUA_CHARACTER,
        _('String literal')    : GTKSCINTILLA.LUA_LITERAL_STRING,
        _('Preprocessor')      : GTKSCINTILLA.LUA_PREPROCESSOR,
        _('Operator')          : GTKSCINTILLA.LUA_OPERATOR,
        _('Identifier')        : GTKSCINTILLA.LUA_IDENTIFIER,
        _('String EOL')        : GTKSCINTILLA.LUA_STRING_EOL },
    GTKSCINTILLA.LEXER_ERRORLIST : {
        _('White space')   : GTKSCINTILLA.ERROR_DEFAULT,
        _('Python')        : GTKSCINTILLA.ERROR_PYTHON,
        _('GCC')           : GTKSCINTILLA.ERROR_GCC,
        _('Microsoft')     : GTKSCINTILLA.ERROR_MS,
        _('Command')       : GTKSCINTILLA.ERROR_CMD,
        _('Borland')       : GTKSCINTILLA.ERROR_BORLAND,
        _('Perl')          : GTKSCINTILLA.ERROR_PERL,
        _('Diff changed')  : GTKSCINTILLA.ERROR_DIFF_CHANGED,
        _('Diff added')    : GTKSCINTILLA.ERROR_DIFF_ADDITION,
        _('Diff removed')  : GTKSCINTILLA.ERROR_DIFF_DELETION,
        _('Diff message')  : GTKSCINTILLA.ERROR_DIFF_MESSAGE },
    GTKSCINTILLA.LEXER_BATCH : {
        _('White space')   : GTKSCINTILLA.BATCH_DEFAULT,
        _('Comment')       : GTKSCINTILLA.BATCH_COMMENT,
        _('Keyword')       : GTKSCINTILLA.BATCH_WORD,
        _('Label')         : GTKSCINTILLA.BATCH_LABEL,
        _('Hide (@)')      : GTKSCINTILLA.BATCH_HIDE,
        _('Command')       : GTKSCINTILLA.BATCH_COMMAND,
        _('Variable')      : GTKSCINTILLA.BATCH_IDENTIFIER,
        _('Operator')      : GTKSCINTILLA.BATCH_OPERATOR },
    GTKSCINTILLA.LEXER_MAKEFILE : {
        _('Default')       : GTKSCINTILLA.MAKEFILE_WHITE_SPACE,
        _('Comment')       : GTKSCINTILLA.MAKEFILE_COMMENT,
        _('Preprocessor')  : GTKSCINTILLA.MAKEFILE_PREPROCESSOR,
        _('Variable')      : GTKSCINTILLA.MAKEFILE_IDENTIFIER,
        _('Operator')      : GTKSCINTILLA.MAKEFILE_OPERATOR,
        _('Error')         : GTKSCINTILLA.MAKEFILE_IDEOL },
    GTKSCINTILLA.LEXER_CONF : {
        _('White space')       : GTKSCINTILLA.CONF_DEFAULT,
        _('Comment')           : GTKSCINTILLA.CONF_COMMENT,
        _('Number')            : GTKSCINTILLA.CONF_NUMBER,
        _('Identifier')        : GTKSCINTILLA.CONF_IDENTIFIER,
        _('File extension')    : GTKSCINTILLA.CONF_EXTENSION,
        _('Parameter')         : GTKSCINTILLA.CONF_PARAMETER,
        _('String')            : GTKSCINTILLA.CONF_STRING,
        _('Operator')          : GTKSCINTILLA.CONF_OPERATOR,
        _('IP address')        : GTKSCINTILLA.CONF_IP,
        _('Directive')         : GTKSCINTILLA.CONF_DIRECTIVE },
    GTKSCINTILLA.LEXER_AVE : {
        _('White space')   : GTKSCINTILLA.AVE_DEFAULT,
        _('Comment')       : GTKSCINTILLA.AVE_COMMENT,
        _('Number')        : GTKSCINTILLA.AVE_NUMBER,
        _('Word')          : GTKSCINTILLA.AVE_WORD,
        _('Keyword')       : GTKSCINTILLA.AVE_KEYWORD,
        _('Statement')     : GTKSCINTILLA.AVE_STATEMENT,
        _('String')        : GTKSCINTILLA.AVE_STRING,
        _('Enum')          : GTKSCINTILLA.AVE_ENUM,
        _('String EOL')    : GTKSCINTILLA.AVE_STRINGEOL,
        _('Identifier')    : GTKSCINTILLA.AVE_IDENTIFIER,
        _('Operator')      : GTKSCINTILLA.AVE_OPERATOR },
    GTKSCINTILLA.LEXER_ADA : {
        _('White space')   : GTKSCINTILLA.ADA_DEFAULT,
        _('Comment')       : GTKSCINTILLA.ADA_COMMENT,
        _('Number')        : GTKSCINTILLA.ADA_NUMBER,
        _('Keyword')       : GTKSCINTILLA.ADA_WORD,
        _('String')        : GTKSCINTILLA.ADA_STRING,
        _('Character')     : GTKSCINTILLA.ADA_CHARACTER,
        _('Operator')      : GTKSCINTILLA.ADA_OPERATOR,
        _('Identifier')    : GTKSCINTILLA.ADA_IDENTIFIER,
        _('String EOL')    : GTKSCINTILLA.ADA_STRINGEOL },
    GTKSCINTILLA.LEXER_BAAN : {
    },
    GTKSCINTILLA.LEXER_LISP : {
        _('Default')       : GTKSCINTILLA.LISP_DEFAULT,
        _('Comment')       : GTKSCINTILLA.LISP_COMMENT,
        _('Number')        : GTKSCINTILLA.LISP_NUMBER,
        _('Keyword')       : GTKSCINTILLA.LISP_KEYWORD,
        _('String')        : GTKSCINTILLA.LISP_STRING,
        _('String EOL')    : GTKSCINTILLA.LISP_STRING_EOL,
        _('Identifier')    : GTKSCINTILLA.LISP_IDENTIFIER,
        _('Operator')      : GTKSCINTILLA.LISP_OPERATOR },
    GTKSCINTILLA.LEXER_RUBY : {
        _('White space')               : GTKSCINTILLA.RUBY_WHITE_SPACE,
        _('Comment line')              : GTKSCINTILLA.RUBY_COMMENT_LINE,
        _('Number')                    : GTKSCINTILLA.RUBY_NUMBER,
        _('String double quote')       : GTKSCINTILLA.RUBY_STRING_DOUBLE,
        _('String single quote')       : GTKSCINTILLA.RUBY_STRING_SINGLE,
        _('Keyword')                   : GTKSCINTILLA.RUBY_KEYWORD,
        _('Long string double quote')  : GTKSCINTILLA.RUBY_TRIPLE_DOUBLE,
        _('Long string single quote')  : GTKSCINTILLA.RUBY_TRIPLE_SINGLE,
        _('Class name definition')     : GTKSCINTILLA.RUBY_CLASS_STATEMENT,
        _('Function name definition')  : GTKSCINTILLA.RUBY_DEF_STATEMENT,
        _('Operator')                  : GTKSCINTILLA.RUBY_OPERATOR,
        _('Identifier')                : GTKSCINTILLA.RUBY_IDENTIFIER,
        _('Comment block')             : GTKSCINTILLA.RUBY_COMMENT_BLOCK,
        _('String EOL')                : GTKSCINTILLA.RUBY_STRING_EOL },
    GTKSCINTILLA.LEXER_EIFFEL : {
    },
    GTKSCINTILLA.LEXER_EIFFELKW : {
    },
    GTKSCINTILLA.LEXER_NNCRONTAB : {
    },
    GTKSCINTILLA.LEXER_SQL : {
        _('White space')           : GTKSCINTILLA.SQL_WHITE_SPACE,
        _('Comment block')         : GTKSCINTILLA.SQL_COMMENT_BLOCK,
        _('Comment line')          : GTKSCINTILLA.SQL_COMMENT_LINE,
        _('Comment doc')           : GTKSCINTILLA.SQL_COMMENT_DOC,
        _('Number')                : GTKSCINTILLA.SQL_NUMBER,
        _('Keyword')               : GTKSCINTILLA.SQL_KEYWORD,
        _('String double quote')   : GTKSCINTILLA.SQL_DOUBLE_STRING,
        _('String single quote')   : GTKSCINTILLA.SQL_SINGLE_STRING,
        _('Symbols')               : GTKSCINTILLA.SQL_SYMBOLS,
        _('Preprocessor')          : GTKSCINTILLA.SQL_PREPROCESSOR,
        _('Operator')              : GTKSCINTILLA.SQL_OPERATOR,
        _('Identifier')            : GTKSCINTILLA.SQL_IDENTIFIER,
        _('String EOL')            : GTKSCINTILLA.SQL_STRING_EOL },
    GTKSCINTILLA.LEXER_VB : {
        _('White space')       : GTKSCINTILLA.VB_WHITE_SPACE,
        _('Comment block')     : GTKSCINTILLA.VB_COMMENT_BLOCK,
        _('Comment line')      : GTKSCINTILLA.VB_COMMENT_LINE,
        _('Comment doc')       : GTKSCINTILLA.VB_COMMENT_DOC,
        _('Number')            : GTKSCINTILLA.VB_NUMBER,
        _('Keyword')           : GTKSCINTILLA.VB_WORD,
        _('String')            : GTKSCINTILLA.VB_STRING,
        _('Character')         : GTKSCINTILLA.VB_CHARACTER,
        _('UUID')              : GTKSCINTILLA.VB_UUID,
        _('Preprocessor')      : GTKSCINTILLA.VB_PREPROCESSOR,
        _('Operator')          : GTKSCINTILLA.VB_OPERATOR,
        _('Identifier')        : GTKSCINTILLA.VB_IDENTIFIER,
        _('String EOL')        : GTKSCINTILLA.VB_STRING_EOL },
    GTKSCINTILLA.LEXER_PROPERTIES : {
        _('White space')       : GTKSCINTILLA.PROPERTIES_DEFAULT,
        _('Comment')           : GTKSCINTILLA.PROPERTIES_COMMENT,
        _('Section')           : GTKSCINTILLA.PROPERTIES_SECTION,
        _('Assignment')        : GTKSCINTILLA.PROPERTIES_OPERATOR,
        _('Default value (@)') : GTKSCINTILLA.PROPERTIES_DEFAULT_VAL
        },
    GTKSCINTILLA.LEXER_DIFF : {
        _('Context lines') : GTKSCINTILLA.DIFF_DEFAULT,
        _('Comment')       : GTKSCINTILLA.DIFF_COMMENT,
        _('Command')       : GTKSCINTILLA.DIFF_COMMAND,
        _('File name')     : GTKSCINTILLA.DIFF_FILES,
        _('Position')      : GTKSCINTILLA.DIFF_POSITION,
        _('Line removed')  : GTKSCINTILLA.DIFF_DEL_LINE,
        _('Line added')    : GTKSCINTILLA.DIFF_ADD_LINE },
    GTKSCINTILLA.LEXER_PASCAL : {
        _('White space')           : GTKSCINTILLA.PASCAL_WHITE_SPACE,
        _('Comment block')         : GTKSCINTILLA.PASCAL_COMMENT_BLOCK,
        _('Comment line')          : GTKSCINTILLA.PASCAL_COMMENT_LINE,
        _('Comment doc')           : GTKSCINTILLA.PASCAL_COMMENT_DOC,
        _('Number')                : GTKSCINTILLA.PASCAL_NUMBER,
        _('Keyword')               : GTKSCINTILLA.PASCAL_KEYWORD,
        _('String double quote')   : GTKSCINTILLA.PASCAL_DOUBLE_STRING,
        _('String single quote')   : GTKSCINTILLA.PASCAL_SINGLE_STRING,
        _('Symbols')               : GTKSCINTILLA.PASCAL_SYMBOLS,
        _('Preprocessor')          : GTKSCINTILLA.PASCAL_PREPROCESSOR },
}

def _merge_font(font):
    """Merge the given X font descripton with the deafult font"""
    df = string.split(Moleskine.app.prefs['default style/font'], '-')
    sf = string.split(font, '-')
    for i in range(min(len(df), len(sf))):
        if sf[i] == '*':
            sf[i] = df[i]
    return string.join(sf, '-')

def _split_font(font):
    """Split the given X font descripton from the deafult font
    
    If the given font is equivalent to the default, the function returns None.
    """
    df = string.split(Moleskine.app.prefs['default style/font'], '-')
    sf = string.split(font, '-')
    count = 0
    for i in range(min(len(df), len(sf))):
        if sf[i] == df[i]:
            sf[i] = '*'
            count = count + 1
    if count == min(len(df), len(sf)):
        return None
    else:
        return string.join(sf, '-')[1:]

class Language:
    def __init__(self, name, lexer):
        self.name = name
        if lexer is None:
            self.load()
        else:
            self.lexer = lexer
            self.init_lang()
            self.init_styles()
    
    def init_lang(self):
        self.keywords = ['', '', '', '', '']
        self.tab_size = 4
        self.soft_tabs = 1
        self.indent_size = 4
        self.show_indent = 1
        self.auto_indent = 1
        self.edge_indicator = 0
        self.edge_column = 80
        self.edge_color = 0
        self.rules = []

    def init_styles(self):
        self.styles = {}
        if not Styles.has_key(self.lexer):
            self.styles[0] = {
                'font': None,
                'fore': None,
                'back': None }
            return
        for i in Styles[self.lexer].values():
            self.styles[i] = {
                'font': None,
                'fore': None,
                'back': None }
    
    def load(self):
        gnome.config.push_prefix(os.path.join(_CONFIG_FILE, self.name) + '/')
        
        self.lexer = gnome.config.get_int('lexer')
        self.api_file = gnome.config.get_string('api_file')
        self.keywords = [None, None, None, None, None]
        self.keywords[0] = gnome.config.get_string('keywords1')
        self.keywords[1] = gnome.config.get_string('keywords2')
        self.keywords[2] = gnome.config.get_string('keywords3')
        self.keywords[3] = gnome.config.get_string('keywords4')
        self.keywords[4] = gnome.config.get_string('keywords5')
        self.tab_size = gnome.config.get_int('tab_size')
        self.soft_tabs = gnome.config.get_bool('soft_tabs')
        self.indent_size = gnome.config.get_int('indent_size')
        self.show_indent = gnome.config.get_bool('show_indent')
        self.auto_indent = gnome.config.get_bool('auto_indent')
        self.edge_indicator = gnome.config.get_int('edge_indicator')
        self.edge_column = gnome.config.get_int('edge_column')
        self.edge_color = gnome.config.get_int('edge_color')
        
        self.load_styles()
        self.load_rules()
        
        gnome.config.pop_prefix()
    
    def load_rules(self):
        self.rules = []
        rules_count = gnome.config.get_int('rules_count')
        for i in range(rules_count):
            type = gnome.config.get_int('type_%i_type' % i)
            rule = gnome.config.get_string('type_%i_rule' % i)
            self.rules.append((type, rule))
    
    def load_styles(self):
        self.init_styles()
        
        if not Styles.has_key(self.lexer):
            return
        
        for style in Styles[self.lexer].values():
            s = self.styles[style]
            s['font'] = gnome.config.get_string('style_%i_font' % style)
            s['fore'] = gnome.config.get_string('style_%i_fore' % style)
            s['back'] = gnome.config.get_string('style_%i_back' % style)
    
    def save(self):
        gnome.config.push_prefix(os.path.join(_CONFIG_FILE, self.name) + '/')
        
        gnome.config.set_int('lexer', self.lexer)
        if self.api_file is not None:
            gnome.config.set_string('api_file', self.api_file)
        else:
            gnome.config.set_string('api_file', '')
        if self.keywords[0] is not None:
            gnome.config.set_string('keywords1', self.keywords[0])
        if self.keywords[1] is not None:
            gnome.config.set_string('keywords2', self.keywords[1])
        if self.keywords[2] is not None:
            gnome.config.set_string('keywords3', self.keywords[2])
        if self.keywords[3] is not None:
            gnome.config.set_string('keywords4', self.keywords[3])
        if self.keywords[4] is not None:
            gnome.config.set_string('keywords5', self.keywords[4])
        gnome.config.set_int('tab_size', self.tab_size)
        gnome.config.set_bool('soft_tabs', self.soft_tabs)
        gnome.config.set_int('indent_size', self.indent_size)
        gnome.config.set_bool('show_indent', self.show_indent)
        gnome.config.set_bool('auto_indent', self.auto_indent)
        gnome.config.set_int('edge_indicator', self.edge_indicator)
        gnome.config.set_int('edge_column', self.edge_column)
        gnome.config.set_int('edge_color', self.edge_color)
        
        self.save_styles()
        self.save_rules()
        
        gnome.config.pop_prefix()
    
    def save_rules(self):
        gnome.config.set_int('rules_count', len(self.rules))
        for i in range(len(self.rules)):
            gnome.config.set_int('type_%i_type' % i, self.rules[i][0])
            gnome.config.set_string('type_%i_rule' % i, self.rules[i][1])
    
    def save_styles(self):
        if not Styles.has_key(self.lexer):
            return
        
        for style in self.styles.keys():
            s = self.styles[style]
            if s['font'] is not None:
                if Moleskine.app.prefs['default style/font'] != s['font']:
                    gnome.config.set_string('style_%i_font' % style, s['font'])
                else:
                    s['font'] = None
            if s['fore'] is not None:
                if Moleskine.app.prefs['default style/fore'] != s['fore']:
                    gnome.config.set_string('style_%i_fore' % style, s['fore'])
                else:
                    s['fore'] = None
            if s['back'] is not None:
                if Moleskine.app.prefs['default style/back'] != s['back']:
                    gnome.config.set_string('style_%i_back' % style, s['back'])
                else:
                    s['back'] = None

class LanguagesList(UserDict):
    def new(self, name, lexer = None):
        self.data[name] = Language(name, lexer)
    
    def remove(self, name):
        del self.data[name]
    
    def load(self):
        gnome.config.push_prefix(_CONFIG_FILE)
        stock_langs = gnome.config.get_string('Languages/stock_languages')
        langs = gnome.config.get_string('Languages/languages')
        gnome.config.pop_prefix()
        
        langs_list = []
        if langs is not None:
            langs_list = string.split(langs, '\n')
        stock_langs_list = []
        if stock_langs is not None:
            stock_langs_list = string.split(stock_langs, '\n')
        
        d = {}
        for i in langs_list:
            d[i] = None
        for i in stock_langs_list:
            d[i] = None
        langs_list = d.keys()
        
        for lang_name in langs_list:
            self.new(lang_name)
        
        gnome.config.drop_all()
    
    def save(self):
        gnome.config.clean_file(_CONFIG_FILE)
        gnome.config.push_prefix(_CONFIG_FILE)
        langs = string.join(self.keys(), '\n')
        gnome.config.set_string('Languages/languages', langs)
        gnome.config.pop_prefix()
        
        for lang in self.values():
            lang.save()
        
        gnome.config.sync()
        gnome.config.drop_all()
