
########################################################################
#  Copyright (c) 1999 Annelise Anderson
#  All rights reserved.
# 
#  Redistribution and use in source and binary forms, with or without
#  modification, are permitted provided that the following conditions
#  are met:
#  1. Redistributions of source code must retain the above copyright
#     notice, this list of conditions and the following disclaimer.
#  2. Redistributions in binary form must reproduce the above copyright
#     notice, this list of conditions and the following disclaimer in the
#     documentation and/or other materials provided with the distribution.
# 
#  THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
#  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
#  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
#  ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
#  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
#  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
#  OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
#  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
#  LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
#  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
#  SUCH DAMAGE.
# 
# 	$Id: pagin.awk,v 1.0 1999/08/26 aanderson Exp $
#######################################################################

# pagin.awk calculates how many lines to print based on the lines per
# inch or vmi printer code, ejects the page, and prints a page number
# on the next page in the upper-right corner in a default font.  It then
# switches back to the font in use before.  It moves footnotes to
# the end of the file, places footnote numbers in the text, and
# numbers the footnotes.  Command line option double=1 will turn on
# double-spacing if vmi codes have been used, as they will be by default;
# double=2 should do space-and-a-half.  Also inchesperpage (the portrait
# default) can be set as a command-line variable, as can landlen (the
# default for landscape printing).  If landscape printing is enabled
# with pcl code, landlen will be used instead of inchesperpage.
#                             --Annelise Anderson, August 20, 1999

BEGIN {
	DEFAULT = "(s0p10h0s0b4099T"  # default font for page numbers
	count = 0
	curfont = ""
	orient = 0
	double = 0
	linesperinch = 6
	inchesperpage = 9.5 
	landlen = 6.5
	# change 64 and 92 below to get page number further right or left
	spaceperline = 1 / linesperinch
	pn = 2
	i = 1
	j = 1
	k = 1
}

# Find the footnote text and save it for later printing.  It assumes
# text is marked with FNB at the beginning and FNE at the end.
# Number the footnotes to be printed.

/FNB/,/FNE/ {	
	footnote[i] = sprintf("%s\n", $0)
	if ($0 ~ /FNB/)
		{
		sub(/FNB/,j ".",footnote[i])
		++j
		}
	if ($0 ~ /FNE/)
		{
		sub(/FNE/,"\n", footnote[i])
		}
	++i
	itotal = i
	next
}  

function do_pagenum()
{
		print ("&l0H"  )
		if ( !orient && !double )
			printf ("\n%s%64s\n%s\n", DEFAULT, pn, curfont )
		if ( !orient && double )
			printf ("%s%64s%s\n", DEFAULT, pn, curfont )
		if ( orient && !double )
			printf ("\n%s%92s\n%s\n", DEFAULT, pn, curfont )
		if ( orient && double  )
			printf ("%s%92s%s\n", DEFAULT, pn, curfont )
		pn++
		return
}

# If there's a printer reset at the end of the file, save it so
# it can be printed after the footnotes, if there are any.

{
if ($0 ~ /[E][ \t]*$/)
	{
		endcode = sprintf("%s",$0)
		next
	}
}

# If landscape, adjust inchesperpage  
{ 
if ($0 ~ /&l1O/)
	inchesperpage = landlen
}

# Calculate how much space is being used by each line to be printed
# using any lines per inch codes found or any vmi codes.

{
if ( match($0,/&l[1-9]+D/ )) # lines per inch code; vmi is better
{
	linesperinch = substr($0, RSTART + 2, 1) 
	spaceperline = 1 / linesperinch 
}
}

{
if ( match($0,/&l[0-9][0-8]*C/ ))
{
	vmi = substr($0, RSTART + 2, 2) 
# doublespace option here; also space and a half	
	if ( double == 1 )
	{ 
		sub(/l[0-9][0-8]*[C]/, (("l" (vmi * 2)) "C"), $0 )
		spaceperline = vmi / 24 
	}
	else if ( double == 2 )
	{ 
		sub(/l[0-9][0-8]*[C]/, ("l" (int(vmi * 1.5) "C")), $0 )
		spaceperline = vmi / 36 
	}
	else spaceperline = vmi / 48  
}
# Find font strings and save the last one in case it needs to
# be printed

oldtypenr = typenr
oldtype = curtype
curstr = $0
{
while (match(curstr,/s[01]p[0-9][0-9]?[vh][0-5]s[0-4]b[0-9]+T|s0p16.67h8.5v0s0b0T/ ))
	{
		curtype = "(" substr(curstr, RSTART, RLENGTH)
		curstr = substr(curstr,RSTART + RLENGTH)
		typenr = NR
	}
}

# Find style strings (bold, italic, underline) and save most recent

oldstylenr = stylenr
oldstyle = curstyle
curstr = $0
{
while (match(curstr,/[&(][sd][0-9]*[SBD@]/))
{
	curstyle = "" substr(curstr,RSTART,RLENGTH)
	curstr = substr(curstr,RSTART + RLENGTH)
	stylenr = NR
}
}

# Establish current font, type and style; what happens if a style
# has been superceded by a font?  Probably taken care of.

if (oldstylenr < oldtypenr )
	curfont = oldtype
else curfont = oldtype oldstyle

# Establish orientation
	orientstr = $0
	while (match(orientstr,/&l[01]O/))
	{
		orient = substr(orientstr, RSTART + 2, 1)
		orientstr = substr(orientstr, RSTART + RLENGTH)
	}

# Number any footnotes in the text sequentially.
# Numbers will be superscripted if proper printer code is used in text.
# Print the next line if there's still room on the page
# Otherwise kick the page out, print a page number on the next page
# in the default font, return to prior font

fnstr = $0
while (match(fnstr,/FN#/))
{
	sub(/FN#/,k)
		++k
	fnstr = substr(fnstr, RSTART + RLENGTH)
}
{
count +=  spaceperline
	if (count < inchesperpage )
	print ($0)
	if (count >= inchesperpage )
	{ 
		do_pagenum()
		print ($0)
		count = 4 * spaceperline
		}
	}
} 

# print the footnotes; continue to paginate; print the endcode.

END {
	{
	for ( i = 1; i <= itotal; ++i)
	{
		count += spaceperline
		if (count < inchesperpage )
			printf("%s", footnote[i])
    	if (count >= inchesperpage )
    	{ 
			do_pagenum()
			printf("%s",footnote[i])
			count = 3 * spaceperline
		}
	}
}
	printf("%s", endcode)
}

