/* 
Copyright Paul James Mutton, 2001-2003, http://www.jibble.org/

This file is part of PircBot.

You may freely use PircBot to make your own IRC bots and clients.

Any software that makes use of the PircBot package must clearly
state that it makes use of the PircBot Java IRC Bot Framework (or
some similar wording) found at http://www.jibble.org/

This source code, or parts thereof, may not be distributed in any
modified form without permission from the author.

PircBot is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*/

package org.jibble.pircbot;

import java.io.*;
import java.net.*;

/**
 * A Thread which is responsible for sending messages to the IRC server.
 * Messages are obtained from the outgoing message queue and sent
 * immediately if possible.  If there is a flood of messages, then to
 * avoid getting kicked from a channel, we put a small delay between
 * each one.
 *
 * @author  Paul James Mutton,
 *          <a href="http://www.jibble.org/">http://www.jibble.org/</a>
 * @version 1.0.3 (Build time: Sat Jan 18 23:29:54 2003)
 */
public class OutputThread extends Thread {
    
    
    /**
     * Constructs an OutputThread for the underlying PircBot.  All messages
     * sent to the IRC server are sent by this OutputThread to avoid hammering
     * the server.  Messages are sent immediately if possible.  If there are
     * multiple messages queued, then there is a delay imposed.
     * 
     * @param bot The underlying PircBot instance.
     * @param outQueue The Queue from which we will obtain our messages.
     */
    protected OutputThread(PircBot bot, Queue outQueue) {
        _bot = bot;
        _outQueue = outQueue;
    }
    
    
    /**
     * A static method to write a line to a BufferedWriter and then pass
     * the line to the log method of the supplied PircBot instance.
     * 
     * @param bot The underlying PircBot instance.
     * @param writer The BufferedWriter to write to.
     * @param line The line to be written. "\r\n" is appended to the end.
     */
    public static void sendRawLine(PircBot bot, BufferedWriter writer, String line) {
        if (line.length() > bot.getMaxLineLength() - 2) {
            line = line.substring(0, bot.getMaxLineLength() - 2);
        }
        synchronized(writer) {
            try {
                writer.write(line + "\r\n");
                writer.flush();
                bot.log(">>>" + line);
            }
            catch (Exception e) {
                // Silent response - just lose the line.
            }
        }
    }
    
    
    /**
     * This method starts the Thread consuming from the outgoing message
     * Queue and sending lines to the server.
     *
     */
    public void run() {
        boolean running = true;
        while (running) {
            // Small delay to prevent spamming of the channel
            try {
                Thread.sleep(_bot.getMessageDelay());
            }
            catch (InterruptedException e) {
                // Do nothing.
            }
            String line = (String) _outQueue.next();
            _bot.sendRawLine(line);
        }
    }
    
    private PircBot _bot = null;
    private Queue _outQueue = null;
    
}