/* 
 * PVM++ - A C++ library for easy use of PVM
 * Copyright (C) 1997-2001 Sebastian Wilhelmi; University of Karlsruhe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef __PVM_CLASS_HH__
#define __PVM_CLASS_HH__

#include <pvm++/types.hh>

namespace Pvm
{
  /** Singelton for querying the Pvm.
      
      This singelton class represents the global state of the Parallel
      Virtual Machine (PVM). During construction (this happens on first
      call to any PVM++ function) all necessary internal information is
      created and the current tasks is introduced to the PVM. Access to
      the class is given by the global function const PvmClass& Pvm().  */
  
  class Class
  {
  public:
    ~Class ();
    
    /** returns a \Ref{Task}-instance representing the current task.
	I.e. the task, this function is called from. */
    Task I () const;
    
    /** returns the Number of different architectures, that are present
	on the pvm. */
    unsigned int NumOfArchs () const;
    
    /** a set of all hosts in the PVM is returned in the reference
	parameter Result. */
    void Hosts (HostSet &Result) const;
    
    /** a set of all tasks currently running on the PVM is returned in
	the reference parameter Result. */
    void Tasks (TaskSet &Result) const;
    
    /** this function handles all pending messages, for which receive
	actions other than normal receive are defined. Such are message
	handler, automatic unpack, swallow on receive. This is executed
	automatically by all Send() and Receive*() commands of the
	\Ref{Struct} and \Ref{StructSet} classes. If you have
	message handlers installed, it is important to call this
	function before using data, provided by such handlers. E.g. a
	task gets messages from time to time, telling it, which tasks it
	has to send messages to. It then has to provide this information
	to the Send()-function. Send() calls Update(), but when the
	message-handlers are invoked for the received message, this new
	information can't be provided to the Send()-function, as the old
	information is already there. */
    void Update () const;
  private:
    friend class HostTableType;
    static unsigned int IntNumOfArchs;
    int MyId;
    Class ();
    friend const Class & Pvm ();
  };
  
  const Class & Pvm ();

} // namespace Pvm

#endif /* __PVM_CLASS_HH__ */
