import pychart_util
import color
import line_style
import chart_object
import object_set
import types
import canvas
import theme
import fill_style_doc

from scaling import *

_keys = {
    "bgcolor" : (color.T, 1, color.white, "The background color."),
    "line_style": (line_style.T, 1, line_style.default,
                   pychart_util.line_desc),
    "line_interval": (pychart_util.NumType, 0, 3,
                      "The interval between successive stitch lines.")
    }

class T(chart_object.T):
    __doc__ = fill_style_doc.doc
    keys = _keys
##AUTOMATICALLY GENERATED
##END AUTOMATICALLY GENERATED
    def __str__(self):
        s = name_table().lookup(self)
        if s:
            return s
        return "<fillstyle: bg=%s line=%s interval=%s>" % \
               (self.bgcolor, self.line_style, self.line_interval)

class Plain(T):
    """This class just fills the region with the solid background color.
Attributes line_style and line_interval are ignored."""
    def draw(self, x1, y1, x2, y2):
        pass
    
class Diag(T):
    "This class fills the region with diagonal lines."

    def draw(self, x1, y1, x2, y2):
        line_width = self.line_style.width
        interval = self.line_interval * 1.414
        x1 -= line_width
        y1 -= line_width
        x2 += line_width
        y2 += line_width
        height = y2 - y1
        width = x2 - x2
        len = max(height, width)
        curx = x1 - len
        while curx < x2:
            canvas.line(self.line_style, curx, y1, curx+len, y1+len)
            curx += interval

class Rdiag(T):
    """Fills the region with diagonal lines, but tilted in the opposite
direction from fill_style.Diag."""
    def draw(self, x1, y1, x2, y2):    
        line_width = self.line_style.width
        interval = self.line_interval * 1.414

        x1 -= line_width
        y1 -= line_width
        x2 += line_width
        y2 += line_width
        height = y2 - y1
        width = x2 - x2
        len = max(height, width)
        curx = x1
        endx = x2 + len
        while curx < endx:
            canvas.line(self.line_style, curx, y1, curx-len, y1+len)
            curx += interval

class Vert(T):
    "Fills the region with vertical lines"
    def draw(self, x1, y1, x2, y2):    
        interval = self.line_interval
        curx = x1
        while curx < x2:
            canvas.line(self.line_style, curx, y1, curx, y2)
            curx += interval
            
class Horiz(T):
    "Fills the region with horizontal lines"
    def draw(self, x1, y1, x2, y2):
        interval = self.line_interval
        cury = y1
        while cury < y2:
            canvas.line(self.line_style, x1, cury, x2, cury)            
            cury += interval
            
class Stitch(T):
    "Fills the region with horizontal and vertical lines."
    def draw(self, x1, y1, x2, y2):
        interval = self.line_interval
        cury = y1
        while cury < y2:
            canvas.line(self.line_style, x1, cury, x2, cury)
            cury += interval
        curx = x1
        while curx < x2:
            canvas.line(self.line_style, curx, y1, curx, y2)
            curx += interval

class Wave(T):
    "Fills the region with horizontal wavy lines."
    def draw(self, x1, y1, x2, y2):
        x1 = xscale(x1)
        x2 = xscale(x2)
        y1 = yscale(y1)
        y2 = yscale(y2)
        line_width = nscale(self.line_style.width)
        interval = nscale(self.line_interval)
        
        canvas.out.set_line_style(self.line_style)        
        x1 -= line_width
        x2 += line_width
        cury = y1
        half = interval/2.0
        while cury < y2:
            curx = x1
            canvas.out.newpath()
            canvas.out.moveto(curx, cury)
            while curx < x2:
                canvas.out.lineto(curx + half, cury + half)
                canvas.out.lineto(curx + interval, cury)
                curx += interval
            canvas.out.stroke()
            cury += interval

class Vwave(T):
    """Fills the region with vertical wavy lines."""
    def draw(self, x1, y1, x2, y2):
        x1 = xscale(x1)
        x2 = xscale(x2)
        y1 = yscale(y1)
        y2 = yscale(y2)
        line_width = nscale(self.line_style.width)
        interval = nscale(self.line_interval)
        
        canvas.out.set_line_style(self.line_style)
        y1 -= line_width
        y2 += line_width
        curx = x1
        half = interval/2.0
        while curx < x2:
            cury = y1
            canvas.out.newpath()
            canvas.out.moveto(curx, cury)
            while cury < y2:
                canvas.out.lineto(curx + half, cury + half)
                canvas.out.lineto(curx, cury + interval)
                cury += interval
            canvas.out.stroke()
            curx += interval        

class Lines(T):
    """Fills the region with a series of short line segments."""
    def draw(self, x1, y1, x2, y2):
        interval = nscale(self.line_interval)
        cury = y1
        j = 0
        while cury < y2:
            curx = x1
            if j % 2 == 1:
                curx += interval/2.0
            while curx < x2:
                canvas.line(self.line_style, curx, cury,
                            curx+interval/2.0, cury)
                curx += interval * 1.5
            j += 1
            cury += interval

default = Plain()

color_standards = object_set.T()
gray_standards = object_set.T()

def _intern_both(style):
    global color_standards, gray_standards
    color_standards.add(style)
    gray_standards.add(style)
    return style

def _intern_color(style):
    global color_standards, gray_standards    
    color_standards.add(style)
    return style

def _intern_grayscale(style):
    global color_standards, gray_standards    
    gray_standards.add(style)
    color_standards.add(style)
    return style

black = _intern_both(Plain(bgcolor=color.gray_scale(0.0), line_style=None))

#
# Fill styles for grayscale charts.
#
gray70 = _intern_grayscale(Plain(bgcolor=color.gray70, line_style=None))
diag = _intern_grayscale(Diag(line_style=line_style.T(cap_style=2)))
gray30 = _intern_grayscale(Plain(bgcolor=color.gray30, line_style=None))
rdiag = _intern_grayscale(Rdiag(line_style=line_style.T(cap_style=2)))
gray10 = _intern_grayscale(Plain(bgcolor=color.gray10, line_style=None))
diag2 = _intern_grayscale(Diag(line_style=line_style.T(width=3, cap_style=2),
                      line_interval=6))
white = _intern_grayscale(Plain(bgcolor=color.gray_scale(1.0), line_style=None))
rdiag2 = _intern_grayscale(Rdiag(line_style=line_style.T(width=3, cap_style=2),
                        line_interval=6))
vert = _intern_grayscale(Vert())
diag3 = _intern_grayscale(Diag(line_style=line_style.T(width=3, color=color.gray50, cap_style=2),
                      line_interval=6))
gray50 = _intern_grayscale(Plain(bgcolor=color.gray50, line_style=None))
horiz = _intern_grayscale(Horiz())
gray90 = _intern_grayscale(Plain(bgcolor=color.gray90, line_style=None))
rdiag3 = _intern_grayscale(Rdiag(line_style=line_style.T(width=3,
                                                          color=color.gray50,
                                                          cap_style=2),
                        line_interval=6))

wave = _intern_grayscale(Wave(line_style=line_style.T(cap_style=2, join_style=1)))
vwave = _intern_grayscale(Vwave(line_style=line_style.T(cap_style=2, join_style=1)))
stitch = _intern_grayscale(Stitch(line_style=line_style.T(cap_style=2, join_style=1)))
lines = _intern_grayscale(Lines(line_style=line_style.T()))

#
# Fill styles for color charts.
#

red = _intern_color(Plain(bgcolor=color.red))
darkseagreen = _intern_color(Plain(bgcolor=color.darkseagreen))
aquamarine1 = _intern_color(Plain(bgcolor=color.aquamarine1))
_intern_color(gray70)
brown = _intern_color(Plain(bgcolor=color.brown))
darkorchid = _intern_color(Plain(bgcolor=color.darkorchid))    
_intern_color(gray50)
_intern_color(diag)
_intern_color(vert)
_intern_color(horiz)
goldenrod = _intern_color(Plain(bgcolor=color.goldenrod))
_intern_color(rdiag)
_intern_color(white)

standards = None
_name_table = None

def init():
    global standards, _name_table
    if theme.use_color:
        standards = color_standards
    else:
        standards = gray_standards
    _name_table = None

def name_table():
    global _name_table
    if not _name_table:
        _name_table = pychart_util.symbol_lookup_table(globals(), standards)
    return _name_table

init()
theme.add_reinitialization_hook(init)

