/*-*- c++ -*-******************************************************************
 * Qwt Widget Library 
 * Copyright (C) 1997   Josef Wilgen
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *****************************************************************************/

#ifndef QWT_DBLRANGE_H
#define QWT_DBLRANGE_H

#include <qglobal.h>

/*!
  \brief A class which controls a value within an interval

  This class is useful as a base class or a member for sliders.
  It represents an interval of type double within which a value can
  be moved. The value can be either an arbitrary point inside 
  the interval (see QwtDblRange::setValue), or it can be fitted
  into a step raster (see QwtDblRange::fitValue and
  QwtDblRange::incValue).

  As a special case, a QwtDblRange can be periodic, which means that
  a value outside the interval will be mapped to a value inside the
  interval when setValue(), fitValue(), incValue() or incPages() are called.
*/
class QwtDblRange
{
    
private:

    static const double MinRelStep;
    static const double DefaultRelStep;
    static const double MinEps;
    
    double d_minValue;
    double d_maxValue;
    double d_step;
    int d_pageSize;
    double d_value;
    double d_exactValue;
    double d_exactPrevValue;
    double d_prevValue;
    bool d_periodic;

    void setNewValue(double x,int align = 0);
    
public:
    
    QwtDblRange();
    
    virtual void fitValue(double);
    virtual void incValue(int);
    virtual void incPages(int);
    void setPeriodic(bool tf);
    void setRange(double vmin, double vmax, double vstep = 0.0,
		  int pagesize = 1);
    void setStep(double);
    virtual void setValue(double);
    
  /*!
    \brief Returns the value of the second border of the range
    
    maxValue returns the value which has been specified
    as the second parameter in  QwtDblRange::setRange.
    
    \sa setRange()
  */
  double maxValue() const { return d_maxValue; }

  /*!
    \brief Returns the value at the first border of the range
    
    minValue returns the value which has been specified
    as the first parameter in  setRange().
    
    \sa setRange()
  */
  double minValue() const { return d_minValue; }

  /*!
    \brief Returns TRUE if the range is periodic
    \sa setPeriodic()
  */
  bool periodic() const { return d_periodic; }

  double step() const;
  //! Returns the page size in steps.
  int pageSize() const { return d_pageSize; }
  //! Returns the current value.
  double value() const { return d_value; }
    
protected:
  
  /*!
    \brief Returns the exact value
    
    The exact value is the value which QwtDblRange::value would return
    if the value were not adjusted to the step raster. It differs from
    the current value only if QwtDblRange::fitValue or
    QwtDblRange::incValue have been used before. This function
    is intended for internal use in derived classes.
  */
    double exactValue() const { return d_exactValue; }

  //! Returns the exact previous value
    double exactPrevValue() const { return d_exactPrevValue; }

  //! Returns the previous value
    double prevValue() const { return d_prevValue; }
    virtual void valueChange();
    virtual void stepChange();
    virtual void rangeChange();
    
};



#endif




