/*-*- c++ -*-******************************************************************
 * Qwt Widget Library 
 * Copyright (C) 1997   Josef Wilgen
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *****************************************************************************/

#ifndef QWT_LEGEND_H
#define QWT_LEGEND_H

#include "qwt_symbol.h"
#include <qstring.h>
#include <qlist.h>
#include <qtable.h>
#include <qpen.h>

class QPainter;

/*!
  \brief A legend item
  \sa QwtLegend
*/
class QwtLegendItem
{
	
private:

    
    QPen d_pen;
    QwtSymbol d_sym;
    QString d_text;
    long d_key;

public:

    QwtLegendItem();
    QwtLegendItem(const QString &txt, const QwtSymbol &sym, const QPen &pn, long key);

    const QPen& pen() { return d_pen; }
    const QwtSymbol& symbol() { return d_sym; }
    const QString& text() const { return d_text; }
    long key() const { return d_key; }

    void setSymbol(const QwtSymbol &s) {d_sym = s;}
    void setPen(const QPen &pn) {d_pen = pn;}
    void setKey(long key) {d_key = key; }
    void setText(const QString &s) {d_text = s;}
    
    void draw(QPainter *p, const QRect &symRect, const QRect& lblRect, int align);
    
};



/*!
  \brief The legend widget

  The QwtLegend widget is a tabular arrangement of cells,
  called legend items, which consist of a label, a line
  a symbol, and a key of type long as an 
  identifier. The symbol is drawn on top of
  the line at the left side of the label. An item's
  property can be accessed using the item's index,
  which is independent of the key. The cells can
  be arranged in row or column order with QwtLegend::setMaxRows
  or QwtLegend::setMaxCols. Scrollbars are displayed
  automatically if	the cells don't fit into the widget area.
*/
class QwtLegend : public QTable
{
    Q_OBJECT
private:

    int d_align;
    QRect d_symRect;
    QRect d_lblRect;
    int d_maxCols;
    int d_maxRows;

    struct selection 
    {
	bool active;
	int r;
	int c;
    } d_sel;

    uint findIndex(int row, int col);
    void findRowCol(uint index, int &row, int &col);

	virtual void paintFocus(QPainter *, const QRect &);

protected:

    QList<QwtLegendItem> d_item;
    virtual void paintCell(QPainter *p, int row, int col, 
		const QRect &, bool selected);
    virtual void contentsMousePressEvent(QMouseEvent *e);
    virtual void contentsMouseReleaseEvent(QMouseEvent *e);
	virtual QWidget *createEditor(int row, int col, 
		bool initFromCell) const;

	virtual void resizeEvent(QResizeEvent *);

    void resizeCells();
    void resizeTable();
    void updateTable();

public:

    QwtLegend(QWidget *w = 0, const char *name = 0);
    ~QwtLegend();
    
    bool insertItem(const QString &txt, const QwtSymbol &s, const QPen &p,
		    long key, uint index);
    uint appendItem(const QString &txt, const QwtSymbol &s, const QPen &p, long key);
    
    bool setText(uint index, const QString &s);
    bool setSymbol(uint index,  const QwtSymbol &s);
    bool setPen(uint index, const QPen &pn);
    bool setKey(uint index, long key);

    long key(uint index);
    bool removeItem(uint index);

    void setMaxCols(int n);
    void setMaxRows(int n);

    //! the column width
    int colWidth() const { return QTable::columnWidth(0); }
    //! the row height
    int rowHeight() const { return QTable::rowHeight(0); }
	
    void clear();
    QSize sizeHint() const;
    //! the number of items
    uint itemCnt() const { return d_item.count(); }
    uint findFirstKey(long key);

protected:
    virtual void fontChange(const QFont &oldFont);

signals:
    
    /*!
      \brief Emitted when the user presses a mouse button down on an item
      \param index	The item's index
    */
    void pressed(int index);
    /*!
      \brief Emitted when the user clicks on an item.
      \param index The item's index
    */
    void clicked(int index);
    
    
};



#endif
