/*-*- c++ -*-******************************************************************
 * Qwt Widget Library 
 * Copyright (C) 1997   Josef Wilgen
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *****************************************************************************/

#ifndef QWT_PIXFRAME_H
#define QWT_PIXFRAME_H

#include <qframe.h>
#include <qpixmap.h>
#include <qpen.h>
#include "qwt.h"

/*!
  \brief A frame containing a pixmap into which
	can be drawn.

   QwtPixFrame is a frame which contains a pixmap. The size of the
   pixmap is dynamically adjusted in order to fit the frame's
   contents.

  \par Example
  \verbatim
#include "../include/qwt_pixframe.h>

class ParentClass : ...
{
        ...
        QwtPixFrame *pixframe;
        ...
        void newPicture();
        ...
private slots:
        void repaintPixmap();
}

//
//     Connect pixmapResized() to a slot
//
ParentClass::ParentClass(...)
{
        ...
        connect(pixframe, SIGNAL(pixmapResized()), SLOT(repaintPixmap()));
}

//
// Repaint when the pixmap has been resized
//
ParentClass::repaintPixmap()
{
        QPainter p;
        p.begin(pixframe->pixmap());
	redraw_picture(&p, pixframe->pixmap().rect());
        p.end();
        // Note: No updatePixmap() required here
}

//
// Renew the pixmap's contents independently
//
ParentClass::newPicture()
{
        // Change the pixmap's contents
        draw_new_picture(pixframe->pixmap());
         // Generate paint event
        pixframe->updatePixmap();
}
\endverbatim
*/
class QwtPixFrame : public QFrame
{

    Q_OBJECT

private:
    
    QColor d_bg;
       
protected:

    QPixmap d_pix;
    bool d_adjustPixmap;
    
public:

    QwtPixFrame (QWidget *parent, const char *name =0);
    ~QwtPixFrame();

    void setPixmap(const QPixmap &pix);
    /*!
      \return a pointer to the enclosed pixmap
     */
    const QPixmap *pixmap() const { return &d_pix; }
    void fillPixmap(const QColor &c);
    void clearPixmap();
    void updatePixmap();
    void updatePixmapRect(const QRect &r);
    void setPixmapBackground(const QColor &c);
    void setVariableSize(bool tf);
    /*!
      \return the pixmap's background color
      \sa setPixmapBackground(const QColor &c)
     */
    const QColor &pixmapBackground() const {return d_bg;}
    
protected:

    void adjustPixmap();
    void adjustFrame();
    virtual void paintEvent(QPaintEvent *e);
    virtual void resizeEvent(QResizeEvent *e);
    virtual void frameChanged();

signals:

    /*!
      \brief A signal which is emitted when the size of the pixmap
      has been changed.

      When resized, the pixmap is cleared and filled with
      its background color.
     */
    void pixmapResized();
};
    


#endif








