module Irc

  # Keyword class
  #
  # Encapsulates a keyword ("foo is bar" is a keyword called foo, with type
  # is, and has a single value of bar).
  # Keywords can have multiple values, to_s() will choose one at random
  class Keyword
    
    # type of keyword (e.g. "is" or "are")
    attr_reader :type
    
    # type::   type of keyword (e.g "is" or "are")
    # values:: array of values
    # 
    # create a keyword of type +type+ with values +values+
    def initialize(type, values)
      @type = type.downcase
      @values = values
    end

    # pick a random value for this keyword and return it
    def to_s
      if(@values.length > 1)
        Keyword.unescape @values[rand @values.length]
      else
        Keyword.unescape @values[0]
      end
    end

    # describe the keyword (show all values without interpolation)
    def desc
      @values.join(" | ")
    end

    # return the keyword in a form ready to write to a file
    def dump
      Keyword.unescape @values.join("<=or=>")
    end

    # values:: array of values to add
    # add values to a keyword
    def <<(values)
      if(@values.length > 1 || values.length > 1)
        values.each {|v|
          @values << v
        }
      else
        @values[0] += " or " + values[0]
      end
    end

    # unescape special words/characters in a keyword
    def Keyword.unescape(str)
      str.gsub(/\\\|/, "|").gsub(/ \\is /, " is ").gsub(/ \\are /, " are ").gsub(/\\\?(\s*)$/, "?\1")
    end

    # escape special words/characters in a keyword
    def Keyword.escape(str)
      str.gsub(/\|/, "\\|").gsub(/ is /, " \\is ").gsub(/ are /, " \\are ").gsub(/\?(\s*)$/, "\\?\1")
    end
  end

  # keywords class. 
  #
  # Handles all that stuff like "bot: foo is bar", "bot: foo?"
  #
  # Fallback after core and auth have had a look at a message and refused to
  # handle it, checks for a keyword command or lookup, otherwise the message
  # is delegated to plugins
  class Keywords
    
    # create a new Keywords instance, associated to bot +bot+
    def initialize(bot)
      @bot = bot
      @statickeywords = Hash.new
      @keywords = Hash.new
      @keywords["offended"] = Keyword.new("is", ["<reply><who> is offended!"])
      @keywords["lb"] = Keyword.new("is", ["http://linuxbrit.co.uk"])
      @keywords["giblet"] = Keyword.new("is", ["My master!"])
      @keywords["rbot"] = Keyword.new("is", ["<reply>That's me! :-))"])
      scan
      if(File.exist?("#{@bot.botclass}/keywords.rbot"))
        IO.foreach("#{@bot.botclass}/keywords.rbot") do |line|
          if(line =~ /^(.*?)<=(is|are)?=?>(.*)$/)
            lhs = $1
            mhs = $2
            rhs = $3
            mhs = "is" unless mhs
            rhs = Keyword.escape rhs
            values = rhs.split("<=or=>")
            @keywords[lhs] = Keyword.new(mhs, values)
          end
        end
      end
    end
    
    # drop static keywords and reload them from files, picking up any new
    # keyword files that have been added
    def rescan
      @statickeywords = Hash.new
      scan
    end

    # load static keywords from files, picking up any new keyword files that
    # have been added
    def scan
      Dir["#{@bot.botclass}/keywords/*"].each {|f|
        IO.foreach(f) {|line|
          if(line =~ /^(.*?)<=(is|are)?=?>(.*)$/)
            lhs = $1
            mhs = $2
            rhs = $3
            mhs = "is" unless mhs
            rhs = Keyword.escape rhs
            values = rhs.split("<=or=>")
            @statickeywords[lhs] = Keyword.new(mhs, values)
          end
        }
      }
    end

    # save dynamic keywords to file
    def save
      File.open("#{@bot.botclass}/keywords.rbot", "w") do |file|
        @keywords.each do |key, value|
          file.puts "#{key}<=#{value.type}=>#{value.dump}"
        end
      end
    end
    
    # lookup keyword +key+, return it or nil
    def [](key)
      if(@keywords.has_key?(key))
        return @keywords[key]
      elsif(@statickeywords.has_key?(key))
        return @statickeywords[key]
      else
        return nil
      end
    end

    # does +key+ exist as a keyword?
    def has_key?(key)
      @keywords.has_key?(key) || @statickeywords.has_key?(key)
    end

    # m::     PrivMessage containing message info
    # key::   key being queried
    # dunno:: optional, if true, reply "dunno" if +key+ not found
    # 
    # handle a message asking about a keyword
    def keyword(m, key, dunno=true)
       unless(kw = self[key])
         m.reply @bot.lang.get("dunno") if (dunno)
         return
       end
       response = kw.to_s
       response.gsub!(/<who>/, m.sourcenick)
       if(response =~ /^<reply>\s*(.*)/)
         m.reply "#$1"
       elsif(response =~ /^<action>\s*(.*)/)
         @bot.action m.replyto, "#$1"
       elsif(m.public? && response =~ /^<topic>\s*(.*)/)
         topic = $1
         @bot.topic m.target, topic
       else
         m.reply "#{key} #{kw.type} #{response}"
       end
    end

    
    # m::      PrivMessage containing message info
    # target:: channel/nick to tell about the keyword
    # key::    key being queried
    # 
    # handle a message asking the bot to tell someone about a keyword
    def keyword_tell(m, target, key)
      unless(kw = self[key])
        @bot.say m.sourcenick, @bot.lang.get("dunno_about_X") % key
        return
      end
      response = kw.to_s
      response.gsub!(/<who>/, m.sourcenick)
      if(response =~ /^<reply>\s*(.*)/)
        @bot.say target, "#{m.sourcenick} wanted me to tell you: (#{key}) #$1"
        m.reply "okay, I told #{target}: (#{key}) #$1"
      elsif(response =~ /^<action>\s*(.*)/)
        @bot.action target, "#$1 (#{m.sourcenick} wanted me to tell you)"
        m.reply "okay, I told #{target}: * #$1"
      else
        @bot.say target, "#{m.sourcenick} wanted me to tell you that #{key} #{kw.type} #{response}"
        m.reply "okay, I told #{target} that #{key} #{kw.type} #{response}"
      end
    end

    # handle a message which alters a keyword
    # like "foo is bar", or "no, foo is baz", or "foo is also qux"
    def keyword_command(sourcenick, target, lhs, mhs, rhs, quiet=false)
      debug "got keyword command #{lhs}, #{mhs}, #{rhs}"
      overwrite = false
      overwrite = true if(lhs.gsub!(/^no,\s*/, ""))
      also = true if(rhs.gsub!(/^also\s+/, ""))
      values = rhs.split(/\s+\|\s+/)
      lhs = Keyword.unescape lhs
      if(overwrite || also || !has_key?(lhs))
        if(also && has_key?(lhs))
          self[lhs] << values
        else
          @keywords[lhs] = Keyword.new(mhs, values)
        end
        @bot.okay target if !quiet
      elsif(has_key?(lhs))
        @bot.say target, "but #{lhs} #{self[lhs].type} #{self[lhs].desc}" if !quiet
      end
    end

    # return help string for Keywords with option topic +topic+
    def help(topic="")
      case topic
        when "overview"
          return "set: <keyword> is <definition>, overide: no, <keyword> is <definition>, add to definition: <keyword> is also <definition>, random responses: <keyword> is <definition> | <definition> [| ...], plurals: <keyword> are <definition>, escaping: \\is, \\are, \\|, specials: <reply>, <action>, <who>"
        when "set"
          return "set => <keyword> is <definition>"
        when "plurals"
          return "plurals => <keywords> are <definition>"
        when "override"
          return "overide => no, <keyword> is <definition>"
        when "also"
          return "also => <keyword> is also <definition>"
        when "random"
          return "random responses => <keyword> is <definition> | <definition> [| ...]"
        when "get"
          return "asking for keywords => (with addressing) \"<keyword>?\", (without addressing) \"'<keyword>\""
        when "tell"
          return "tell <nick> about <keyword> => if <keyword> is known, tell <nick>, via /msg, its definition"
        when "forget"
          return "forget <keyword> => forget fact <keyword>"
        when "keywords"
          return "keywords => show current keyword counts"
        when "<reply>"
          return "<reply> => normal response is \"<keyword> is <definition>\", but if <definition> begins with <reply>, the response will be \"<definition>\""
        when "<action>"
          return "<action> => makes keyword respnse \"/me <definition>\""
        when "<who>"
          return "<who> => replaced with questioner in reply"
        when "<topic>"
          return "<topic> => respond by setting the topic to the rest of the definition"
        else
          return "Keyword module (Fact learning and regurgitation) topics: overview, set, plurals, override, also, random, get, tell, forget, keywords, <reply>, <action>, <who>, <topic>"
      end
    end

    # privmsg handler
    def privmsg(m)
      if(m.address?)
        if(!(m.message =~ /\\\?\s*$/) && m.message =~ /^(.*\S)\s*\?\s*$/)
          keyword m, $1 if(@bot.auth.allow?("keyword", m.source, m.replyto))
        elsif(m.message =~ /^(.*?)\s+(is|are)\s+(.*)$/)
          keyword_command(m.sourcenick, m.replyto, $1, $2, $3) if(@bot.auth.allow?("keycmd", m.source, m.replyto))
        elsif (m.message =~ /^tell\s+(\S+)\s+about\s+(.+)$/)
          keyword_tell(m, $1, $2) if(@bot.auth.allow?("keyword", m.source, m.replyto))
        elsif (m.message =~ /^forget\s+(.*)$/)
          key = $1
          if((@bot.auth.allow?("keycmd", m.source, m.replyto)) && @keywords.has_key?(key))
            @keywords.delete(key)
            @bot.okay m.replyto
          end
        elsif (m.message =~ /^keywords$/)
          if(@bot.auth.allow?("keyword", m.source, m.replyto))
            m.reply "There are currently #{@keywords.length} keywords, #{@statickeywords.length} static facts defined."
          end
        end
      else
        # in channel message, not to me
        if(m.message =~ /^'(.*)$/)
          keyword m, $1, false if(@bot.auth.allow?("keyword", m.source))
        elsif(@bot.config["KEYWORD_LISTEN"] == "true" && (m.message =~ /^(.*?)\s+(is|are)\s+(.*)$/))
          # TODO MUCH more selective on what's allowed here
          keyword_command(m.sourcenick, m.replyto, $1, $2, $3, true) if(@bot.auth.allow?("keycmd", m.source))
        end
      end
    end
  end
end
