# We're responsible for generating all the HTML files
# from the object tree defined in code_objects.rb. We
# generate:
#
# [files]   an html file for each input file given. These
#           input files appear as objects of class
#           TopLevel
#
# [classes] an html file for each class or module encountered.
#           These classes are not grouped by file: if a file
#           contains four classes, we'll generate an html
#           file for the file itself, and four html files 
#           for the individual classes. 
#
# [indices] we generate three indices for files, classes,
#           and methods. These are displayed in a browser
#           like window with three index panes across the
#           top and the selected description below
#
# Method descriptions appear in whatever entity (file, class,
# or module) that contains them.
#
# We generate files in a structure below a specified subdirectory,
# normally +doc+.
#
#  opdir
#     |
#     |___ files
#     |       |__  per file summaries
#     |
#     |___ classes
#             |__ per class/module descriptions
#
# HTML is generated using the Template class.
#

require 'ftools'

require 'rdoc/options'
require 'rdoc/template'
require 'markup/simple_markup'
require 'markup/simple_markup/to_html'

module Generators

  # Name of sub-direcories that hold file and class/module descriptions

  FILE_DIR = "files"
  CLASS_DIR = "classes"

  CSS_NAME = "rdoc-style.css"
  

  ##
  # Build a hash of all items that can be cross-referenced.
  # This is used when we output required and included names: 
  # if the names appear in this hash, we can generate
  # an html cross reference to the appropriate description.
  # We also use this when parsing comment blocks: any decorated 
  # words matching an entry in this list are hyperlinked.

  class AllReferences
    @@refs = {}

    def AllReferences.add(name, html_class)
      @@refs[name] = html_class
    end

    def AllReferences.[](name)
      @@refs[name]
    end
  end


  ##
  # Subclass of the SM::ToHtml class that supports looking
  # up words in the AllReferences list. Those that are
  # found (like AllReferences in this comment) will
  # be hyperlinked

  class HyperlinkHtml < SM::ToHtml
    # We need to record the html path of our caller so we can generate
    # correct relative paths for any hyperlinks that we find
    def initialize(from_path, context)
      super()
      @from_path = from_path

      @parent_name = context.parent_name
      @parent_name += "::" if @parent_name
    end

    # We're invoked when any text matches the CROSSREF pattern
    # (defined in MarkUp). If we fine the corresponding reference,
    # generate a hyperlink. If the name we're looking for contains
    # no punctuation, we look for it up the module/class chain. For
    # example, HyperlinkHtml is found, even without the Generators::
    # prefix, because we look for it in module Generators first.

    def handle_special_CROSSREF(special)
      name = special.text
      if name[0,1] == '#'
        lookup = name[1..-1]
        name = lookup unless Options.instance.show_hash
      else
        lookup = name
      end

      ref = nil

      if @parent_name and lookup !~ /\.:/
        ref = AllReferences[@parent_name  + lookup]
      end

      if !ref
        ref = AllReferences[lookup]
      end

      if ref and ref.document_self
        ref = HTMLGenerator.gen_url(@from_path, ref.path)
        "<a href=\"#{ref}\">#{name}</a>"
      else
        name
      end
    end

    # And we're invoked with a potential external hyperlink
    # mailto: just gets inserted. http: links are checked to
    # see if they reference an image. If so, that image gets inserted
    # using an <img> tag. Otherwise a conventional <a href> is used

    def handle_special_HYPERLINK(special)
      url = special.text
      url =~ /([A-Za-z]+):(.*)/
      protocol = ($1 || "http").downcase

      if protocol == "http" and url =~ /(gif|png|jpg|jpeg|bmp)$/
        "<img src=\"#{url}\">"
      else
        "<a href=\"#{url}\">#{url}</a>"
      end
    end
  end

  #####################################################################
  #
  # Handle common markup tasks for the various Html classes
  #

  module MarkUp

    # Convert a string in markup format into HTML. We keep a cached
    # SimpleMarkup object lying around after the first time we're
    # called per object.

    def markup(str)
      return '' unless str
      unless defined? @markup
        @markup = SM::SimpleMarkup.new
        # class names, variable names, file names, or instance variables
        @markup.add_special(/(
                               \b([A-Z]\w+(::\w+)*)
                             | \#\w+[!?=]?
                             | \b\w+([_\/\.]+\w+)+[!?=]?
                             )/x, 
                            :CROSSREF)

        # external hyperlinks
        @markup.add_special(/((http:|mailto:|ftp:|www\.)\S+\w)/, :HYPERLINK)

      end
      unless defined? @html_formatter
        @html_formatter = HyperlinkHtml.new(self.path, self)
      end

      # Convert leading comment markers to spaces
      @markup.convert(str.gsub(/^\s*(#+)/) { $1.tr('#',' ') }, 
                      @html_formatter)
    end
  end


  #####################################################################
  #
  # A Context is built by the parser to represent a container: contexts
  # hold classes, modules, methods, require lists and include lists.
  # ClassModule and TopLevel are the context objects we process here
  # 
  module ContextUser

    include MarkUp

    # convenience method to build a hyperlink
    def href(link, cls, name)
      %{<a href="#{link}" class="#{cls}">#{name}</a>} #"
    end

    # Create a list of HtmlMethod objects for each method
    # in the corresponding context object. If the @options.show_all
    # variable is set (corresponding to the <tt>--all</tt> option,
    # we include all methods, otherwise just the public ones.

    def collect_methods
      list = @context.method_list
      unless @options.show_all
        list = list.find_all {|m| m.visibility == :public }
      end
      @methods = list.collect {|m| HtmlMethod.new(m, self) }
    end

    # Build a summary list of all the methods in this context
    def build_method_summary_list
      collect_methods unless @methods
      meths = @methods.sort
      res = []
      meths.each do |meth|
	res << { "name" => meth.name, "aref" => "\##{meth.aref}" }
      end
      res
    end
    
    def build_requires_list(context)
      potentially_referenced_list(context.requires) {|fn| [fn + ".rb"] }
    end

    def build_include_list(context)
      potentially_referenced_list(context.includes)
    end

    # Build a list from an array of <i>Htmlxxx</i> items. Look up each
    # in the AllReferences hash: if we find a corresponding entry,
    # we generate a hyperlink to it, otherwise just output the name.
    # However, some names potentially need massaging. For example,
    # you may require a Ruby file without the .rb extension,
    # but the file names we know about may have it. To deal with
    # this, we pass in a block which performs the massaging,
    # returning an array of alternative names to match

    def potentially_referenced_list(array)
      res = []
      array.each do |i|
        ref = AllReferences[i.name] 
        if !ref && block_given?
          possibles = yield(i.name)
          while !ref and !possibles.empty?
            ref = AllReferences[possibles.shift]
          end
        end
        if ref and ref.document_self
          path = url(ref.path)
          res << { "name" => i.name, "aref" => path }
        else
          res << { "name" => i.name }
        end
      end
      res
    end

    # Build an array of arrays of method details. The outer array has up
    # to six entries, public, private, and protected for both class
    # methods, the other for instance methods. The inner arrays contain
    # a hash for each method

    def build_method_detail_list
      outer = []
      for singleton in [true, false]
        for vis in [ :public, :protected, :private ] 
          res = []
          @methods.each do |m|
            if m.document_self and m.visibility == vis and m.singleton? == singleton
              row = {}
              row["name"]        = m.name
              desc = m.description.strip
              row["description"] = desc unless desc.empty?
              row["params"]      = m.params
              row["aref"]        = m.aref
              row["visibility"]  = m.visibility.to_s
              code = m.src_url
              if code
                row["codeurl"] = code
                row["imgurl"]  = m.img_url
              end
              res << row
            end
          end
          if res.size > 0 
            outer << {
              "type"    => vis.to_s.capitalize,
              "category"    => singleton ? "Class" : "Instance",
              "methods" => res
            }
          end
        end
      end
      outer
    end

    # Build the structured list of classes and modules contained
    # in this context. 

    def build_class_list(level, from)
      res = ""
      prefix = "&nbsp;&nbsp;::" * level;

      from.modules.each do |mod|
        if mod.document_self
          res << prefix << "Module " 
          res << href(url(mod.viewer.path), "link", mod.full_name) << "<br>\n";
          res << build_class_list(level + 1, mod)
        end
      end

      from.classes.each do |cls|
        if cls.document_self
          res << prefix << "Class " 
          res << href(url(cls.viewer.path), "link", cls.full_name) << "<br>\n";
          res << build_class_list(level + 1, cls)
        end
      end

      res
    end
    
    def url(target)
      HTMLGenerator.gen_url(path, target)
    end

    def document_self
      @context.document_self
    end

    def diagram_reference(diagram)
      res = diagram.gsub(/((?:src|href)=")(.*?)"/) {
        $1 + url($2) + '"'
      }
      res
    end

  end

  #####################################################################
  #
  # Wrap a ClassModule context

  class HtmlClass 
    include ContextUser

    attr_reader :path

    def initialize(context, html_file, prefix, options)
      @context, @html_file, @options = context, html_file, options
      @is_module = context.is_module?
      @template = TemplatePage.new(RDoc::Page::BODY,
                                   RDoc::Page::CLASS_PAGE,
                                   RDoc::Page::METHOD_LIST)
      @values = {}
      context.viewer = self

      # map << ... names to something acceptable to DOS
      @path = @context.http_url(prefix)
      if @path['<<']
        @path.gsub!(/<<\s*(\w*)/) { "from-#$1" }
      end
      collect_methods

      AllReferences.add(name, self)
    end

    def name
      @context.full_name
    end

    def parent_name
      @context.parent.full_name
    end

    def index_name
      name
    end

    def write_on(f)
      class_attribute_values

      @values["style-url"] = url(CSS_NAME)

      d = markup(@context.comment)
      @values["description"] = d if d.size > 0

      ml = build_method_summary_list
      @values["methods"] = ml if ml.size > 0

      il = build_include_list(@context)
      @values["includes"] = il if il.size > 0

      al = build_attribute_list
      @values["attributes"] = al if al.size > 0
      
      cl = build_class_list(0, @context)
      @values["classlist"] = cl if cl.size > 0

      mdl = build_method_detail_list
      @values["method_list"] = mdl if mdl.size > 0

      @template.write_html_on(f, @values)
    end

    def build_attribute_list
      atts = @context.attributes.sort
      res = []
      atts.each do |att|
        res << {
          "name" => att.name, 
          "rw" => att.rw, 
          "description" => markup(att.comment)
        }
      end
      res
    end

    def class_attribute_values
      @values["classmod"]  = @is_module ? "Module" : "Class"
      @values["title"] = "#{@values['classmod']}: #{name}"

      c = @context
      c = c.parent while c and !c.diagram
      if c && c.diagram
        @values["diagram"] = diagram_reference(c.diagram)
      end

      @values["full_name"] = name

      parent_class = @context.superclass

      if parent_class
	@values["parent"] = parent_class

	if parent_name
	  lookup = parent_name + "::" + parent_class
	else
	  lookup = parent_class
	end

	parent_url = AllReferences[lookup] || AllReferences[parent_class]

	if parent_url and parent_url.document_self
	  @values["par_url"] = url(parent_url.path)
	end
      end

      files = []
      @context.in_files.each do |f|
        res = {}
        res["full_path"] = f.file_absolute_name
        res["full_path_url"] = url(f.viewer.path) if f.document_self
        files << res
      end
      @values['files'] = files

    end

    def <=>(other)
      self.name <=> other.name
    end

  end

  #####################################################################
  #
  # Handles the mapping of a file's information to HTML. In reality,
  # a file corresponds to a +TopLevel+ object, containing modules,
  # classes, and top-level methods. In theory it _could_ contain
  # attributes and aliases, but we ignore these for now.

  class HtmlFile 
    include ContextUser

    attr_reader :path
    attr_reader :name

    def initialize(context, options, file_dir)
      @context, @options = context, options
      @template = TemplatePage.new(RDoc::Page::BODY,
                                   RDoc::Page::FILE_PAGE,
                                   RDoc::Page::METHOD_LIST)
      @values = {}
      @path = File.join(file_dir, @context.file_relative_name.tr('.', '_')) +
        ".html"

      @name = @context.file_relative_name

      collect_methods
      AllReferences.add(name, self)
      context.viewer = self
    end

    def index_name
      name
    end

    def parent_name
      nil
    end

    def write_on(f)
      file_attribute_values

      @values["style-url"] = url(CSS_NAME)

      if @context.comment
        d = markup(@context.comment)
        @values["description"] = d if d.size > 0
      end

      ml = build_method_summary_list
      @values["methods"] = ml if ml.size > 0

      il = build_include_list(@context)
      @values["includes"] = il if il.size > 0

      rl = build_requires_list(@context)
      @values["requires"] = rl if rl.size > 0

      cl = build_class_list(0, @context)
      @values["classlist"] = cl if cl.size > 0

      mdl = build_method_detail_list
      @values["method_list"] = mdl if mdl.size > 0

      @template.write_html_on(f, @values)
    end

    def file_attribute_values
      full_path = @context.file_absolute_name
      short_name = File.basename(full_path)
      
      @values["title"] = "File: #{short_name}"
      if @context.diagram
        @values["diagram"] = diagram_reference(@context.diagram)
      end
      @values["short_name"]   = short_name
      @values["full_path"]    = full_path
      @values["dtm_created"]  = @context.file_stat.ctime.to_s
      @values["dtm_modified"] = @context.file_stat.mtime.to_s
    end

    def <=>(other)
      self.name <=> other.name
    end
  end

  #####################################################################

  class HtmlMethod
    include MarkUp

    attr_reader :context
    attr_reader :src_url
    attr_reader :img_url

    @@seq = "M000000"

    @@all_methods = []

    def initialize(context, html_class)
      @context = context
      @html_class = html_class
      @@seq = @@seq.succ
      @seq  = @@seq
      @@all_methods << self

      @src_url = create_source_code
      @img_url = HTMLGenerator.gen_url(path, 'source.png')

      AllReferences.add(name, self)
    end
    
    def name
      @context.name
    end

    def index_name
      "#{@context.name} (#{@html_class.name})"
    end

    def parent_name
      if @context.parent.parent
        @context.parent.parent.full_name
      else
        nil
      end
    end

    def aref
      @seq
    end

    def path
      @html_class.path + "#" + aref
    end

    def description
      markup(@context.comment)
    end

    def visibility
      @context.visibility
    end

    def singleton?
      @context.singleton?
    end

    def params
      p = @context.params.gsub(/\s*\#.*/, '')
      p = p.tr("\n", " ").squeeze(" ")
      p = "(" + p + ")" unless p[0] == ?(

      if (block = @context.block_params)
        block.gsub!(/\s*\#.*/, '')
        block = block.tr("\n", " ").squeeze(" ")
        if block[0] == ?(
          block.sub!(/^\(/, '').sub!(/\)/, '')
        end
        p << " {|#{block}| ...}"
      end

      CGI.escapeHTML(p)
    end
    
    def create_source_code
      ts = @context.token_stream
      return nil unless ts
      if @html_class .nil?
        raise "Nil htmlclass"
      end
      if @html_class.path.nil?
        exit!
      end
      meth_path = @html_class.path.sub(/\.html$/, '.src')
      File.makedirs(meth_path)
      file_path = File.join(meth_path, @seq) + ".html"

      template = TemplatePage.new(RDoc::Page::SRC_PAGE)
      File.open(file_path, "w") do |f|
        values = { 'title' => index_name, 'code' => markup_code(ts) }
        template.write_html_on(f, values)
      end
      HTMLGenerator.gen_url(path, file_path)
    end

    def HtmlMethod.all_methods
      @@all_methods
    end

    def <=>(other)
      @context <=> other.context
    end

    ##
    # Given a sequence of source tokens, mark up the source code
    # to make it look purty.
    
    def markup_code(tokens)
      src = ""
      tokens.each do |t|
        #    p t.class
        style = nil
        case t
        when RubyToken::TkKW
          style = "kw"
        when RubyToken::TkCOMMENT
          style = "cmt"
        when RubyToken::TkSTRING
          style = "str"
        when RubyToken::TkREGEXP
          style = "re"
        end
        if style
          src << "<span class=\"#{style}\">#{t.text}</span>"
        else
          src << t.text if t
        end
      end
      src
    end

    def document_self
      @context.document_self
    end

  end

  #####################################################################

  class HTMLGenerator

    include MarkUp

    # Set up a new HTML generator. Basically all we do here is load
    # up the correct output temlate

    def initialize(options)
      @options    = options
      load_html_template
    end

    ##
    # convert a target url to one that is relative to a given
    # path
    
    def HTMLGenerator.gen_url(path, target)
      from          = File.dirname(path)
      to, to_file   = File.split(target)
      
      from = from.split("/")
      to   = to.split("/")
      
      while from.size > 0 and to.size > 0 and from[0] == to[0]
        from.shift
        to.shift
      end
      
      from.fill("..")
      from.concat(to)
      from << to_file
      File.join(*from)
    end


    ##
    # Build the initial indices and output objects
    # based on an array of TopLevel objects containing
    # the extracted information. 

    def generate(info)
      @info       = info
      @files      = []
      @classes    = []
      @hyperlinks = {}

#        write_source_icon
      write_style_sheet
      gen_sub_directories()
      build_indices
      generate_html
    end

    private

    ##
    # Load up the HTML template specified in the options
    #
    def load_html_template
      template = "rdoc/generators/html_template/" + @options.template
      require template
      extend RDoc::Page
    rescue LoadError
      $stderr.puts "Could not find HTML template '#{template}'"
      exit 99
    end

    ##
    # Write out the style sheet used by the main frames
    #
    
    def write_style_sheet
      template = TemplatePage.new(RDoc::Page::STYLE)
      File.open(CSS_NAME, "w") do |f|
        values = { "fonts" => RDoc::Page::FONTS }
        template.write_html_on(f, values)
      end
    end

    ##
    # Generate:
    #
    # * a list of HtmlFile objects for each TopLevel object.
    # * a list of HtmlClass objects for each first level
    #   class or module in the TopLevel objects
    # * a complete list of all hyperlinkable terms (file,
    #   class, module, and method names)

    def build_indices

      @info.each do |toplevel|
        @files << HtmlFile.new(toplevel, @options, FILE_DIR)
      end

      RDoc::TopLevel.all_classes_and_modules.each do |cls|
        build_class_list(cls, @files[0], CLASS_DIR)
      end
    end

    def build_class_list(from, html_file, class_dir)
      @classes << HtmlClass.new(from, html_file, class_dir, @options)
      from.each_classmodule do |mod|
        build_class_list(mod, html_file, class_dir)
      end
    end

    ##
    # Generate all the HTML
    #
    def generate_html
      # the individual descriptions for files and classes
      gen_into(@files)
      gen_into(@classes)
      # and the index files
      gen_file_index
      gen_class_index
      gen_method_index
      gen_main_index
      
      # this method is defined in the template file
      write_extra_pages if defined? write_extra_pages
    end

    def gen_into(list)
      list.each do |item|
        op_file = item.path
        File.makedirs(File.dirname(op_file))
        File.open(op_file, "w") { |file| item.write_on(file) }
      end

    end

    def gen_file_index
      gen_an_index(@files, 'Files', 
                   RDoc::Page::FILE_INDEX, 
                   "fr_file_index.html")
    end

    def gen_class_index
      gen_an_index(@classes, 'Classes',
                   RDoc::Page::CLASS_INDEX,
                   "fr_class_index.html")
    end

    def gen_method_index
      gen_an_index(HtmlMethod.all_methods, 'Methods', 
                   RDoc::Page::METHOD_INDEX,
                   "fr_method_index.html")
    end

    
    def gen_an_index(collection, title, template, filename)
      template = TemplatePage.new(RDoc::Page::FR_INDEX_BODY, template)
      res = []
      collection.sort.each do |f|
        if f.document_self
          res << { "href" => f.path, "name" => f.index_name }
        end
      end

      values = {
        "entries" => res,
        'list_title' => title,
        'index_url'  => main_url
      }

      File.open(filename, "w") do |f|
        template.write_html_on(f, values)
      end
    end

    # The main index page is mostly a template frameset, but includes
    # the initial page. If the <tt>--main</tt> option was given,
    # we use this as our main page, otherwise we use the
    # first file specified on the command line.

    def gen_main_index
      template = TemplatePage.new(RDoc::Page::INDEX)
      File.open("index.html", "w") do |f|
        values = { "initial_page" => main_url, 'title' => @options.title }
        template.write_html_on(f, values)
      end
    end

    # return the url of the main page
    def main_url
      main_page = @options.main_page
      ref = nil
      if main_page
        ref = AllReferences[main_page]
        if ref
          ref = ref.path
        else
          $stderr.puts "Could not find main page #{main_page}"
        end
      end

      ref = @files[0].path unless ref

      ref
    end

    ##
    # See the comments at the top for a description of the
    # directory structure

    def gen_sub_directories
      File.makedirs(FILE_DIR, CLASS_DIR)
    rescue 
      $stderr.puts $!.message
      exit 1
    end

  end

end
