/*=============================================================================
    Parser exceptions

    Spirit V1.2
    Copyright (c) 2001, Joel de Guzman

    This software is provided 'as-is', without any express or implied
    warranty. In no event will the copyright holder be held liable for
    any damages arising from the use of this software.

    Permission is granted to anyone to use this software for any purpose,
    including commercial applications, and to alter it and redistribute
    it freely, subject to the following restrictions:

    1.  The origin of this software must not be misrepresented; you must
        not claim that you wrote the original software. If you use this
        software in a product, an acknowledgment in the product documentation
        would be appreciated but is not required.

    2.  Altered source versions must be plainly marked as such, and must
        not be misrepresented as being the original software.

    3.  This notice may not be removed or altered from any source
        distribution.

    Acknowledgements:

        Special thanks to Dan Nuffer, John (EBo) David, Chris Uzdavinis,
        and Doug Gregor. These people are most instrumental in steering
        Spirit in the right direction.

        Special thanks also to people who have contributed to the code base
        and sample code, ported Spirit to various platforms and compilers,
        gave suggestions, reported and provided bug fixes. Alexander
        Hirner, Andy Elvey, Bogdan Kushnir, Brett Calcott, Bruce Florman,
        Changzhe Han, Colin McPhail, Hakki Dogusan, Jan Bares, Joseph
        Smith, Martijn W. van der Lee, Raghavendra Satish, Remi Delcos, Tom
        Spilman, Vladimir Prus, W. Scott Dillman, David A. Greene, Bob
        Bailey, Hartmut Kaiser.

        Finally special thanks also to people who gave feedback and
        valuable comments, particularly members of Spirit's Source Forge
        mailing list and boost.org.

    URL: http://spirit.sourceforge.net/

=============================================================================*/
#ifndef SPIRIT_EXCEPTIONS_HPP
#define SPIRIT_EXCEPTIONS_HPP

///////////////////////////////////////////////////////////////////////////////

#include "boost/spirit/parser.hpp"
#include "boost/spirit/composite.hpp"

///////////////////////////////////////////////////////////////////////////////
namespace spirit {

///////////////////////////////////////////////////////////////////////////////
//
//  parser_error_base class
//
//      This is the base class of parser_error (see below). This may
//      be used to catch any type of parser error.
//
///////////////////////////////////////////////////////////////////////////////
class parser_error_base {

protected:

    parser_error_base();
    ~parser_error_base();
    parser_error_base(parser_error_base const&);
    parser_error_base& operator=(parser_error_base const&);
};

///////////////////////////////////////////////////////////////////////////////
//
//  parser_error class
//
//      Generic parser exception class. This is the base class for all
//      parser exceptions. The exception holds the iterator position
//      where the error was encountered. This can be queried through
//      the exception's where() member function. In addition to the
//      iterator, parser_error also holds information regarding the
//      error (error descriptor). This can be queried through the
//      exception's what() member function.
//
//      The throw_ function creates and throws a parser_error given
//      an iterator and an error descriptor.
//
///////////////////////////////////////////////////////////////////////////////
template <typename ErrorDescrT, typename IteratorT = char const*>
class parser_error : public parser_error_base {

public:

    parser_error(IteratorT where, ErrorDescrT what);

    IteratorT
    where() const;

    ErrorDescrT
    what() const;

private:

    IteratorT   iter;
    ErrorDescrT info;
};

//////////////////////////////////
template <typename ErrorDescrT, typename IteratorT>
void
throw_(IteratorT where, ErrorDescrT what);

///////////////////////////////////////////////////////////////////////////////
//
//  assertive_parser class
//
//      An assertive_parser class is a parser that throws an exception
//      in response to a parsing failure. The assertive_parser throws a
//      parser_error exception rather than returning an unsuccessful
//      match to signal that the parser failed to match the input.
//
///////////////////////////////////////////////////////////////////////////////
template <typename ErrorDescrT, typename ParserT>
class assertive_parser
:   public unary<ParserT>,
    public parser<assertive_parser<ErrorDescrT, ParserT> > {

public:

    assertive_parser(ParserT const& parser, ErrorDescrT what);

    template <typename IteratorT>
    match
    parse(IteratorT& first, IteratorT const& last) const;

private:

    ErrorDescrT info;
};

///////////////////////////////////////////////////////////////////////////////
//
//  assertion class
//
//      assertive_parsers are never instantiated directly. The assertion
//      class is used to indirectly create an assertive_parser object.
//      Before declaring the grammar, we declare some assertion objects.
//      Examples:
//
//          enum Errors {
//
//              program_expected, begin_expected, end_expected
//          };
//
//          assertion<Errors>   expect_program(program_expected);
//          assertion<Errors>   expect_begin(begin_expected);
//          assertion<Errors>   expect_end(end_expected);
//
//      Now, we can use these assertions as wrappers around parsers:
//
//          expect_end(str_p("end"))
//
//      Take note that although the example uses enums to hold the
//      information regarding the error (error desccriptor), we are free
//      to use other types such as integers and strings. Enums are
//      convenient for error handlers to easily catch since C++ treats
//      enums as unique types.
//
///////////////////////////////////////////////////////////////////////////////
template <typename ErrorDescrT>
class assertion {

public:

    assertion(ErrorDescrT what);

    template <typename ParserT>
    assertive_parser<ErrorDescrT, ParserT>
    operator()(ParserT const& parser) const
    {
        //  Borland 5.5 reports an internal compiler
        //  error if this is not defined here.
        return assertive_parser<ErrorDescrT, ParserT>(parser, info);
    }

private:

    ErrorDescrT info;
};

///////////////////////////////////////////////////////////////////////////////
//
//  fallback_parser class
//
//      Handles exceptions of type parser_error<ErrorDescrT, IteratorT>
//      thrown somewhere inside its embedded ParserT object. The class
//      sets up a try block before delegating parsing to its subject.
//      When an exception is caught, the catch block then calls the
//      HandlerT object. HandlerT may be a function or a functor (with
//      an operator() member function) compatible with the interface:
//
//          match   f(
//                      IteratorT&  first,
//                      IteratorT   last,
//                      IteratorT   where,
//                      ErrorDescrT what);
//
//      Where first points to the current input, last points to one
//      after the end of the input (same as STL algorithms), where
//      points to the position where the error was found and what
//      describes the nature of the error (error descriptor).
//
//      HandlerT may recover from the error and return a successful
//      match while advancing the iterator in the process. Otherwise,
//      it may return a no-match or re-throw another exception.
//
///////////////////////////////////////////////////////////////////////////////
template <typename ErrorDescrT, typename ParserT, typename HandlerT>
class fallback_parser
:   public unary<ParserT>,
    public parser<fallback_parser<ErrorDescrT, ParserT, HandlerT> > {

public:

    fallback_parser(ParserT const& parser, HandlerT const& handler);

    template <typename IteratorT>
    match
    parse(IteratorT& first, IteratorT const& last) const;

private:

    HandlerT    handler;
};

///////////////////////////////////////////////////////////////////////////////
//
//  guard class
//
//      fallback_parser objects are not instantiated directly. The guard
//      class is used to indirectly create a fallback_parser object.
//      guards are typically predeclared just like assertions (see the
//      assertion class above; the example extends the previous example
//      introduced in the assertion class above):
//
//          guard<Errors>   my_guard;
//
//      Errors, in this example is the error descriptor type we want to
//      detect; This is essentially the ErrorDescrT template parameter
//      of the fallback_parser class.
//
//      my_guard may now be used in a grammar declaration as:
//
//          my_guard(p, h)
//
//      where p is a parser, h is a function or functor compatible with
//      fallback_parser's HandlerT (see above).
//
///////////////////////////////////////////////////////////////////////////////
template <typename ErrorDescrT>
struct guard {

    template <typename ParserT, typename HandlerT>
    fallback_parser<ErrorDescrT, ParserT, HandlerT>
    operator()(ParserT const& parser, HandlerT const& handler)
    {
        //  Borland 5.5 reports an internal compiler
        //  error if this is not defined here.
        return fallback_parser<ErrorDescrT, ParserT, HandlerT>(parser, handler);
    }
};

///////////////////////////////////////////////////////////////////////////////
}   //  namespace Spirit

#endif

