/*=============================================================================
    Parser primitives

    Spirit V1.2
    Copyright (c) 2001, Joel de Guzman

    This software is provided 'as-is', without any express or implied
    warranty. In no event will the copyright holder be held liable for
    any damages arising from the use of this software.

    Permission is granted to anyone to use this software for any purpose,
    including commercial applications, and to alter it and redistribute
    it freely, subject to the following restrictions:

    1.  The origin of this software must not be misrepresented; you must
        not claim that you wrote the original software. If you use this
        software in a product, an acknowledgment in the product documentation
        would be appreciated but is not required.

    2.  Altered source versions must be plainly marked as such, and must
        not be misrepresented as being the original software.

    3.  This notice may not be removed or altered from any source
        distribution.

    Acknowledgements:

        Special thanks to Dan Nuffer, John (EBo) David, Chris Uzdavinis,
        and Doug Gregor. These people are most instrumental in steering
        Spirit in the right direction.

        Special thanks also to people who have contributed to the code base
        and sample code, ported Spirit to various platforms and compilers,
        gave suggestions, reported and provided bug fixes. Alexander
        Hirner, Andy Elvey, Bogdan Kushnir, Brett Calcott, Bruce Florman,
        Changzhe Han, Colin McPhail, Hakki Dogusan, Jan Bares, Joseph
        Smith, Martijn W. van der Lee, Raghavendra Satish, Remi Delcos, Tom
        Spilman, Vladimir Prus, W. Scott Dillman, David A. Greene, Bob
        Bailey, Hartmut Kaiser.

        Finally special thanks also to people who gave feedback and
        valuable comments, particularly members of Spirit's Source Forge
        mailing list and boost.org.

    URL: http://spirit.sourceforge.net/

=============================================================================*/
#ifndef SPIRIT_PRIMITIVES_IPP
#define SPIRIT_PRIMITIVES_IPP

///////////////////////////////////////////////////////////////////////////////

#include <cassert>
#include <cctype>
#if __GNUC__ > 2 || defined(__MWERKS__)
    #include <cwctype>
#endif
#include <string>

#include "boost/spirit/primitives.hpp"
#include "boost/spirit/iterators.hpp"

///////////////////////////////////////////////////////////////////////////////
namespace spirit {

///////////////////////////////////////////////////////////////////////////////
//
//  char_action class implementation
//
///////////////////////////////////////////////////////////////////////////////
template <typename ParserT, typename ActionT>
inline char_action<ParserT, ActionT>::char_action(
    ParserT const&  subject,
    ActionT const&   actor_)
:
    unary<ParserT>(subject),
    actor(actor_) {}

//////////////////////////////////
template <typename ParserT, typename ActionT>
template <typename IteratorT>
inline match
char_action<ParserT, ActionT>::
    parse(IteratorT& first, IteratorT const& last) const
{
    if (first != last)
    {
        typedef impl::strip_scanner<IteratorT> strip_scanner;
        typename strip_scanner::iterator_type
            begin = strip_scanner::get(first);

        if (match hit = this->subject().parse(first, last))
        {
            actor(*begin);
            return hit;
        }
    }
    return match();
}

///////////////////////////////////////////////////////////////////////////////
//
//  char_parser class implementation
//
///////////////////////////////////////////////////////////////////////////////
template <typename DerivedT>
template <typename ActionT>
inline char_action<DerivedT, ActionT>
char_parser<DerivedT>::operator[](ActionT const& actor) const
{
    return char_action<DerivedT, ActionT>(this->derived(), actor);
}

//////////////////////////////////
template <typename DerivedT>
template <typename IteratorT>
inline match
char_parser<DerivedT>::parse(IteratorT& first, IteratorT const& last) const
{
    if (first != last)
        if (this->derived().test(*first))
        {
            ++first;
            return match(1);
        }
    return match();
}

///////////////////////////////////////////////////////////////////////////////
//
//  chlit class implementation
//
///////////////////////////////////////////////////////////////////////////////
template <typename CharT>
inline chlit<CharT>::chlit(CharT ch_)
:   ch(ch_) {}

//////////////////////////////////
template <typename CharT>
template <typename T>
inline bool
chlit<CharT>::test(T ch_) const
{
    return T(ch) == ch_;
}

///////////////////////////////////////////////////////////////////////////////
//
//  range class implementation
//
///////////////////////////////////////////////////////////////////////////////
template <typename CharAT, typename CharBT>
inline range<CharAT, CharBT>::range(CharAT first_, CharBT last_)
:   first(first_), last(last_)
{
    assert(first <= last);
}

//////////////////////////////////
template <typename CharAT, typename CharBT>
template <typename T>
inline bool
range<CharAT, CharBT>::test(T ch) const
{
    return (CharAT(ch) >= first) && (CharBT(ch) <= last);
}

///////////////////////////////////////////////////////////////////////////////
//
//  cstring class implementation
//
///////////////////////////////////////////////////////////////////////////////
template <typename CharT>
inline cstring<CharT>::cstring(const_iterator str, unsigned len)
:   first(str), last(str + len) {}

//////////////////////////////////
template <typename CharT>
inline cstring<CharT>::cstring(const_iterator str)
:   first(str)
{
    while (*str)
        str++;
    last = str;
}

//////////////////////////////////
template <typename CharT>
inline typename cstring<CharT>::const_iterator
cstring<CharT>::begin() const
{
    return first;
}

//////////////////////////////////
template <typename CharT>
inline typename cstring<CharT>::const_iterator
cstring<CharT>::end() const
{
    return last;
}

//////////////////////////////////
template <typename CharT>
inline std::size_t
cstring<CharT>::length() const
{
    return last-first;
}

///////////////////////////////////////////////////////////////////////////////
//
//  user_string class implementation
//
///////////////////////////////////////////////////////////////////////////////
template <typename StringRefT>
inline user_string<StringRefT>::user_string(StringRefT const& ref_)
:   ref(ref_) {}

//////////////////////////////////
template <typename StringRefT>
inline typename user_string<StringRefT>::const_iterator
user_string<StringRefT>::begin() const
{
    return static_cast<element_type&>(ref).begin();
}

//////////////////////////////////
template <typename StringRefT>
inline typename user_string<StringRefT>::const_iterator
user_string<StringRefT>::end() const
{
    return static_cast<element_type&>(ref).end();
}

//////////////////////////////////
template <typename StringRefT>
inline std::size_t
user_string<StringRefT>::length() const
{
    return static_cast<element_type&>(ref).length();
}

///////////////////////////////////////////////////////////////////////////////
//
//  strlit class implementation
//
///////////////////////////////////////////////////////////////////////////////
template <typename StringT>
inline strlit<StringT>::strlit(StringT str_)
:   str(str_) {}

//////////////////////////////////
template <typename StringT>
inline strlit<StringT>::strlit(raw_string_type str_)
:   str(str_) {}

//////////////////////////////////
template <typename StringT>
template <typename IteratorT>
inline match
strlit<StringT>::parse(IteratorT& first, IteratorT const& last) const
{
    typedef typename impl::strip_scanner<IteratorT>
        ::iterator_type plain_iter;
    plain_iter  i1 = impl::strip_scanner<IteratorT>::get(first);
    plain_iter  i2 = impl::strip_scanner<IteratorT>::get(last);

    typename string_type::const_iterator strEnd = str.end();

    for (typename string_type::const_iterator strFirst = str.begin();
        strFirst != strEnd; ++strFirst, ++i1)
        if ((i1 == i2) || (*strFirst != *i1))
            return match();
    first = i1;
    return match(str.length());
}

///////////////////////////////////////////////////////////////////////////////
//
//  Convenience functions
//
///////////////////////////////////////////////////////////////////////////////
//
//  On some systems the is* functions are defined as macros

#if defined(_MSC_VER) || _STLPORT_VERSION >= 0x450 
// is* functions already setup
#else
#ifndef isalnum
inline bool isalnum(int c)     { return std::isalnum(c); }
#endif
#ifndef isalpha
inline bool isalpha(int c)     { return std::isalpha(c); }
#endif
#ifndef iscntrl
inline bool iscntrl(int c)     { return std::iscntrl(c); }
#endif
#ifndef isdigit
inline bool isdigit(int c)     { return std::isdigit(c); }
#endif
#ifndef isgraph
inline bool isgraph(int c)     { return std::isgraph(c); }
#endif
#ifndef islower
inline bool islower(int c)     { return std::islower(c); }
#endif
#ifndef isprint
inline bool isprint(int c)     { return std::isprint(c); }
#endif
#ifndef ispunct
inline bool ispunct(int c)     { return std::ispunct(c); }
#endif
#ifndef isspace
inline bool isspace(int c)     { return std::isspace(c); }
#endif
#ifndef isupper
inline bool isupper(int c)     { return std::isupper(c); }
#endif
#ifndef isxdigit
inline bool isxdigit(int c)    { return std::isxdigit(c); }
#endif
#ifndef isalnum
inline bool isalnum(wchar_t c)  { return std::iswalnum(c); }
#endif
#ifndef isalpha
inline bool isalpha(wchar_t c)  { return std::iswalpha(c); }
#endif
#ifndef iscntrl
inline bool iscntrl(wchar_t c)  { return std::iswcntrl(c); }
#endif
#ifndef isdigit
inline bool isdigit(wchar_t c)  { return std::iswdigit(c); }
#endif
#ifndef isgraph
inline bool isgraph(wchar_t c)  { return std::iswgraph(c); }
#endif
#ifndef islower
inline bool islower(wchar_t c)  { return std::iswlower(c); }
#endif
#ifndef isprint
inline bool isprint(wchar_t c)  { return std::iswprint(c); }
#endif
#ifndef ispunct
inline bool ispunct(wchar_t c)  { return std::iswpunct(c); }
#endif
#ifndef isspace
inline bool isspace(wchar_t c)  { return std::iswspace(c); }
#endif
#ifndef isupper
inline bool isupper(wchar_t c)  { return std::iswupper(c); }
#endif
#ifndef isxdigit
inline bool isxdigit(wchar_t c) { return std::iswxdigit(c); }
#endif
#endif // !defined(_MSC_VER)

///////////////////////////////////////////////////////////////////////////////
//
//  Predefined parser primitives implementation
//
///////////////////////////////////////////////////////////////////////////////
template <typename CharT>
inline bool
anychar_::test(CharT) const
{
    return true;
}

//////////////////////////////////
template <typename IteratorT>
inline match
nothing_::parse(IteratorT& /*first*/, IteratorT const& /*last*/) const
{
    return match();
}

//////////////////////////////////
template <typename IteratorT>
inline match
epsilon_::parse(IteratorT&, IteratorT const&) const
{
    return match(0);
}

//////////////////////////////////
template <typename CharT>
inline bool
alnum_::test(CharT ch) const
{
    return isalnum(ch);
}

//////////////////////////////////
template <typename CharT>
inline bool
alpha_::test(CharT ch) const
{
    return isalpha(ch);
}

//////////////////////////////////
template <typename CharT>
inline bool
cntrl_::test(CharT ch) const
{
    return iscntrl(ch);
}

//////////////////////////////////
template <typename CharT>
inline bool
digit_::test(CharT ch) const
{
    return isdigit(ch);
}

//////////////////////////////////
template <typename CharT>
inline bool
graph_::test(CharT ch) const
{
    return isgraph(ch);
}

//////////////////////////////////
template <typename CharT>
inline bool
lower_::test(CharT ch) const
{
    return islower(ch);
}

//////////////////////////////////
template <typename CharT>
inline bool
print_::test(CharT ch) const
{
    return isprint(ch);
}

//////////////////////////////////
template <typename CharT>
inline bool
punct_::test(CharT ch) const
{
    return ispunct(ch);
}

//////////////////////////////////
template <typename CharT>
inline bool
space_::test(CharT ch) const
{
    return isspace(ch);
}

//////////////////////////////////
template <typename CharT>
inline bool
upper_::test(CharT ch) const
{
    return isupper(ch);
}

//////////////////////////////////
template <typename CharT>
inline bool
xdigit_::test(CharT ch) const
{
    return isxdigit(ch);
}

///////////////////////////////////////////////////////////////////////////////
//
//  Generator functions implementation
//
///////////////////////////////////////////////////////////////////////////////
template <typename ParamT>
inline chlit<ParamT>
ch_p(ParamT param)
{
    return chlit<ParamT>(param);
}

template <typename ParamAT, typename ParamBT>
inline range<ParamAT, ParamBT>
range_p(ParamAT first, ParamBT last)
{
    return range<ParamAT, ParamBT>(first, last);
}

template <typename CharT>
inline strlit<cstring<CharT> >
str_p(CharT const* str)
{
    return strlit<cstring<CharT> >(str);
}

template <typename StringT>
inline strlit<StringT&>
str_p(reference_wrapper<StringT> const& ref)
{
    return strlit<StringT&>(ref);
}

///////////////////////////////////////////////////////////////////////////////
}   //  namespace Spirit

#endif
