/*=============================================================================
    Numeric parsers

    Spirit V1.2
    Copyright (c) 2001, Joel de Guzman

    This software is provided 'as-is', without any express or implied
    warranty. In no event will the copyright holder be held liable for
    any damages arising from the use of this software.

    Permission is granted to anyone to use this software for any purpose,
    including commercial applications, and to alter it and redistribute
    it freely, subject to the following restrictions:

    1.  The origin of this software must not be misrepresented; you must
        not claim that you wrote the original software. If you use this
        software in a product, an acknowledgment in the product documentation
        would be appreciated but is not required.

    2.  Altered source versions must be plainly marked as such, and must
        not be misrepresented as being the original software.

    3.  This notice may not be removed or altered from any source
        distribution.

    Acknowledgements:

        Special thanks to Dan Nuffer, John (EBo) David, Chris Uzdavinis,
        and Doug Gregor. These people are most instrumental in steering
        Spirit in the right direction.

        Special thanks also to people who have contributed to the code base
        and sample code, ported Spirit to various platforms and compilers,
        gave suggestions, reported and provided bug fixes. Alexander
        Hirner, Andy Elvey, Bogdan Kushnir, Brett Calcott, Bruce Florman,
        Changzhe Han, Colin McPhail, Hakki Dogusan, Jan Bares, Joseph
        Smith, Martijn W. van der Lee, Raghavendra Satish, Remi Delcos, Tom
        Spilman, Vladimir Prus, W. Scott Dillman, David A. Greene, Bob
        Bailey, Hartmut Kaiser.

        Finally special thanks also to people who gave feedback and
        valuable comments, particularly members of Spirit's Source Forge
        mailing list and boost.org.

    URL: http://spirit.sourceforge.net/

=============================================================================*/
#ifndef SPIRIT_NUMERICS_HPP
#define SPIRIT_NUMERICS_HPP

///////////////////////////////////////////////////////////////////////////////
namespace spirit {

///////////////////////////////////////////////////////////////////////////////
//
//  numeric_action class
//
//      Links a numeric parser with a user defined semantic action.
//      The semantic action may be a function or a functor. A function
//      should be compatible with the interface:
//
//          void f(NumT num);
//
//      A functor should have a member operator() with a compatible
//      signature as above. The matching number is passed into the
//      function/functor. This is the default class that numeric
//      parsers use when dealing with the construct:
//
//          p[f]
//
//      where p is a parser and f is a function or functor. The header
//      file <actions.hpp> has some useful predefined generic functors
//      that may be used to directly link variables with numeric parsers
//      (see reference_wrapper and context_wrapper classes.
//
///////////////////////////////////////////////////////////////////////////////
template <typename ParserT, typename ActionT>
class numeric_action
:   public unary<ParserT>,
    public parser<numeric_action<ParserT, ActionT> > {

public:

    numeric_action(ActionT const& actor);

    template <typename IteratorT>
    match
    parse(IteratorT& first, IteratorT const& last) const;

private:

    ActionT   actor;
};

///////////////////////////////////////////////////////////////////////////////
template <typename T = unsigned>    struct uint_parser;
template <typename T = int>         struct int_parser;
template <typename T = float>       struct ureal_parser;
template <typename T = float>       struct real_parser;

///////////////////////////////////////////////////////////////////////////////
template <typename T>
struct numeric_action_traits {

    typedef typename T::arg_type arg_type;
    typedef numeric_action<uint_parser<arg_type>, T>    uint_action;
    typedef numeric_action<int_parser<arg_type>, T>     int_action;
    typedef numeric_action<ureal_parser<arg_type>, T>   ureal_action;
    typedef numeric_action<real_parser<arg_type>, T>    real_action;
};

///////////////////////////////////////////////////////////////////////////////
template <typename T>
struct numeric_action_traits<void (*)(T)> {

    typedef void (*func_type)(T);
    typedef numeric_action<uint_parser<T>, func_type>     uint_action;
    typedef numeric_action<int_parser<T>, func_type>      int_action;
    typedef numeric_action<ureal_parser<T>, func_type>    ureal_action;
    typedef numeric_action<real_parser<T>, func_type>     real_action;
};

///////////////////////////////////////////////////////////////////////////////
template <typename T>
struct numeric_action_traits<reference_wrapper<T> > {

    typedef uint_parser<T&>     uint_action;
    typedef int_parser<T&>      int_action;
    typedef ureal_parser<T&>    ureal_action;
    typedef real_parser<T&>     real_action;
};

///////////////////////////////////////////////////////////////////////////////
//
//  uint_parser class
//
//      Parses unsigned integers. A uint_parser is never directly
//      instantiated. An instance of the uint_parser_gen, uint_p,
//      generates a uint_parser through its [] operator:
//
//          uint_p[f]
//
//      where f is a semantic action which may be a function or a
//      functor. A function should be compatible with the interface:
//
//          void f(NumT num);
//
//      A functor should have a member operator() with a compatible
//      signature as above. The matching number is passed into the
//      function/functor (see numeric_action class above).
//
///////////////////////////////////////////////////////////////////////////////
template <typename T>
struct uint_parser : public parser<uint_parser<T> > {

    uint_parser(T n);

    template <typename IteratorT>
    match
    parse(IteratorT& first, IteratorT const& last) const;

    mutable T n;
};

///////////////////////////////////////////////////////////////////////////////
struct uint_parser_gen : public uint_parser<unsigned> {

    uint_parser_gen()
    :   uint_parser<unsigned>(0) {}

    template <typename T>
    typename numeric_action_traits<T>::uint_action
    operator [] (T const& actor) const
    {
        //  Borland 5.5 reports an internal compiler
        //  error if this is not defined here.
        return numeric_action_traits<T>::uint_action(actor);
    }
};

//////////////////////////////////
const uint_parser_gen   uint_p = uint_parser_gen();

///////////////////////////////////////////////////////////////////////////////
//
//  int_parser class
//
//      Parses signed integers. An int_parser is never directly
//      instantiated. An instance of the int_parser_gen, int_p,
//      generates an int_parser through its [] operator:
//
//          int_p[f]
//
//      where f is a semantic action which may be a function or a
//      functor. A function should be compatible with the interface:
//
//          void f(NumT num);
//
//      A functor should have a member operator() with a compatible
//      signature as above. The matching number is passed into the
//      function/functor (see numeric_action class above).
//
///////////////////////////////////////////////////////////////////////////////
template <typename T>
struct int_parser : public parser<int_parser<T> > {

    int_parser(T n);

    template <typename IteratorT>
    match
    parse(IteratorT& first, IteratorT const& last) const;

    mutable T n;
};


///////////////////////////////////////////////////////////////////////////////
struct int_parser_gen : public int_parser<int> {

    int_parser_gen()
    :   int_parser<int>(0) {}

    template <typename T>
    typename numeric_action_traits<T>::int_action
    operator [] (T const& actor) const
    {
        //  Borland 5.5 reports an internal compiler
        //  error if this is not defined here.
        return numeric_action_traits<T>::int_action(actor);
    }
};

//////////////////////////////////
const int_parser_gen    int_p = int_parser_gen();

///////////////////////////////////////////////////////////////////////////////
//
//  ureal_parser class
//
//      Parses unsigned reals. A ureal_parser is never directly
//      instantiated. An instance of the ureal_parser_gen, ureal_p,
//      generates a ureal_parser through its [] operator:
//
//          ureal_p[f]
//
//      where f is a semantic action which may be a function or a
//      functor. A function should be compatible with the interface:
//
//          void f(NumT num);
//
//      A functor should have a member operator() with a compatible
//      signature as above. The matching number is passed into the
//      function/functor (see numeric_action class above).
//
///////////////////////////////////////////////////////////////////////////////
template <typename T>
struct ureal_parser : public parser<ureal_parser<T> > {

    ureal_parser(T n);

    template <typename IteratorT>
    match
    parse(IteratorT& first, IteratorT const& last) const;

    mutable T n;
};

///////////////////////////////////////////////////////////////////////////////
struct ureal_parser_gen : public ureal_parser<float> {

    ureal_parser_gen()
    :   ureal_parser<float>(0) {}

    template <typename T>
    typename numeric_action_traits<T>::ureal_action
    operator [] (T const& actor) const
    {
        //  Borland 5.5 reports an internal compiler
        //  error if this is not defined here.
        return numeric_action_traits<T>::ureal_action(actor);
    }
};

//////////////////////////////////
const ureal_parser_gen  ureal_p = ureal_parser_gen();

///////////////////////////////////////////////////////////////////////////////
//
//  real_parser class
//
//      Parses signed reals. A real_parser is never directly
//      instantiated. An instance of the real_parser_gen, real_p,
//      generates a real_parser through its [] operator:
//
//          real_p[f]
//
//      where f is a semantic action which may be a function or a
//      functor. A function should be compatible with the interface:
//
//          void f(NumT num);
//
//      A functor should have a member operator() with a compatible
//      signature as above. The matching number is passed into the
//      function/functor (see numeric_action class above).
//
///////////////////////////////////////////////////////////////////////////////
template <typename T>
struct real_parser : public parser<real_parser<T> > {

    real_parser(T n);

    template <typename IteratorT>
    match
    parse(IteratorT& first, IteratorT const& last) const;

    mutable T n;
};

///////////////////////////////////////////////////////////////////////////////
struct real_parser_gen : public real_parser<float> {

    real_parser_gen()
    :   real_parser<float>(0) {}

    template <typename T>
    typename numeric_action_traits<T>::real_action
    operator [] (T const& actor) const
    {
        //  Borland 5.5 reports an internal compiler
        //  error if this is not defined here.
        return numeric_action_traits<T>::real_action(actor);
    }
};

//////////////////////////////////
const real_parser_gen   real_p = real_parser_gen();

///////////////////////////////////////////////////////////////////////////////
}   //  namespace Spirit

#endif
