/*=============================================================================
    Parser operators

    Spirit V1.2
    Copyright (c) 2001, Joel de Guzman

    This software is provided 'as-is', without any express or implied
    warranty. In no event will the copyright holder be held liable for
    any damages arising from the use of this software.

    Permission is granted to anyone to use this software for any purpose,
    including commercial applications, and to alter it and redistribute
    it freely, subject to the following restrictions:

    1.  The origin of this software must not be misrepresented; you must
        not claim that you wrote the original software. If you use this
        software in a product, an acknowledgment in the product documentation
        would be appreciated but is not required.

    2.  Altered source versions must be plainly marked as such, and must
        not be misrepresented as being the original software.

    3.  This notice may not be removed or altered from any source
        distribution.

    Acknowledgements:

        Special thanks to Dan Nuffer, John (EBo) David, Chris Uzdavinis,
        and Doug Gregor. These people are most instrumental in steering
        Spirit in the right direction.

        Special thanks also to people who have contributed to the code base
        and sample code, ported Spirit to various platforms and compilers,
        gave suggestions, reported and provided bug fixes. Alexander
        Hirner, Andy Elvey, Bogdan Kushnir, Brett Calcott, Bruce Florman,
        Changzhe Han, Colin McPhail, Hakki Dogusan, Jan Bares, Joseph
        Smith, Martijn W. van der Lee, Raghavendra Satish, Remi Delcos, Tom
        Spilman, Vladimir Prus, W. Scott Dillman, David A. Greene, Bob
        Bailey, Hartmut Kaiser.

        Finally special thanks also to people who gave feedback and
        valuable comments, particularly members of Spirit's Source Forge
        mailing list and boost.org.

    URL: http://spirit.sourceforge.net/

=============================================================================*/
#ifndef SPIRIT_OPERATORS_HPP
#define SPIRIT_OPERATORS_HPP

///////////////////////////////////////////////////////////////////////////////

#include "boost/spirit/parser.hpp"
#include "boost/spirit/composite.hpp"

///////////////////////////////////////////////////////////////////////////////
namespace spirit {

//  Note: G++2.95 reports an internal compiler error if some
//  parse(...) member functions are defined outside its class
//  declaration body.

///////////////////////////////////////////////////////////////////////////////
//
//  sequence class
//
//      Handles expressions of the form:
//
//          a >> b
//
//      where a and b are parsers. The expression returns a composite
//      parser that matches a and b in sequence. One (not both) of the
//      operands may be a literal char, wchar_t or a primitive string
//      char const*, wchar_t const*.
//
///////////////////////////////////////////////////////////////////////////////
template <typename A, typename B>
struct sequence
:   public binary<A, B>,
    public parser<sequence<A, B> > {

    sequence(A const& a, B const& b)
    :   binary<A, B>(a, b) {}

    template <typename IteratorT>
    match
    parse(IteratorT& first, IteratorT const& last) const
    {
        match       ma, mb;

        if ((ma = this->left().parse(first, last)) &&
            (mb = this->right().parse(first, last)))
        {
            return ma + mb;
        }
        return match();
    }
};

//////////////////////////////////
template <typename A, typename B>
sequence<A, B>
operator>>(parser<A> const& a, parser<B> const& b);

//////////////////////////////////
template <typename A>
sequence<A, chlit<char> >
operator>>(parser<A> const& a, char b);

//////////////////////////////////
template <typename B>
sequence<chlit<char>, B>
operator>>(char a, parser<B> const& b);

//////////////////////////////////
template <typename A>
sequence<A, strlit<cstring<char> > >
operator>>(parser<A> const& a, char const* b);

//////////////////////////////////
template <typename B>
sequence<strlit<cstring<char> >, B>
operator>>(char const* a, parser<B> const& b);

//////////////////////////////////
template <typename A>
sequence<A, chlit<wchar_t> >
operator>>(parser<A> const& a, wchar_t b);

//////////////////////////////////
template <typename B>
sequence<chlit<wchar_t>, B>
operator>>(wchar_t a, parser<B> const& b);

//////////////////////////////////
template <typename A>
sequence<A, strlit<cstring<wchar_t> > >
operator>>(parser<A> const& a, wchar_t const* b);

//////////////////////////////////
template <typename B>
sequence<strlit<cstring<wchar_t> >, B>
operator>>(wchar_t const* a, parser<B> const& b);

///////////////////////////////////////////////////////////////////////////////
//
//  sequential-and operators
//
//      Handles expressions of the form:
//
//          a && b
//
//      Same as a >> b.
//
///////////////////////////////////////////////////////////////////////////////
template <typename A, typename B>
sequence<A, B>
operator&&(parser<A> const& a, parser<B> const& b);

//////////////////////////////////
template <typename A>
sequence<A, chlit<char> >
operator&&(parser<A> const& a, char b);

//////////////////////////////////
template <typename B>
sequence<chlit<char>, B>
operator&&(char a, parser<B> const& b);

//////////////////////////////////
template <typename A>
sequence<A, strlit<cstring<char> > >
operator&&(parser<A> const& a, char const* b);

//////////////////////////////////
template <typename B>
sequence<strlit<cstring<char> >, B>
operator&&(char const* a, parser<B> const& b);

//////////////////////////////////
template <typename A>
sequence<A, chlit<wchar_t> >
operator&&(parser<A> const& a, wchar_t b);

//////////////////////////////////
template <typename B>
sequence<chlit<wchar_t>, B>
operator&&(wchar_t a, parser<B> const& b);

//////////////////////////////////
template <typename A>
sequence<A, strlit<cstring<wchar_t> > >
operator&&(parser<A> const& a, wchar_t const* b);

//////////////////////////////////
template <typename B>
sequence<strlit<cstring<wchar_t> >, B>
operator&&(wchar_t const* a, parser<B> const& b);

///////////////////////////////////////////////////////////////////////////////
//
//  sequential-or class
//
//      Handles expressions of the form:
//
//          a || b
//
//      Equivalent to
//
//          a | b | a >> b;
//
//      where a and b are parsers. The expression returns a composite
//      parser that matches matches a or b in sequence. One (not both) of
//      the operands may be a literal char, wchar_t or a primitive string
//      char const*, wchar_t const*.
//
///////////////////////////////////////////////////////////////////////////////
template <typename A, typename B>
struct sequential_or
:   public binary<A, B>,
    public parser<sequential_or<A, B> > {

    sequential_or(A const& a, B const& b)
    :   binary<A, B>(a, b) {}

    template <typename IteratorT>
    match
    parse(IteratorT& first, IteratorT const& last) const
    {
        { // scope for s
            IteratorT   s = first;
            match       ma = this->left().parse(s, last);
            if (ma)
            {
                IteratorT t = s;
                match mb = this->right().parse(t, last);
                if (mb)
                {
                    // matched a b
                    first = t;
                    return ma + mb;
                }
                else
                {
                    // matched a
                    first = s;
                    return ma;
                }
            }
        }

        return this->right().parse(first, last);
    }
};

//////////////////////////////////
template <typename A, typename B>
sequential_or<A, B>
operator||(parser<A> const& a, parser<B> const& b);

//////////////////////////////////
template <typename A>
sequential_or<A, chlit<char> >
operator||(parser<A> const& a, char b);

//////////////////////////////////
template <typename B>
sequential_or<chlit<char>, B>
operator||(char a, parser<B> const& b);

//////////////////////////////////
template <typename A>
sequential_or<A, strlit<cstring<char> > >
operator||(parser<A> const& a, char const* b);

//////////////////////////////////
template <typename B>
sequential_or<strlit<cstring<char> >, B>
operator||(char const* a, parser<B> const& b);
//////////////////////////////////
template <typename A>
sequential_or<A, chlit<wchar_t> >
operator||(parser<A> const& a, wchar_t b);

//////////////////////////////////
template <typename B>
sequential_or<chlit<wchar_t>, B>
operator||(wchar_t a, parser<B> const& b);

//////////////////////////////////
template <typename A>
sequential_or<A, strlit<cstring<wchar_t> > >
operator||(parser<A> const& a, wchar_t const* b);

//////////////////////////////////
template <typename B>
sequential_or<strlit<cstring<wchar_t> >, B>
operator||(wchar_t const* a, parser<B> const& b);

///////////////////////////////////////////////////////////////////////////////
//
//  alternative class
//
//      Handles expressions of the form:
//
//          a | b
//
//      where a and b are parsers. The expression returns a composite
//      parser that matches a or b. One (not both) of the operands may
//      be a literal char, wchar_t or a primitive string char const*,
//      wchar_t const*.
//
//      The expression is short circuit evaluated. b is never touched
//      when a is returns a successful match.
//
///////////////////////////////////////////////////////////////////////////////
template <typename A, typename B>
struct alternative
:   public binary<A, B>,
    public parser<alternative<A, B> > {

    alternative(A const& a, B const& b)
    :   binary<A, B>(a, b) {}

    template <typename IteratorT>
    match
    parse(IteratorT& first, IteratorT const& last) const
    {
        { // scope for s
            IteratorT s = first;
            if (match hit = this->left().parse(s, last))
            {
                first = s;
                return hit;
            }
        }
        return this->right().parse(first, last);
    }
};

//////////////////////////////////
template <typename A, typename B>
alternative<A, B>
operator|(parser<A> const& a, parser<B> const& b);

//////////////////////////////////
template <typename A>
alternative<A, chlit<char> >
operator|(parser<A> const& a, char b);

//////////////////////////////////
template <typename B>
alternative<chlit<char>, B>
operator|(char a, parser<B> const& b);

//////////////////////////////////
template <typename A>
alternative<A, strlit<cstring<char> > >
operator|(parser<A> const& a, char const* b);

//////////////////////////////////
template <typename B>
alternative<strlit<cstring<char> >, B>
operator|(char const* a, parser<B> const& b);

//////////////////////////////////
template <typename A>
alternative<A, chlit<wchar_t> >
operator|(parser<A> const& a, wchar_t b);

//////////////////////////////////
template <typename B>
alternative<chlit<wchar_t>, B>
operator|(wchar_t a, parser<B> const& b);

//////////////////////////////////
template <typename A>
alternative<A, strlit<cstring<wchar_t> > >
operator|(parser<A> const& a, wchar_t const* b);

//////////////////////////////////
template <typename B>
alternative<strlit<cstring<wchar_t> >, B>
operator|(wchar_t const* a, parser<B> const& b);

///////////////////////////////////////////////////////////////////////////////
//
//  intersection class
//
//      Handles expressions of the form:
//
//          a & b
//
//      where a and b are parsers. The expression returns a composite
//      parser that matches a and b. One (not both) of the operands may
//      be a literal char, wchar_t or a primitive string char const*,
//      wchar_t const*.
//
//      The expression is short circuit evaluated. b is never touched
//      when a is returns a no-match.
//
///////////////////////////////////////////////////////////////////////////////
template <typename A, typename B>
struct intersection
:   public binary<A, B>,
    public parser<intersection<A, B> > {

    intersection(A const& a, B const& b)
    :   binary<A, B>(a, b) {}

    template <typename IteratorT>
    match
    parse(IteratorT& first, IteratorT const& last) const
    {
        IteratorT   ls = first;
        if (match l = this->left().parse(ls, last))
        {
            IteratorT   rs = first;
            if (match r = this->right().parse(rs, last))
            {
                bool    less = l.length() < r.length();
                first = less ? rs : ls;
                return less ? r : l;
            }
        }

        return match();
    }
};

//////////////////////////////////
template <typename A, typename B>
intersection<A, B>
operator&(parser<A> const& a, parser<B> const& b);

//////////////////////////////////
template <typename A>
intersection<A, chlit<char> >
operator&(parser<A> const& a, char b);

//////////////////////////////////
template <typename B>
intersection<chlit<char>, B>
operator&(char a, parser<B> const& b);

//////////////////////////////////
template <typename A>
intersection<A, strlit<cstring<char> > >
operator&(parser<A> const& a, char const* b);

//////////////////////////////////
template <typename B>
intersection<strlit<cstring<char> >, B>
operator&(char const* a, parser<B> const& b);

//////////////////////////////////
template <typename A>
intersection<A, chlit<wchar_t> >
operator&(parser<A> const& a, wchar_t b);

//////////////////////////////////
template <typename B>
intersection<chlit<wchar_t>, B>
operator&(wchar_t a, parser<B> const& b);

//////////////////////////////////
template <typename A>
intersection<A, strlit<cstring<wchar_t> > >
operator&(parser<A> const& a, wchar_t const* b);

//////////////////////////////////
template <typename B>
intersection<strlit<cstring<wchar_t> >, B>
operator&(wchar_t const* a, parser<B> const& b);

///////////////////////////////////////////////////////////////////////////////
//
//  difference: a - b; Matches a but not b
//
//      Handles expressions of the form:
//
//          a - b
//
//      where a and b are parsers. The expression returns a composite
//      parser that matches a but not b. One (not both) of the operands
//      may be a literal char, wchar_t or a primitive string char const*,
//      wchar_t const*.
//
///////////////////////////////////////////////////////////////////////////////
template <typename A, typename B>
struct difference
:   public binary<A, B>,
    public parser<difference<A, B> > {

    difference(A const& a, B const& b)
    :   binary<A, B>(a, b) {}

    template <typename IteratorT>
    match
    parse(IteratorT& first, IteratorT const& last) const
    {
        IteratorT   ls = first;
        IteratorT   rs = first;
        match       hit, hit2;

        if ((hit = this->left().parse(ls, last)) &&
            !(hit2 = this->right().parse(rs, last)))
        {
            first = ls;
            return hit;
        }
        if (hit2.length() < hit.length())
        {
            first = ls;
            return hit;
        }
        return match();
    }
};

//////////////////////////////////
template <typename A, typename B>
difference<A, B>
operator-(parser<A> const& a, parser<B> const& b);

//////////////////////////////////
template <typename A>
difference<A, chlit<char> >
operator-(parser<A> const& a, char b);

//////////////////////////////////
template <typename B>
difference<chlit<char>, B>
operator-(char a, parser<B> const& b);

//////////////////////////////////
template <typename A>
difference<A, strlit<cstring<char> > >
operator-(parser<A> const& a, char const* b);

//////////////////////////////////
template <typename B>
difference<strlit<cstring<char> >, B>
operator-(char const* a, parser<B> const& b);

//////////////////////////////////
template <typename A>
difference<A, chlit<wchar_t> >
operator-(parser<A> const& a, wchar_t b);

//////////////////////////////////
template <typename B>
difference<chlit<wchar_t>, B>
operator-(wchar_t a, parser<B> const& b);

//////////////////////////////////
template <typename A>
difference<A, strlit<cstring<wchar_t> > >
operator-(parser<A> const& a, wchar_t const* b);

//////////////////////////////////
template <typename B>
difference<strlit<cstring<wchar_t> >, B>
operator-(wchar_t const* a, parser<B> const& b);

///////////////////////////////////////////////////////////////////////////////
//
//  exclusive_or class
//
//      Handles expressions of the form:
//
//          a ^ b
//
//      where a and b are parsers. The expression returns a composite
//      parser that matches a or b but not both. One (not both) of the
//      operands may be a literal char, wchar_t or a primitive string
//      char const*, wchar_t const*.
//
///////////////////////////////////////////////////////////////////////////////
template <typename A, typename B>
struct exclusive_or
:   public binary<A, B>,
    public parser<exclusive_or<A, B> > {

    exclusive_or(A const& a, B const& b)
    :   binary<A, B>(a, b) {}

    template <typename IteratorT>
    match
    parse(IteratorT& first, IteratorT const& last) const
    {
        IteratorT   ls = first;
        IteratorT   rs = first;
        match       l = this->left().parse(ls, last);
        match       r = this->right().parse(rs, last);

        if (bool(l) ^ bool(r))
        {
            first = l ? ls : rs;
            return l ? l : r;
        }

        return match();
    }
};

//////////////////////////////////
template <typename A, typename B>
exclusive_or<A, B>
operator^(parser<A> const& a, parser<B> const& b);

//////////////////////////////////
template <typename A>
exclusive_or<A, chlit<char> >
operator^(parser<A> const& a, char b);

//////////////////////////////////
template <typename B>
exclusive_or<chlit<char>, B>
operator^(char a, parser<B> const& b);

//////////////////////////////////
template <typename A>
exclusive_or<A, strlit<cstring<char> > >
operator^(parser<A> const& a, char const* b);

//////////////////////////////////
template <typename B>
exclusive_or<strlit<cstring<char> >, B>
operator^(char const* a, parser<B> const& b);

//////////////////////////////////
template <typename A>
exclusive_or<A, chlit<wchar_t> >
operator^(parser<A> const& a, wchar_t b);

//////////////////////////////////
template <typename B>
exclusive_or<chlit<wchar_t>, B>
operator^(wchar_t a, parser<B> const& b);

//////////////////////////////////
template <typename A>
exclusive_or<A, strlit<cstring<wchar_t> > >
operator^(parser<A> const& a, wchar_t const* b);

//////////////////////////////////
template <typename B>
exclusive_or<strlit<cstring<wchar_t> >, B>
operator^(wchar_t const* a, parser<B> const& b);

///////////////////////////////////////////////////////////////////////////////
//
//  optional class
//
//      Handles expressions of the form:
//
//          !a
//
//      where a is a parser. The expression returns a composite
//      parser that matches its subject zero (0) or one (1) time.
//
///////////////////////////////////////////////////////////////////////////////
template <typename S>
struct optional
:   public unary<S>,
    public parser<optional<S> > {

    optional(S const& a)
    :   unary<S>(a) {}

    template <typename IteratorT>
    match
    parse(IteratorT& first, IteratorT const& last) const
    {
        IteratorT s = first;
        match   r = this->subject().parse(s, last);
        if (r)
        {
            first = s;
            return r;
        }
        else
        {
            return match(0);
        }
    }
};

//////////////////////////////////
template <typename S>
optional<S>
operator!(parser<S> const& a);

///////////////////////////////////////////////////////////////////////////////
//
//  kleene_star class
//
//      Handles expressions of the form:
//
//          *a
//
//      where a is a parser. The expression returns a composite
//      parser that matches its subject zero (0) or more times.
//
///////////////////////////////////////////////////////////////////////////////
template <typename S>
struct kleene_star
:   public unary<S>,
    public parser<kleene_star<S> > {

    kleene_star(S const& a)
    :   unary<S>(a) {}

    template <typename IteratorT>
    match
    parse(IteratorT& first, IteratorT const& last) const
    {
        match hit(0);
        while (true)
        {
            IteratorT s = first;
            match next = this->subject().parse(s, last);
            if (next)
            {
                first = s;
                hit += next;
            }
            else
            {
                return hit;
            }
        }
    }
};

//////////////////////////////////
template <typename S>
kleene_star<S>
operator*(parser<S> const& a);

///////////////////////////////////////////////////////////////////////////////
//
//  positive class
//
//      Handles expressions of the form:
//
//          +a
//
//      where a is a parser. The expression returns a composite
//      parser that matches its subject one (1) or more times.
//
///////////////////////////////////////////////////////////////////////////////
template <typename S>
struct positive
:   public unary<S>,
    public parser<positive<S> > {

    positive(S const& a)
    :   unary<S>(a) {}

    template <typename IteratorT>
    match
    parse(IteratorT& first, IteratorT const& last) const
    {
        match   hit = this->subject().parse(first, last);
        if (hit)
        {
            while (true)
            {
                IteratorT s = first;
                match next = this->subject().parse(s, last);
                if (next)
                {
                    first = s;
                    hit += next;
                }
                else
                {
                    break;
                }
            }
        }
        return hit;
    }
};

//////////////////////////////////
template <typename S>
inline positive<S>
operator + (parser<S> const& a);

///////////////////////////////////////////////////////////////////////////////
}   //  namespace Spirit

#endif
