/*=============================================================================
    Symbol Tables

    Spirit V1.2
    Copyright (c) 2001, Joel de Guzman

    This software is provided 'as-is', without any express or implied
    warranty. In no event will the copyright holder be held liable for
    any damages arising from the use of this software.

    Permission is granted to anyone to use this software for any purpose,
    including commercial applications, and to alter it and redistribute
    it freely, subject to the following restrictions:

    1.  The origin of this software must not be misrepresented; you must
        not claim that you wrote the original software. If you use this
        software in a product, an acknowledgment in the product documentation
        would be appreciated but is not required.

    2.  Altered source versions must be plainly marked as such, and must
        not be misrepresented as being the original software.

    3.  This notice may not be removed or altered from any source
        distribution.

    Acknowledgements:

        Special thanks to Dan Nuffer, John (EBo) David, Chris Uzdavinis,
        and Doug Gregor. These people are most instrumental in steering
        Spirit in the right direction.

        Special thanks also to people who have contributed to the code base
        and sample code, ported Spirit to various platforms and compilers,
        gave suggestions, reported and provided bug fixes. Alexander
        Hirner, Andy Elvey, Bogdan Kushnir, Brett Calcott, Bruce Florman,
        Changzhe Han, Colin McPhail, Hakki Dogusan, Jan Bares, Joseph
        Smith, Martijn W. van der Lee, Raghavendra Satish, Remi Delcos, Tom
        Spilman, Vladimir Prus, W. Scott Dillman, David A. Greene, Bob
        Bailey, Hartmut Kaiser.

        Finally special thanks also to people who gave feedback and
        valuable comments, particularly members of Spirit's Source Forge
        mailing list and boost.org.

    URL: http://spirit.sourceforge.net/

=============================================================================*/
#ifndef SPIRIT_SYMBOLS_HPP
#define SPIRIT_SYMBOLS_HPP

///////////////////////////////////////////////////////////////////////////////

#include "boost/spirit/parser.hpp"

///////////////////////////////////////////////////////////////////////////////
namespace spirit {

///////////////////////////////////////////////////////////////////////////////
//  Forward Declarations

template <typename CharT, typename T> class tst;
template <typename T> class symbol_match;
template <typename T, typename SetT> class symbol_inserter;
template <typename ParserT, typename ActionT> class symbol_action;

///////////////////////////////////////////////////////////////////////////////
//
//  symbols class
//
//      This class implements a symbol table. The symbol table holds a
//      dictionary of symbols where each symbol is a sequence of CharTs.
//      The template class can work efficiently with 8, 16 and 32 bit
//      characters. Mutable data of type T is associated with each
//      symbol.
//
//      The class is a parser. The parse member function returns
//      additional information in the symbol_match class (see below).
//      The additional data is a pointer to some data associated with
//      the matching symbol.
//
//      The actual set implementation is supplied by the SetT template
//      parameter. By default, this uses the tst class (see set.ipp).
//
//      Symbols are added into the symbol table statically using the
//      construct:
//
//          sym = a, b, c, d ...;
//
//      where sym is a symbol table and a..d are strings. Example:
//
//          sym = "pineapple", "orange", "banana", "apple";
//
//      Alternatively, symbols may be added dynamically through the
//      member functor 'add' (see symbol_inserter below). The member
//      functor 'add' may be attached to a parser as a semantic action:
//
//          p[sym.add]
//
//      where p is a parser (and sym is a symbol table). On success,
//      the matching portion of the input is added to the symbol table.
//
///////////////////////////////////////////////////////////////////////////////
template <
    typename T = int,
    typename CharT = char,
    typename SetT = tst<T, CharT> >
class symbols
:   private SetT,
    public parser<symbols<T, CharT, SetT> > {

public:

    typedef symbol_match<T> symbol_match;

    symbols();
    symbols(symbols const& other);
    ~symbols();

    symbols&
    operator=(symbols const& other);

    symbol_inserter<T, SetT> const&
    operator=(CharT const* str);

    template <typename IteratorT>
    symbol_match
    parse(IteratorT& first, IteratorT const& last) const;

    template <typename ActionT>
    symbol_action<symbols<T, CharT, SetT>, ActionT>
    operator[](ActionT const& actor) const
    {
        //  Borland 5.5 complains if this is not defined here
        return symbol_action<symbols<T, CharT, SetT>, ActionT>(*this, actor);
    }
    symbol_inserter<T, SetT> const add;
};

///////////////////////////////////////////////////////////////////////////////
//
//  Symbol table utilities
//
//  add
//
//      adds a symbol 'sym' (string) to a symbol table 'table' plus an
//      optional data 'data' associated with the symbol. Returns a pointer to
//      the data associated with the symbol or NULL if add failed (e.g. when
//      the symbol is already added before).
//
//  find
//
//      finds a symbol 'sym' (string) from a symbol table 'table'. Returns a
//      pointer to the data associated with the symbol or NULL if not found
//
///////////////////////////////////////////////////////////////////////////////
template <typename T, typename CharT, typename SetT>
T*  add(symbols<T, CharT, SetT>& table, CharT const* sym, T const& data = T());

template <typename T, typename CharT, typename SetT>
T*  find(symbols<T, CharT, SetT>& table, CharT const* sym);

///////////////////////////////////////////////////////////////////////////////
//
//  symbol_action class
//
//      Links a symbol table with a user defined semantic action.
//      The semantic action may be a function or a functor. A function
//      should be compatible with the interface:
//
//          void f(IteratorT first, IteratorT last, T& data);
//
//      Where first points to the current input, last points to one
//      after the end of the input (same as STL algorithms) and data
//      is any information associated with the matching symbol in the
//      symbol table (see symbols class above).
//
//      A functor should have a member operator() with a compatible
//      signature as above. The matching portion is passed to the
//      function/functor. This is the default class that symbol
//      tables use when dealing with the construct:
//
//          p[f]
//
//      where p is a parser and f is a function or functor.
//
///////////////////////////////////////////////////////////////////////////////
template <typename ParserT, typename ActionT>
class symbol_action
:   public unary<ParserT>,
    public parser<symbol_action<ParserT, ActionT> > {

public:

    symbol_action(ParserT const& parser, ActionT const& actor);

    template <typename IteratorT>
    match
    parse(IteratorT& first, IteratorT const& last) const;

private:

    ActionT   actor;
};

///////////////////////////////////////////////////////////////////////////////
//
//  symbol_match class
//
//      An instance of this class is returned by the symbols::parse
//      member function. symbol_match is a subclass of match with the
//      added T* data() member function. This is the data associated
//      with each entry in the symbol table (see symbols above). The
//      data is null on unsuccessful matches.
//
///////////////////////////////////////////////////////////////////////////////
template <typename T = int>
class symbol_match : public match {

public:
                symbol_match();
                symbol_match(unsigned length, T* info);
    T*          data() const;

private:

    T* info;
};

///////////////////////////////////////////////////////////////////////////////
//
//  symbol_inserter class
//
//      The symbols class holds an instance of this class named 'add'.
//      This can be called directly just like a member function:
//
//          sym.add(first, last, data);
//
//      where sym is a symbol table. The third argument is optional.
//      This may also be used as a semantic action since it conforms
//      to the action interface (see action.hpp):
//
//          p[sym.add]
//
///////////////////////////////////////////////////////////////////////////////
template <typename T, typename SetT>
class symbol_inserter {

public:

    symbol_inserter(SetT& set);

    template <typename IteratorT>
    T*
    operator()(
        IteratorT           first,
        IteratorT const&    last,
        T const&            data = T()) const;

    template <typename CharT>
    symbol_inserter const&
    operator,(CharT const* str) const;

private:

    SetT& set;
};

///////////////////////////////////////////////////////////////////////////////
}   //  namespace Spirit

#endif
